/*
    Plee The Bear - Level editor

    Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/item_class.hpp
 * \brief A class that stores the type of the fields of an item class.
 * \author Julien Jorge
 */
#ifndef __BF_ITEM_CLASS_HPP__
#define __BF_ITEM_CLASS_HPP__

#include <string>
#include <map>
#include <list>
#include <claw/functional.hpp>
#include <claw/iterator.hpp>

#include "bf/type_field.hpp"

namespace bf
{
  /**
   * \brief A class that stores the type of the fields of an item class.
   * \author Julien Jorge
   */
  class item_class
  {
  public:
    typedef std::map<std::string, const type_field*> field_map_type;

    typedef claw::wrapped_iterator
    < const type_field,
      field_map_type::const_iterator,
      claw::unary_compose
      < claw::const_dereference<type_field>,
        claw::const_pair_second<field_map_type::value_type> > >
    ::iterator_type field_iterator;

  public:
    item_class();
    item_class( const item_class& that );
    ~item_class();

    item_class& operator=(const item_class& that);

    field_iterator field_begin() const { return m_field.begin(); }
    field_iterator field_end() const { return m_field.end(); }

    void add_field( const std::string& name, const type_field& field );

    void set_class_name( const std::string& class_name );
    void set_category( const std::string& category );
    void set_color( const std::string& color );
    void set_fixable( bool fixable );
    void set_super_classes( const std::list<std::string>& super_classes );
    void add_super_class( const std::string& super_class );

    const type_field& get_field( const std::string& field_name ) const;
    const std::string& get_class_name() const;
    const std::string& get_category() const;
    const std::string& get_color() const;
    bool get_fixable() const;
    bool get_need_a_name() const;
    const std::list<std::string>& get_super_classes() const;
    void find_hierarchy( std::list<std::string>& hierarchy ) const;

    bool has_field( const std::string& name ) const;
    bool has_field( const std::string& name, type_field::field_type t ) const;

    bool field_unicity_test(std::string& error_msg) const;

  private:
    void clear();
    void copy( const item_class& that );

    const type_field* search_field( const std::string& field_name ) const;

  private:
    /** \brief The name of the class. */
    std::string m_class_name;

    /** \brief The category in which this class fits. */
    std::string m_category;

    /** \brief The color used for displaying the item. */
    std::string m_color;

    /** \brief Tell if the item can be fixed. */
    bool m_fixable;

    /** \brief The classes from which this one inherit. */
    std::list<std::string> m_super_classes;

    /** \brief The fields of this class. */
    field_map_type m_field;

  }; // class item_class
} // namespace bf

#endif // __BF_ITEM_CLASS_HPP__
