/*
    Plee The Bear - Level compiler

    Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/code/sprite.cpp
 * \brief Implementation of the bf::sprite class.
 * \author Julien Jorge
 */
#include "bf/sprite.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Default constructor.
 */
bf::sprite::sprite()
  : m_top(0), m_left(0), m_clip_width(0), m_clip_height(0), m_width(0),
    m_height(0), m_flip_x(false), m_flip_y(false), m_alpha(1)
{

} // sprite::sprite()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the name of the image resource to use.
 * \param name The new value.
 */
void bf::sprite::set_image_name( const std::string& name )
{
  m_image_name = name;
} // sprite::set_image_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the Y-coordinate in the image resource.
 * \param top The new value.
 */
void bf::sprite::set_top( const unsigned int top )
{
  m_top = top;
} // sprite::set_top()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the X-coordinate in the image resource.
 * \param left The new value.
 */
void bf::sprite::set_left( const unsigned int left )
{
  m_left = left;
} // sprite::set_left()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the width in the image resource.
 * \param width The new value.
 */
void bf::sprite::set_clip_width( const unsigned int width )
{
  m_clip_width = width;
} // sprite::set_clip_width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the height in the image resource.
 * \param height The new value.
 */
void bf::sprite::set_clip_height( const unsigned int height )
{
  m_clip_height = height;
} // sprite::set_clip_height()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the width on the screen.
 * \param width The new value.
 */
void bf::sprite::set_width( const unsigned int width )
{
  m_width = width;
} // sprite::set_width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the height on the screen.
 * \param height The new value.
 */
void bf::sprite::set_height( const unsigned int height )
{
  m_height = height;
} // sprite::set_height()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the horizontal flip attribute of the image.
 * \param b The new value.
 */
void bf::sprite::set_flip_x( bool b )
{
  m_flip_x = b;
} // sprite::set_flip_x()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the vertical flip attribute of the image.
 * \param b The new value.
 */
void bf::sprite::set_flip_y( bool b )
{
  m_flip_y = b;
} // sprite::set_flip_y()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the alpha transparency.
 * \param alpha The new value.
 */
void bf::sprite::set_alpha( double alpha )
{
  m_alpha = alpha;
} // sprite::set_alpha()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the left x-ccordinate of the sprite.
 */
unsigned int bf::sprite::get_left() const
{
  return m_left;
} // sprite::get_left()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the top y-coordinate of the sprite.
 */
unsigned int bf::sprite::get_top() const
{
  return m_top;
} // sprite::get_top()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the width of the sprite in the image source.
 */
unsigned int bf::sprite::get_clip_width() const
{
  return m_clip_width;
} // sprite::get_clip_width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the height of the sprite in the image source.
 */
unsigned int bf::sprite::get_clip_height() const
{
  return m_clip_height;
} // sprite::get_clip_height()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the width of the sprite on the screen.
 */
unsigned int bf::sprite::get_width() const
{
  return m_width;
} // sprite::get_width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the height of the sprite on the screen.
 */
unsigned int bf::sprite::get_height() const
{
  return m_height;
} // sprite::get_height()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the flip_x option of the sprite.
 */
bool bf::sprite::get_flip_x() const
{
  return m_flip_x;
} // sprite::get_flip_x()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the flip_y option of the sprite.
 */
bool  bf::sprite::get_flip_y() const
{
  return m_flip_y;
} // sprite::get_flip_y()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the alpha transparence of the sprite.
 */
double  bf::sprite::get_alpha() const
{
  return m_alpha;
} // sprite::get_alpha()


/*----------------------------------------------------------------------------*/
/**
 * \brief Get the image name of the sprite.
 */
const std::string& bf::sprite::get_image_name() const
{
  return m_image_name;
} // sprite::get_image_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile the sprite.
 * \param f The stream in which we write the compiled sprite.
 */
void bf::sprite::compile( compiled_file& f ) const
{
  f << m_image_name << m_left << m_top << m_clip_width << m_clip_height
    << m_width << m_height << m_flip_x << m_flip_y << m_alpha;
} // sprite::compile()
