/*
  Plee The Bear - Level editor

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/code/properties_frame.cpp
 * \brief Implementation of the bf::properties_frame class.
 * \author Julien Jorge
 */
#include "bf/properties_frame.hpp"

#include "bf/bool_edit.hpp"
#include "bf/custom_type.hpp"
#include "bf/free_edit.hpp"
#include "bf/item_class_pool.hpp"
#include "bf/set_edit.hpp"
#include "bf/wx_facilities.hpp"
#include "bf/animation_edit.hpp"
#include "bf/sprite_edit.hpp"

#include <list>
#include <claw/assert.hpp>

/*----------------------------------------------------------------------------*/
wxString bf::properties_frame::s_default_title( _("Item properties") );

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param parent Pointer to the owner.
 */
bf::properties_frame::properties_frame( wxFrame* parent )
  : wxFrame( parent, wxID_ANY, s_default_title ), m_item(NULL),
    m_windows_layout(NULL)
{
  create_controls();
  
  m_prop->InsertColumn(0, _("Property"));
  m_prop->InsertColumn(1, _("Value"));
  
  Fit();
} // properties_frame::properties_frame()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the windows layout of the program.
 * \param layout The layout.
 */
void bf::properties_frame::set_window_layout( windows_layout& layout )
{
  m_windows_layout = &layout;
} // properties_frame::set_window_layout()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the item for which we want the properties.
 * \param item The item instance concerned by this window.
 */
void bf::properties_frame::set_item( item_instance* item )
{
  if ( m_item != item )
    {
      m_item = item;
      m_prop->DeleteAllItems();

      if ( m_item == NULL )
        {
          SetTitle( s_default_title );
          m_fixed_box->Disable();
          m_id_text->Disable();
        }
      else
        {
          enumerate_properties();
          SetTitle( std_to_wx_string( m_item->get_class_name() ) );
          m_fixed_box->Enable();
          m_id_text->Enable();
	}
    }

  if ( m_item != NULL )
    {
      update_values();
      update_controls();
    }
} // properties_frame::set_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Enumerate all the properties of the current item.
 */
void bf::properties_frame::enumerate_properties()
{
  std::list<std::string> hierarchy;
  std::list<std::string>::const_iterator it;
  
  const item_class& item =
    item_class_pool::get_instance().get_item_class(m_item->get_class_name());
  item.find_hierarchy( hierarchy );

  std::vector<std::string> fields;

  for ( it=hierarchy.begin(); it!=hierarchy.end(); ++it )
    get_fields_of(fields, *it);

  // wxListCtrl::InsertItem add the item at the beging of the list, so we sort
  // the fields in decreasing order to have a display in increasing order.
  std::sort( fields.begin(), fields.end(), std::greater<std::string>() );

  show_fields(fields);
} // properties_frame::enumerate_properties()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the fields of a given class.
 * \param fields (out) The fields of \a classname are pushed back to this table.
 * \param class_name The name of the class for which we want the fields.
 */
void bf::properties_frame::get_fields_of
( std::vector<std::string>& fields, const std::string& class_name ) const
{
  const item_class& item =
    item_class_pool::get_instance().get_item_class(class_name);
  item_class::field_iterator it;

  for ( it=item.field_begin(); it!=item.field_end(); ++it )
    fields.push_back(it->get_name());
} // properties_frame::get_fields_of()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add the fields in the view of the properties.
 * \param fields The names of the fields to add.
 */
void bf::properties_frame::show_fields( const std::vector<std::string>& fields )
{
  for ( std::size_t i=0; i!=fields.size(); ++i )
    {
      wxListItem prop;
      prop.SetText( std_to_wx_string(fields[i]) );
      m_prop->InsertItem(prop);
    }
} // properties_frame::show_fields()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the value of all fields of the current item.
 */
void bf::properties_frame::update_values()
{
  const item_class& item =
    item_class_pool::get_instance().get_item_class(m_item->get_class_name());

  for ( long i=0; i!=m_prop->GetItemCount(); ++i )
    {
      wxListItem prop;
      prop.SetId(i);
      m_prop->GetItem(prop);

      const type_field& f = item.get_field( wx_to_std_string(prop.GetText()) );

      if ( m_item->has_value(f) )
        prop.SetText( convert_value_to_text(f) );
      else
        prop.SetText( wxT("") );

      prop.SetColumn(1);
      m_prop->SetItem(prop);

      if ( f.get_required() )
        set_required_color( i, m_item->has_value(f) );
    }
} // properties_frame::update_values()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the correct font for a required field.
 * \param i The line to change.
 * \param b Tell if the field has a value or not.
 */
void bf::properties_frame::set_required_color( unsigned int i, bool b )
{
  wxListItem prop;
  prop.SetId(i);
  m_prop->GetItem(prop);

  wxFont font( m_prop->GetFont() );
  font.SetWeight( wxFONTWEIGHT_BOLD );
  prop.SetFont( font );

  if (b)
    prop.SetTextColour( *wxBLACK );
  else
    prop.SetTextColour( *wxRED );

  m_prop->SetItem(prop);
} // properties_frame::set_required_color()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert the value of a field in a string.
 * \param f The field for which we want the value.
 */
wxString
bf::properties_frame::convert_value_to_text( const type_field& f ) const
{
  wxString result;

  if ( f.is_list() )
    switch ( f.get_field_type() )
      {
      case type_field::integer_field_type:
        result = convert_value_to_text< std::list<integer_type> >(f.get_name());
        break;
      case type_field::u_integer_field_type:
        result =
          convert_value_to_text< std::list<u_integer_type> >(f.get_name());
        break;
      case type_field::real_field_type:
        result = convert_value_to_text< std::list<real_type> >(f.get_name());
        break;
      case type_field::boolean_field_type:
        result = convert_value_to_text< std::list<bool_type> >(f.get_name());
        break;
      case type_field::string_field_type:
        result = convert_value_to_text< std::list<string_type> >(f.get_name());
        break;
      case type_field::sprite_field_type:
	result = convert_value_to_text< std::list<sprite> >(f.get_name());
	break;
      case type_field::animation_field_type:
	result = convert_value_to_text< std::list<animation> >(f.get_name());
	break;
      case type_field::item_reference_field_type:
	result =
          convert_value_to_text< std::list<item_reference_type> >(f.get_name());
	break;
      }
  else
    switch ( f.get_field_type() )
      {
      case type_field::integer_field_type:
        result = convert_value_to_text<integer_type>(f.get_name());
        break;
      case type_field::u_integer_field_type:
        result = convert_value_to_text<u_integer_type>(f.get_name());
        break;
      case type_field::real_field_type:
        result = convert_value_to_text<real_type>(f.get_name());
        break;
      case type_field::boolean_field_type:
        result = convert_value_to_text<bool_type>(f.get_name());
        break;
      case type_field::string_field_type:
        result = convert_value_to_text<string_type>(f.get_name());
        break;
      case type_field::sprite_field_type:
	result = convert_value_to_text<sprite>(f.get_name());
	break;
      case type_field::animation_field_type:
	result = convert_value_to_text<animation>(f.get_name());
	break;
      case type_field::item_reference_field_type:
	result = convert_value_to_text<item_reference_type>(f.get_name());
	break;
      }

  return result;
} // properties_frame::convert_value_to_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Ajust the size of the last column so there is no empty space on the
 *        right.
 */
void bf::properties_frame::adjust_last_column_size()
{
  m_prop->SetColumnWidth( 1, m_prop->GetSize().x - m_prop->GetColumnWidth(0) );
} // properties_frame::adjust_last_column_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove the value in the selected field.
 */
void bf::properties_frame::delete_selected_field()
{
  long index = m_prop->GetFocusedItem();

  if ( index != wxNOT_FOUND )
    {
      const std::string class_name = m_item->get_class_name();

      const item_class& item =
        item_class_pool::get_instance().get_item_class(class_name);

      const type_field& f =
        item.get_field( wx_to_std_string(m_prop->GetItemText(index)) );

      if ( m_item->has_value(f) )
        {
          ingame_view* view =
            m_windows_layout->get_current_level_view()->get_ingame_view();

          view->start_change();
          m_item->delete_value(f);
          view->render();
          update_values();
        }
    }
} // properties_frame::delete_selected_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Event sent when the user select an item.
 * \param event The event.
 */
void bf::properties_frame::create_field_editor( size_t index )
{
  const item_class& item =
    item_class_pool::get_instance().get_item_class(m_item->get_class_name());

  const type_field& f =
    item.get_field( wx_to_std_string(m_prop->GetItemText(index)) );
  
  switch ( f.get_field_type() )
    {
    case type_field::integer_field_type:
      show_simple_property_dialog<integer_type>(f, _("integer"));
      break;
    case type_field::u_integer_field_type:
      show_simple_property_dialog<u_integer_type>(f, _("unsigned integer"));
      break;
    case type_field::real_field_type:
      show_simple_property_dialog<real_type>(f, _("real"));
      break;
    case type_field::boolean_field_type:
      show_property_dialog<bool_edit>(f, _("boolean"));
      break;
    case type_field::string_field_type:
      show_string_property_dialog(f);
      break;
    case type_field::sprite_field_type:
      show_property_dialog<sprite_edit>(f, _("sprite"));
      break;
    case type_field::animation_field_type:
      show_property_dialog<animation_edit>(f, _("animation"));
      break;
    case type_field::item_reference_field_type:
      show_item_reference_property_dialog(f);
      break;
    }
} // properties_frame::create_field_editor()

/*----------------------------------------------------------------------------*/
/**
 * \brief Show the adequate dialog for editing a given string field.
 * \param f The type of the field we are editing.
 */
void bf::properties_frame::show_string_property_dialog( const type_field& f )
{
  switch ( f.get_range_type() )
    {
    case type_field::field_range_free:
      show_property_dialog< free_edit<string_type> >(f, _("string"));
      break;
    case type_field::field_range_set:
      show_property_dialog< set_edit<string_type> >(f, _("string"));
      break;
    default:
      {
        CLAW_ASSERT(false, "range type is not valid.");
      }
    }
} // properties_frame::show_string_property_dialog()

/*----------------------------------------------------------------------------*/
/**
 * \brief Show the adequate dialog for editing a given item reference field.
 * \param f The type of the field we are editing.
 */
void
bf::properties_frame::show_item_reference_property_dialog( const type_field& f )
{
  wxArrayString values;
  
  const ingame_view* view =
    m_windows_layout->get_current_level_view()->get_ingame_view();
  layer::const_item_iterator it;
  const layer::const_item_iterator eit(view->get_active_layer().item_end());

  for (it=view->get_active_layer().item_begin(); it!=eit; ++it)
    if ( !it->get_id().empty() && (it->get_id() != m_item->get_id()) )
      values.Add( std_to_wx_string(it->get_id()) );

  if ( f.is_list() )
    edit_item_reference_field< std::list<item_reference_type> >(f, values);
  else
    edit_item_reference_field<item_reference_type>(f, values);
} // properties_frame::show_item_reference_property_dialog()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the controls.
 */
void bf::properties_frame::update_controls()
{
  const item_class& current_class
    ( item_class_pool::get_instance().get_item_class
      ( m_item->get_class_name() ) );
      
  if ( current_class.get_fixable() )
    {
      m_fixed_box->Enable();
      m_fixed_box->SetValue(m_item->get_fixed());
    }
  else
    m_fixed_box->Disable();

  m_id_text->SetValue( std_to_wx_string(m_item->get_id()) );
} // properties_frame::update_controls()
  
/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls of the window.
 */
void bf::properties_frame::create_controls()
{
  m_prop = new wxListView( this, IDC_ITEM_PROPERTIES, wxDefaultPosition,
			   wxDefaultSize,
			   wxLC_REPORT | wxLC_VRULES | wxLC_SINGLE_SEL );
  m_fixed_box = new wxCheckBox( this, IDC_FIXED_STATE, _("Fixed") );
  m_id_text =
    new wxTextCtrl( this, IDC_TEXT_IDENTIFIER, wxEmptyString, wxDefaultPosition,
                    wxDefaultSize, wxTE_PROCESS_ENTER );

  wxBoxSizer* sizer = new wxBoxSizer( wxVERTICAL );

  sizer->Add( m_prop, 1, wxEXPAND );
  sizer->AddSpacer(5);

  sizer->Add( m_fixed_box, 0, wxEXPAND );
  sizer->AddSpacer(5);

  wxBoxSizer* s_sizer = new wxBoxSizer( wxHORIZONTAL );
  s_sizer->Add( new wxStaticText(this, wxID_ANY, _("Id: ")), 0,
		wxALIGN_CENTRE_VERTICAL | wxALL );
  s_sizer->Add( m_id_text, 1, wxEXPAND );
  sizer->Add( s_sizer, 0, wxEXPAND );

  m_id_text->Disable();
  m_fixed_box->Disable();

  SetSizer(sizer);
} // properties_frame::create_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Procedure called when closing the window.
 * \param event This event occured.
 */
void bf::properties_frame::on_close(wxCloseEvent& event)
{
  if ( event.CanVeto() )
    {
      Hide();
      event.Veto();
    }
} // properties_frame::on_close()

/*----------------------------------------------------------------------------*/
/**
 * \brief Event sent to a resized window.
 * \param event The event.
 */
void bf::properties_frame::on_size(wxSizeEvent& event)
{
  adjust_last_column_size();
  event.Skip();
} // properties_frame::on_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Event sent when the user starts to resize a column.
 * \param event The event.
 */
void bf::properties_frame::on_column_begin_drag(wxListEvent& event)
{
  if ( event.GetColumn() + 1 == m_prop->GetColumnCount() )
    event.Veto();
  else
    event.Skip();
} // properties_frame::on_column_begin_drag()

/*----------------------------------------------------------------------------*/
/**
 * \brief Event sent when the user ends resizing a column.
 * \param event The event.
 */
void bf::properties_frame::on_column_end_drag(wxListEvent& event)
{
  adjust_last_column_size();
} // properties_frame::on_column_begin_drag()

/*----------------------------------------------------------------------------*/
/**
 * \brief Event sent when the user activates an item.
 * \param event The event.
 */
void bf::properties_frame::on_item_activated(wxListEvent& event)
{
  create_field_editor( event.GetIndex() );
} // properties_frame::on_item_select()

/*----------------------------------------------------------------------------*/
/**
 * \brief Event sent when the user chek or unchek the fixed box.
 * \param event The event.
 */
void bf::properties_frame::on_change_fixed(wxCommandEvent& event)
{
  ingame_view* view =
    m_windows_layout->get_current_level_view()->get_ingame_view();
  
  view->start_change();
  m_item->set_fixed(m_fixed_box->IsChecked());
  view->render();
} // properties_frame::on_change_fixed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Event sent when the user presses a key.
 * \param event The keyboard event that occured.
 */
void bf::properties_frame::on_key_up(wxKeyEvent& event)
{
  switch( event.GetKeyCode() )
    {
    case WXK_DELETE:
      delete_selected_field();
      break;
    default:
      event.Skip();
    }
} // properties_frame::on_key_up()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user pressed the "Enter" key in the identifier text field.
 * \param event The event.
 */
void bf::properties_frame::on_validate_id(wxCommandEvent& event)
{
  const std::string id = wx_to_std_string( m_id_text->GetValue() );

  if ( id.empty() )
    m_item->set_id(id);
  else if ( id != m_item->get_id() )
    {
      ingame_view* view =
        m_windows_layout->get_current_level_view()->get_ingame_view();

      std::pair<bool, layer::const_item_iterator> it =
        view->get_level().find_item_by_id(id);

      if (it.first)
        {
          wxMessageDialog dlg
            ( this, _("This identifier is already in use."),
              _("Bad identifier"), wxID_OK );

          dlg.ShowModal();
        }
      else
        {
          view->start_change();
          m_item->set_id(id);
        }
    }
} // properties_frame::on_validate_id()

/*----------------------------------------------------------------------------*/
BEGIN_EVENT_TABLE(bf::properties_frame, wxFrame)
  EVT_CLOSE( bf::properties_frame::on_close )
  EVT_SIZE( bf::properties_frame::on_size )
  EVT_LIST_COL_BEGIN_DRAG
    ( bf::properties_frame::IDC_ITEM_PROPERTIES,
      bf::properties_frame::on_column_begin_drag )
  EVT_LIST_COL_END_DRAG
    ( bf::properties_frame::IDC_ITEM_PROPERTIES,
      bf::properties_frame::on_column_end_drag )
  EVT_LIST_ITEM_ACTIVATED
    ( bf::properties_frame::IDC_ITEM_PROPERTIES,
      bf::properties_frame::on_item_activated )
  EVT_CHECKBOX
    ( bf::properties_frame::IDC_FIXED_STATE,
      bf::properties_frame::on_change_fixed )
  EVT_KEY_UP( bf::properties_frame::on_key_up )
  EVT_TEXT_ENTER( bf::properties_frame::IDC_TEXT_IDENTIFIER,
                  bf::properties_frame::on_validate_id )
END_EVENT_TABLE()
