/*
    Plee The Bear - Level editor

    Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/code/main_frame.cpp
 * \brief Implementation of the bf::main_frame class.
 * \author Julien Jorge
 */
#include "bf/main_frame.hpp"

#include "bf/image_pool.hpp"
#include "bf/ingame_view_frame.hpp"
#include "bf/item_class_pool.hpp"
#include "bf/item_class_pool_frame.hpp"
#include "bf/item_instance.hpp"
#include "bf/layer_list_frame.hpp"
#include "bf/layer_properties_frame.hpp"
#include "bf/level_file_xml_reader.hpp"
#include "bf/level_properties_frame.hpp"
#include "bf/properties_frame.hpp"
#include "bf/wx_facilities.hpp"

#include <claw/string_algorithm.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Default constructor.
 */
bf::main_frame::main_frame()
  : wxFrame(NULL, wxID_ANY, wxT("Bear Factory")), m_windows_layout(NULL)
{
  m_config.load();
  m_config_frame = new config_frame( this, m_config );

  item_class_pool::get_instance().scan_directory( m_config.item_class_path );
  image_pool::get_instance().scan_directory( m_config.data_path );

  SetSize( m_config.main_rect );
  Show();

  properties_frame* prop = new properties_frame( this );
  prop->SetSize( m_config.properties_rect );
  prop->Show( m_config.properties_visible );

  layer_list_frame* lay = new layer_list_frame( this );
  lay->SetSize( m_config.layer_list_rect );
  lay->Show( m_config.layer_list_visible );

  item_class_pool_frame* items = new item_class_pool_frame( this );
  items->SetSize( m_config.item_class_pool_rect );
  items->Show( m_config.item_class_pool_visible );

  m_windows_layout = new windows_layout( *this, *prop, *lay, *items );

  create_menu();
  create_controls();

  lay->set_window_layout( *m_windows_layout );
  items->set_window_layout( *m_windows_layout );
  prop->set_window_layout( *m_windows_layout );

  turn_level_entries( false );
} // main_frame::main_frame()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bf::main_frame::~main_frame()
{
  if (m_windows_layout == NULL)
    delete m_windows_layout;
} // main_frame::~main_frame()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load a level.
 * \param path The path to the level to load.
 */
void bf::main_frame::load_level( const wxString& path )
{
  level_file_xml_reader reader;
      
  try
    {
      level* lvl = reader.load(path);

      add_level_view
        ( new ingame_view_frame(*m_windows_layout, lvl, path) );
    }
  catch( std::exception& e )
    {
      claw::logger << claw::log_error << e.what() << claw::lendl;
    }
} // main_frame::load_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the active level window.
 * \param view The active level window.
 */
void bf::main_frame::set_active_level( ingame_view_frame* view )
{
  turn_level_entries(view!=NULL);
} // main_frame::set_active_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Enable/disable the controls relative to the levels.
 * \param b Tell if the controls are enabled.
 */
void bf::main_frame::turn_level_entries( bool b )
{
  GetMenuBar()->Enable( wxID_SAVE, b );
  GetMenuBar()->Enable( wxID_SAVEAS, b );
  GetMenuBar()->Enable( IDM_SAVE_ALL_LEVELS, b );

  FindWindowById( wxID_SAVE, this )->Enable( b );
} // main_frame::turn_level_entries()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the menu bar.
 */
void bf::main_frame::create_menu()
{
  wxMenuBar* menu_bar = new wxMenuBar();

  menu_bar->Append(create_level_menu(), _("&Level"));
  menu_bar->Append(create_window_menu(), _("&Window"));

  SetMenuBar(menu_bar);
} // main_frame::create_menu()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls.
 */
void bf::main_frame::create_controls()
{
  wxBoxSizer* sizer = new wxBoxSizer(wxHORIZONTAL);
  const wxSize btn_size(25, 25);

  sizer->Add
    ( new wxButton(this, wxID_NEW, wxT("N"), wxDefaultPosition, btn_size), 0,
      wxALL, 3 );
  sizer->Add
    ( new wxButton(this, wxID_OPEN, wxT("O"), wxDefaultPosition, btn_size), 0,
      wxALL, 3 );
  sizer->Add
    ( new wxButton(this, wxID_SAVE, wxT("S"), wxDefaultPosition, btn_size), 0,
      wxALL, 3 );

  SetSizer(sizer);
} // main_frame::create_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the "Level" menu of the menu bar.
 */
wxMenu* bf::main_frame::create_level_menu() const
{
  wxMenu* menu = new wxMenu();

  menu->Append( wxID_NEW, _("&New..."), _("Create a new level.") );
  menu->Append( wxID_OPEN, _("&Open..."), _("Open an existing level.") );
  menu->Append( wxID_SAVE, _("&Save"), _("Save the current level.") );
  menu->Append( wxID_SAVEAS, _("Save &as..."),
                _("Save the current level under a new name.") );
  menu->Append( IDM_SAVE_ALL_LEVELS, _("Save a&ll"), _("Save all levels.") );
  menu->AppendSeparator();
  menu->Append( wxID_PREFERENCES, _("&Configuration..."),
                _("Configure the paths and programs.") );
  menu->AppendSeparator();
  menu->Append( wxID_EXIT, _("&Exit"), _("Close everything and exit.") );

  return menu;
} // main_frame::create_level_menu()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the "Window" menu of the menu bar.
 */
wxMenu* bf::main_frame::create_window_menu() const
{
  wxMenu* menu = new wxMenu();

  menu->AppendCheckItem( IDM_ITEM_PROPERTIES, _("Item &properties"),
                         _("Display the properties of an item.") );
  menu->AppendCheckItem
    ( IDM_LAYER_LIST, _("&Layer list"), _("Display the list of layers.") );

  menu->AppendCheckItem
    ( IDM_ITEM_CLASS_POOL, _("&Item classes"),
      _("Display the of items for the current layers.") );

  return menu;
} // main_frame::create_window_menu()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the configuration of the program.
 */
void bf::main_frame::save_config()
{
  m_config.main_rect = GetScreenRect();
  m_config.properties_rect =
    m_windows_layout->get_properties_frame().GetScreenRect();
  m_config.layer_list_rect =
    m_windows_layout->get_layer_list_frame().GetScreenRect();
  m_config.item_class_pool_rect =
    m_windows_layout->get_item_class_pool_frame().GetScreenRect();

  m_config.properties_visible =
    m_windows_layout->get_properties_frame().IsShown();
  m_config.layer_list_visible =
    m_windows_layout->get_layer_list_frame().IsShown();
  m_config.item_class_pool_visible =
    m_windows_layout->get_item_class_pool_frame().IsShown();
  
  if ( m_windows_layout->get_current_level_view() != NULL )
    m_config.default_level_window_size =
      m_windows_layout->get_current_level_view()->GetSize();

  m_config.save();
} // main_frame::save_config()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add an ingame_view_frame to the interface.
 * \param view The view to add.
 */
void bf::main_frame::add_level_view( ingame_view_frame* view )
{
  wxSize s;

  if ( m_windows_layout->get_current_level_view() != NULL )
    s = m_windows_layout->get_current_level_view()->GetSize();
  else
    s = m_config.default_level_window_size;

  view->SetSize(s);
  view->Show();
} // main_frame::add_level_view()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on "Configuration" menu.
 * \param event This event occured.
 */
void bf::main_frame::on_configuration_menu(wxCommandEvent& event)
{
  if ( m_config_frame->ShowModal() == wxID_OK )
    {
      // update the item_class_pool_frame
    }
} // main_frame::on_configuration_menu()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on "Item properties" in the window menu.
 * \param event This event occured.
 */
void bf::main_frame::on_item_properties_menu(wxCommandEvent& event)
{
  m_windows_layout->get_properties_frame().Show( event.IsChecked() );
} // main_frame::on_item_properties_menu()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on "Layer list" in the window menu.
 * \param event This event occured.
 */
void bf::main_frame::on_layer_list_menu(wxCommandEvent& event)
{
  m_windows_layout->get_layer_list_frame().Show( event.IsChecked() );
} // main_frame::on_layer_list_menu()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on "Item classes" in the window menu.
 * \param event This event occured.
 */
void bf::main_frame::on_item_class_pool_menu(wxCommandEvent& event)
{
  m_windows_layout->get_item_class_pool_frame().Show( event.IsChecked() );
} // main_frame::on_item_class_pool_menu()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on "New level".
 * \param event This event occured.
 */
void bf::main_frame::on_new_level(wxCommandEvent& event)
{
  level_properties_frame dlg(this);

  if ( dlg.ShowModal() == wxID_OK )
    {
      level* lvl =
        new level( dlg.get_width(), dlg.get_height(), dlg.get_camera_width(),
                   dlg.get_camera_height(), dlg.get_music() );

      add_level_view( new ingame_view_frame(*m_windows_layout, lvl) );
    }
} // main_frame::on_new_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on "Open level".
 * \param event This event occured.
 */
void bf::main_frame::on_open_level(wxCommandEvent& event)
{
  wxFileDialog dlg( this, _("Choose a level"), wxT(""), wxT(""),
                    _("Level files (*.lvl)|*.lvl"),
                    wxFD_OPEN | wxFD_FILE_MUST_EXIST );

  if ( dlg.ShowModal() == wxID_OK )
    {
      level_file_xml_reader reader;
      
      try
        {
          level* lvl = reader.load( dlg.GetPath() );

          add_level_view
            ( new ingame_view_frame(*m_windows_layout, lvl, dlg.GetPath()) );
        }
      catch( std::exception& e )
        {
          wxMessageDialog msg
            ( this, std_to_wx_string( e.what() ),
              _("Error in level loading."), wxOK );
      
          msg.ShowModal();
        }
    }
} // main_frame::on_open_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on "Save as".
 * \param event This event occured.
 */
void bf::main_frame::on_save_as(wxCommandEvent& event)
{
  m_windows_layout->get_current_level_view()->save_as();
} // main_frame::on_save_as()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on "Save".
 * \param event This event occured.
 */
void bf::main_frame::on_save(wxCommandEvent& event)
{
  m_windows_layout->get_current_level_view()->save();
} // main_frame::on_save()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on "Save all".
 * \param event This event occured.
 */
void bf::main_frame::on_save_all(wxCommandEvent& event)
{
  windows_layout::iterator it;

  for ( it=m_windows_layout->begin(); it!=m_windows_layout->end(); ++it )
    it->save();
} // main_frame::on_save_all()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on "Exit".
 * \param event This event occured.
 */
void bf::main_frame::on_exit(wxCommandEvent& event)
{
  Close();
} // main_frame::on_exit()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to the event that occurs when opening menus.
 * \param event This event occured.
 */
void bf::main_frame::on_menu_open(wxMenuEvent& event)
{
  GetMenuBar()->Check
    ( IDM_ITEM_PROPERTIES, m_windows_layout->get_properties_frame().IsShown() );

  GetMenuBar()->Check
    ( IDM_LAYER_LIST, m_windows_layout->get_layer_list_frame().IsShown() );

  GetMenuBar()->Check
    ( IDM_ITEM_CLASS_POOL,
      m_windows_layout->get_item_class_pool_frame().IsShown() );
} // main_frame::on_menu_open()

/*----------------------------------------------------------------------------*/
/**
 * \brief A menu entry is highlighted.
 * \param event This event occured.
 */
void bf::main_frame::on_menu_highlight(wxMenuEvent& event)
{
  ingame_view_frame* current_view = m_windows_layout->get_current_level_view();

  if ( current_view != NULL )
    current_view->GetStatusBar()->SetStatusText
      ( GetMenuBar()->GetHelpString(event.GetMenuId()), 0 );
} // main_frame::on_menu_highlight()

/*----------------------------------------------------------------------------*/
/**
 * \brief Procedure called when closing the window.
 * \param event This event occured.
 */
void bf::main_frame::on_close(wxCloseEvent& event)
{
  save_config();

  bool quit = !event.CanVeto();

  if ( !quit )
    {
      quit = true;

      bool changed = false;
      windows_layout::iterator it;
  
      for ( it=m_windows_layout->begin();
            !changed && it!=m_windows_layout->end(); ++it)
        changed = it->is_changed();

      if ( changed )
        {
          wxMessageDialog dlg
            ( this,
              _T("Some levels are not saved."
                 " Do you want to save them now?"),
              _T("Level is not saved."), wxYES_NO | wxCANCEL );
      
          int answer = dlg.ShowModal();

          if ( answer == wxID_CANCEL )
            quit = false;
          else if ( answer == wxID_YES )
            for ( it=m_windows_layout->begin();
                  quit && it!=m_windows_layout->end(); ++it)
              quit = it->save();
        }
    }

  if ( quit )
    event.Skip();
  else
    event.Veto();
} // main_frame::on_close()

/*----------------------------------------------------------------------------*/
BEGIN_EVENT_TABLE(bf::main_frame, wxFrame)
  EVT_BUTTON( wxID_NEW, bf::main_frame::on_new_level )
  EVT_BUTTON( wxID_OPEN, bf::main_frame::on_open_level )
  EVT_BUTTON( wxID_SAVE, bf::main_frame::on_save )

  EVT_MENU( wxID_PREFERENCES, bf::main_frame::on_configuration_menu )
  EVT_MENU( bf::main_frame::IDM_ITEM_PROPERTIES,
            bf::main_frame::on_item_properties_menu )
  EVT_MENU( bf::main_frame::IDM_LAYER_LIST,
            bf::main_frame::on_layer_list_menu )
  EVT_MENU( bf::main_frame::IDM_ITEM_CLASS_POOL,
            bf::main_frame::on_item_class_pool_menu )

  EVT_MENU( wxID_NEW, bf::main_frame::on_new_level )
  EVT_MENU( wxID_OPEN, bf::main_frame::on_open_level )
  EVT_MENU( wxID_SAVE, bf::main_frame::on_save )
  EVT_MENU( wxID_SAVEAS, bf::main_frame::on_save_as )
  EVT_MENU( bf::main_frame::IDM_SAVE_ALL_LEVELS, bf::main_frame::on_save_all )

  EVT_MENU( wxID_EXIT, bf::main_frame::on_exit )
  EVT_MENU_OPEN( bf::main_frame::on_menu_open )
  EVT_MENU_HIGHLIGHT_ALL( bf::main_frame::on_menu_highlight )

  EVT_CLOSE( bf::main_frame::on_close )
END_EVENT_TABLE()
