/*
  Plee The Bear - Level editor

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/code/level_history.cpp
 * \brief Implementation of the bf::level_history class.
 * \author Julien Jorge
 */
#include "bf/level_history.hpp"

#include <claw/assert.hpp>
#include <limits>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param lvl The initial level, will be deleted in the desctructor.
 */
bf::level_history::level_history( level* lvl )
  : m_level(lvl), m_last_saved_level(m_level),
    m_max_history( std::numeric_limits<std::size_t>::max() )
{
  CLAW_PRECOND(lvl != NULL);
} // level_history::level_history()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bf::level_history::~level_history()
{
  clear_past();
  clear_future();
  delete m_level;
} // level_history::~level_history()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if there is undo information.
 */
bool bf::level_history::can_undo() const
{
  return !m_past.empty();
} // level_history::can_undo()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if there is redo information.
 */
bool bf::level_history::can_redo() const
{
  return !m_future.empty();
} // level_history::can_redo()

/*----------------------------------------------------------------------------*/
/**
 * \brief Undo the last change.
 */
void bf::level_history::undo()
{
  if ( !m_past.empty() )
    {
      m_future.push_front(m_level);
      m_level = m_past.back();
      m_past.pop_back();
    }
} // level_history::undo()

/*----------------------------------------------------------------------------*/
/**
 * \brief Redo the last change.
 */
void bf::level_history::redo()
{
  if ( !m_future.empty() )
    {
      m_past.push_back( m_level );
      m_level = m_future.front();
      m_future.pop_front();
    }
} // level_history::redo()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the undo informations.
 */
void bf::level_history::push()
{
  if ( m_max_history != 0 )
    {
      level* level_copy = new level(*m_level);

      clear_future();

      if ( m_last_saved_level == m_level )
        m_last_saved_level = level_copy;

      if ( m_past.size() == m_max_history )
        {
          delete m_past.front();
          m_past.pop_front();
        }

      m_past.push_back( level_copy );
    }
} // level_history::push()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the undo informations and change the current level.
 */
void bf::level_history::push(level* lvl)
{
  if ( m_max_history != 0 )
    {
      clear_future();

      m_last_saved_level = lvl;

      if ( m_past.size() == m_max_history )
        {
          delete m_past.front();
          m_past.pop_front();
        }

      m_past.push_back( m_level );
      m_level = lvl;
    }
} // level_history::push()

/*----------------------------------------------------------------------------*/
/**
 * \brief Mark the current level as saved.
 */
void bf::level_history::set_saved()
{
  m_last_saved_level = m_level;
} // level_history::set_saved()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if we are on a saved state.
 */
bool bf::level_history::saved_state() const
{
  return m_level == m_last_saved_level;
} // level_history::saved_state()

/*----------------------------------------------------------------------------*/
/**
 * \brief Test if the saved state is in the past.
 */
bool bf::level_history::saved_state_in_past() const
{
  bool result = false;

  std::list<level*>::const_iterator it;

  for ( it = m_past.begin(); ( it != m_past.end() ) && ( !result ); it++)
    result = ( *it == m_last_saved_level );

  return result;
} // level_history::saved_state_in_past()

/*----------------------------------------------------------------------------*/
/**
 * \brief Test if the saved state is in the future.
 */
bool bf::level_history::saved_state_in_future() const
{
  bool result = false;

  std::list<level*>::const_iterator it;

  for ( it = m_future.begin(); ( it != m_future.end() ) && ( !result ); it++)
    result = ( *it == m_last_saved_level );

  return result;
} // level_history::saved_state_in_past()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the level in the current state.
 */
bf::level& bf::level_history::get_level()
{
  return *m_level;
} // level_history::get_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the level in the current state.
 */
const bf::level& bf::level_history::get_level() const
{
  return *m_level;
} // level_history::get_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove all instances of the past.
 */
void bf::level_history::clear_past()
{
  for ( ; !m_past.empty(); m_past.pop_front() )
    delete m_past.front();
} // level_history::clear_past()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove all instances of the future.
 */
void bf::level_history::clear_future()
{
  for ( ; !m_future.empty(); m_future.pop_front() )
    delete m_future.front();
} // level_history::clear_future()

