/*
    Plee The Bear - Level editor

    Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/layer_properties_frame.cpp
 * \brief Implementation of the bf::layer_properties_frame class.
 * \author Julien Jorge
 */
#include "bf/layer_properties_frame.hpp"

#include "bf/wx_facilities.hpp"

#include <limits>

/*----------------------------------------------------------------------------*/
const unsigned int bf::layer_properties_frame::s_min_width(800);
const unsigned int bf::layer_properties_frame::s_min_height(600);

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param parent The window owning this window.
 */
bf::layer_properties_frame::layer_properties_frame( wxWindow* parent )
  : wxDialog(parent, wxID_ANY, _("Layer size")), m_width(s_min_width),
    m_height(s_min_height)
{
  create_controls();
  Fit();
} // layer_properties_frame::layer_properties_frame()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the width of the layer.
 */
unsigned int bf::layer_properties_frame::get_layer_width() const
{
  return m_width;
} // layer_properties_frame::get_layer_width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the height of the layer.
 */
unsigned int bf::layer_properties_frame::get_layer_height() const
{
  return m_height;
} // layer_properties_frame::get_layer_height()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the name of the class of the layer.
 */
const std::string& bf::layer_properties_frame::get_layer_class_name() const
{
  return m_class_name;
} // layer_properties_frame::get_layer_class_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the properties of the layer.
 * \param lay The layer from which we take the info.
 *
 * The new width is max(lay.get_width(), s_min_width), and the new height
 * max(lay.get_height(), s_min_height).
 */
void bf::layer_properties_frame::fill_from( const layer& lay )
{
  m_width = std::max(lay.get_width(), s_min_width);
  m_height = std::max(lay.get_height(), s_min_height);
  m_class_name = lay.get_class_name();

  fill_controls();
} // layer_properties_frame::set_layer_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Fill the controls with the values of the layer.
 */
void bf::layer_properties_frame::fill_controls()
{
  m_width_text->SetValue( wxString::Format(wxT("%d"), m_width) );
  m_height_text->SetValue( wxString::Format(wxT("%d"), m_height) );

  unsigned int i=0;
  bool found = false;
  const wxString ref( std_to_wx_string(m_class_name) );

  while ( !found && (i!=m_class_name_choice->GetCount()) )
    if ( m_class_name_choice->GetString(i) == ref )
      found = true;
    else
      ++i;

  if ( found )
    m_class_name_choice->SetSelection(i);
  else
    {
      m_class_name_choice->SetSelection(0);
      m_class_name =
        wx_to_std_string( m_class_name_choice->GetStringSelection() );
    }
} // layer_properties_frame::fill_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls of the window.
 */
void bf::layer_properties_frame::create_controls()
{
  create_member_controls();
  create_sizer_controls();
  fill_controls();
} // layer_properties_frame::create_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief for which we keep a reference.
 */
void bf::layer_properties_frame::create_member_controls()
{
  wxArrayString layer_names;

  layer_names.Add( wxT("action_layer") );
  layer_names.Add( wxT("decoration_layer") );
  layer_names.Add( wxT("pattern_layer") );

  m_width_text = new wxSpinCtrl( this, wxID_ANY );
  m_height_text = new wxSpinCtrl( this, wxID_ANY );
  m_class_name_choice = new wxChoice
    ( this, wxID_ANY, wxDefaultPosition, wxDefaultSize, layer_names );

  m_width_text->SetRange( s_min_width, std::numeric_limits<int>::max() );
  m_height_text->SetRange( s_min_height, std::numeric_limits<int>::max() );
} // layer_properties_frame::create_member_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls and add them in sizers.
 */
void bf::layer_properties_frame::create_sizer_controls()
{
  wxBoxSizer* sizer = new wxBoxSizer( wxVERTICAL );

  // width
  wxBoxSizer* s_sizer = new wxBoxSizer( wxHORIZONTAL );

  s_sizer->Add( new wxStaticText(this, wxID_ANY, _("Width")),
                1, wxEXPAND | wxALL, 5 );
  s_sizer->Add( m_width_text, 0, wxALL, 5 );

  sizer->Add( s_sizer, 0, wxEXPAND );

  // height
  s_sizer = new wxBoxSizer( wxHORIZONTAL );

  s_sizer->Add( new wxStaticText(this, wxID_ANY, _("Height")),
                1, wxEXPAND | wxALL, 5 );
  s_sizer->Add( m_height_text, 0, wxALL, 5 );
  sizer->Add( s_sizer, 0, wxEXPAND );

  s_sizer = new wxBoxSizer( wxHORIZONTAL );

  s_sizer->Add( new wxStaticText(this, wxID_ANY, _("Class name")),
                1, wxEXPAND | wxALL, 5 );
  s_sizer->Add( m_class_name_choice, 0, wxEXPAND );
  sizer->Add( s_sizer, 0, wxEXPAND );

  sizer->Add( CreateStdDialogButtonSizer(wxOK | wxCANCEL), 0,
              wxALL | wxCENTER, 5 );

  SetSizer(sizer);
} // layer_properties_frame::create_sizer_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the "Ok" button.
 * \param event This event occured.
 */
void bf::layer_properties_frame::on_ok(wxCommandEvent& event)
{
  m_width = m_width_text->GetValue();
  m_height = m_height_text->GetValue();
  m_class_name = wx_to_std_string( m_class_name_choice->GetStringSelection() );

  EndModal(wxID_OK);
} // layer_properties_frame::on_ok()

/*----------------------------------------------------------------------------*/
BEGIN_EVENT_TABLE(bf::layer_properties_frame, wxDialog)
  EVT_BUTTON( wxID_OK, bf::layer_properties_frame::on_ok )
END_EVENT_TABLE()
