/*
    Plee The Bear - Level editor

    Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/code/layer_list_frame.cpp
 * \brief Implementation of the bf::layer_list_frame class.
 * \author Julien Jorge
 */
#include "bf/layer_list_frame.hpp"

#include "bf/item_class_pool.hpp"
#include "bf/item_class_pool_frame.hpp"
#include "bf/layer_properties_frame.hpp"
#include "bf/wx_facilities.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param parent The frame owning this one.
 */
bf::layer_list_frame::layer_list_frame( wxWindow* parent )
  : wxFrame( parent, wxID_ANY, _("Layers") ), m_level_view(NULL),
    m_layer_list(NULL), m_new_layer_button(NULL), m_windows_layout(NULL)
{
  create_controls();
} // layer_list_frame::layer_list_frame()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the level on which we work.
 * \param view The level view in which we take the layers.
 */
void bf::layer_list_frame::set_level_view( ingame_view* view )
{
  m_level_view = view;
  fill();
} // layer_list_frame::set_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the windows layout of the program.
 * \param layout The layout.
 */
void bf::layer_list_frame::set_window_layout( windows_layout& layout )
{
  m_windows_layout = &layout;
} // layer_list_frame::set_window_layout()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls of the window.
 */
void bf::layer_list_frame::create_controls()
{
  create_member_controls();
  create_sizer_controls();
} // layer_list_frame::create_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls for which we keep a reference.
 */
void bf::layer_list_frame::create_member_controls()
{
  m_layer_list = new wxCheckListBox( this, IDC_LAYER_LIST );
  m_new_layer_button = new wxButton(this, wxID_NEW);

  m_new_layer_button->SetLabel(wxT(""));
} // layer_list_frame::create_member_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls and add them in sizers.
 */
void bf::layer_list_frame::create_sizer_controls()
{
  wxBoxSizer* sizer = new wxBoxSizer( wxHORIZONTAL );

  // the sizer for the buttons
  wxBoxSizer* s_sizer = new wxBoxSizer( wxVERTICAL );

  s_sizer->Add( m_new_layer_button, 0, wxEXPAND | wxALL, 0 );
  s_sizer->Add( new wxButton(this, wxID_DELETE), 0, wxEXPAND | wxALL, 0 );
  s_sizer->Add( new wxButton(this, wxID_UP), 0, wxEXPAND | wxALL, 0 );
  s_sizer->Add( new wxButton(this, wxID_DOWN), 0, wxEXPAND | wxALL, 0 );
  s_sizer->Add( new wxButton(this, wxID_PROPERTIES), 0, wxEXPAND | wxALL, 0 );

  sizer->Add( m_layer_list, 1, wxEXPAND );
  sizer->Add( s_sizer, 0, wxEXPAND );

  SetSizer(sizer);
} // layer_list_frame::create_sizer_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Clear the list of layers.
 */
void bf::layer_list_frame::clear()
{
  m_layer_list->Clear();
} // layer_list_frame::clear()

/*----------------------------------------------------------------------------*/
/**
 * \brief Fill the list with the layers of the level.
 */
void bf::layer_list_frame::fill()
{
  clear();

  if ( m_level_view != NULL )
    if ( !m_level_view->empty() )
      {
        const level& lvl = m_level_view->get_level();

        for (unsigned int i=0; i!=lvl.layers_count(); ++i)
          {
            m_layer_list->Append
              ( std_to_wx_string( lvl.get_layer(i).get_class_name() ) );
            m_layer_list->Check( i, m_level_view->get_visible(i) );
          }

        m_layer_list->SetSelection( m_level_view->get_active_index() );
    }

  update_windows_layout();
} // layer_list_frame::fill()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the other windows according to the selection in this window.
 */
void bf::layer_list_frame::update_windows_layout() const
{
  if ( m_windows_layout != NULL )
    {
      int index = m_layer_list->GetSelection();

      if ( index != wxNOT_FOUND )
        {
          m_windows_layout->get_item_class_pool_frame().set_layer
            ( m_layer_list->GetString(index) );
        }
      else
        m_windows_layout->get_item_class_pool_frame().set_layer( _("") );
    }
} // layer_list_frame::update_windows_layout()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on a layer menu entry.
 * \param event This event occured.
 */
void bf::layer_list_frame::on_new_layer(wxCommandEvent& event)
{
  if (m_level_view != NULL)
    {
      layer_properties_frame dlg(this);

      if ( dlg.ShowModal() == wxID_OK )
        {
          m_level_view->add_layer
            ( dlg.get_layer_width(), dlg.get_layer_height(),
              dlg.get_layer_class_name() );
      
          fill();
        }
    }
} // layer_list_frame::on_new_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the button "delete".
 * \param event This event occured.
 */
void bf::layer_list_frame::on_delete(wxCommandEvent& event)
{
  if (m_level_view != NULL)
    if ( !m_level_view->empty() )
      {
        m_level_view->remove_layer();
        fill();
      }
} // layer_list_frame::on_delete()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the button "Properties".
 * \param event This event occured.
 */
void bf::layer_list_frame::on_show_properties(wxCommandEvent& event)
{
  int index = m_layer_list->GetSelection();

  if (index != wxNOT_FOUND)
    {
      layer_properties_frame dlg(this);
      const layer& the_layer = m_level_view->get_active_layer();

      dlg.fill_from(the_layer);

      if ( dlg.ShowModal() == wxID_OK )
        m_level_view->change_layer
          ( dlg.get_layer_width(), dlg.get_layer_height(),
            dlg.get_layer_class_name() );
    }
} // layer_list_frame::on_show_properties()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the visibility of a layer.
 * \param event This event occured.
 */
void bf::layer_list_frame::on_toggle_layer(wxCommandEvent& event)
{
  // I did not find any way to get the index of the changed item, so I update
  // all layers

  for (unsigned int i=0; i!=m_layer_list->GetCount(); ++i)
    m_level_view->set_visible( i, m_layer_list->IsChecked(i) );

  m_level_view->Refresh();
} // layer_list_frame::on_toggle_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform all windows about the selected layer.
 * \param event This event occured.
 */
void bf::layer_list_frame::on_select_layer(wxCommandEvent& event)
{
  if ( event.IsSelection() )
    {
      if ( m_level_view != NULL )
        m_level_view->set_active_index(m_layer_list->GetSelection());

      update_windows_layout();
    }
} // layer_list_frame::on_select_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the button "up".
 * \param event This event occured.
 */
void bf::layer_list_frame::on_move_up(wxCommandEvent& event)
{
  if (m_level_view != NULL)
    {
      m_level_view->move_backward();
      fill();
    }
} // layer_list_frame::on_move_up()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the button "down".
 * \param event This event occured.
 */
void bf::layer_list_frame::on_move_down(wxCommandEvent& event)
{
  if (m_level_view != NULL)
    {
      m_level_view->move_forward();
      fill();
    }
} // layer_list_frame::on_move_down()

/*----------------------------------------------------------------------------*/
/**
 * \brief Procedure called when closing the window.
 * \param event This event occured.
 */
void bf::layer_list_frame::on_close(wxCloseEvent& event)
{
  if ( event.CanVeto() )
    {
      Hide();
      event.Veto();
    }
} // layer_list_frame::on_close()

/*----------------------------------------------------------------------------*/
BEGIN_EVENT_TABLE(bf::layer_list_frame, wxFrame)
  EVT_BUTTON( wxID_NEW, bf::layer_list_frame::on_new_layer )
  EVT_BUTTON( wxID_DELETE, bf::layer_list_frame::on_delete )
  EVT_BUTTON( wxID_UP, bf::layer_list_frame::on_move_up )
  EVT_BUTTON( wxID_DOWN, bf::layer_list_frame::on_move_down )
  EVT_BUTTON( wxID_PROPERTIES, bf::layer_list_frame::on_show_properties )

  EVT_LISTBOX_DCLICK( bf::layer_list_frame::IDC_LAYER_LIST,
                      bf::layer_list_frame::on_show_properties )
  EVT_CHECKLISTBOX( bf::layer_list_frame::IDC_LAYER_LIST,
                    bf::layer_list_frame::on_toggle_layer )
  EVT_LISTBOX( bf::layer_list_frame::IDC_LAYER_LIST,
               bf::layer_list_frame::on_select_layer )

  EVT_CLOSE( bf::layer_list_frame::on_close )
END_EVENT_TABLE()
