/*
    Plee The Bear - Level editor

    Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/code/item_class_xml_parser.cpp
 * \brief Implementation of the bf::item_class_xml_parser.
 * \author Julien Jorge
 */
#include "bf/item_class_xml_parser.hpp"

#include "bf/xml_exception.hpp"
#include "bf/wx_facilities.hpp"

#include <iostream>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param file_path The path of the XML file to parse.
 */
bf::item_class_xml_parser::item_class_xml_parser( const std::string& file_path )
{
  wxXmlDocument doc;

  if ( !doc.Load( std_to_wx_string(file_path) ) )
    throw std::ios_base::failure
      ( "Cannot load the XML file '" + file_path + "'" );

  parse_item_node( doc.GetRoot() );
} // item_class_xml_parser::item_class_xml_parser()

/*----------------------------------------------------------------------------*/
/**
 * \brief Parse a node of the file.
 * \param node The node to explore.
 */
void bf::item_class_xml_parser::parse_item_node( const wxXmlNode* node )
{
  if ( node->GetName() != wxT("item") )
    throw xml_bad_node( wx_to_std_string(node->GetName()) );

  read_item_properties( node );

  for ( node=node->GetChildren(); node!=NULL; node=node->GetNext() )
    if ( node->GetName() == wxT("fields") )
      read_item_fields( node );
    else if ( node->GetName() == wxT("inherit") )
      read_inherit( node );
    else
      throw xml_bad_node( wx_to_std_string(node->GetName()) );
} // item_class_xml_parser::parse_item_node()

/*----------------------------------------------------------------------------*/
/**
 * \brief Read the properties of the item.
 * \param node The node to explore.
 */
void bf::item_class_xml_parser::read_item_properties( const wxXmlNode* node )
{
  wxString val;

  if ( !node->GetPropVal( wxT("class"), &val ) )
    throw xml_missing_property("class");

  item.set_class_name( wx_to_std_string(val) );

  if ( !node->GetPropVal( wxT("category"), &val ) )
    throw xml_missing_property("category");

  item.set_category( wx_to_std_string(val) );

  item.set_color
    ( wx_to_std_string(node->GetPropVal( wxT("box_color"), wxT("#00FF00") )) );

  item.set_fixable
    ( node->GetPropVal( wxT("fixable"), wxT("true") ) == wxT("true") );
} // item_class_xml_parser::read_item_properties()

/*----------------------------------------------------------------------------*/
/**
 * \brief Read the fields of the item.
 * \param node The node to explore.
 */
void bf::item_class_xml_parser::read_item_fields( const wxXmlNode* node )
{
  for ( node=node->GetChildren(); node!=NULL; node=node->GetNext() )
    if ( node->GetName() == wxT("field") )
      read_field_type( node );
    else
      throw xml_bad_node( wx_to_std_string(node->GetName()) );
} // item_class_xml_parser::read_item_fields()

/*----------------------------------------------------------------------------*/
/**
 * \brief Read the super classes of the item.
 * \param node The node to explore.
 */
void bf::item_class_xml_parser::read_inherit( const wxXmlNode* node )
{
  for ( node=node->GetChildren(); node!=NULL; node=node->GetNext() )
    if ( node->GetName() == wxT("class") )
      item.add_super_class
        ( wx_to_std_string(node->GetNodeContent() ) );
    else
      throw xml_bad_node( wx_to_std_string(node->GetName()) );
} // item_class_xml_parser::read_inherit()

/*----------------------------------------------------------------------------*/
/**
 * \brief Read the definition of a field of the item.
 * \param node The node to explore.
 */
void bf::item_class_xml_parser::read_field_type( const wxXmlNode* node )
{
  wxString val;
  std::string name;

  if ( !node->GetPropVal( wxT("name"), &val ) )
    throw xml_missing_property("name");

  name = wx_to_std_string(val);

  if ( !node->GetPropVal( wxT("type"), &val ) )
    throw xml_missing_property("type");

  type_field* field;

  if ( val == wxT("integer") )
    field = add_integer_field( name, node->GetChildren() );
  else if ( val == wxT("u_integer") )
    field = add_u_integer_field( name, node->GetChildren() );
  else if ( val == wxT("real") )
    field = add_real_field( name, node->GetChildren() );
  else if ( val == wxT("string") )
    field = add_string_field( name, node->GetChildren() );
  else if ( val == wxT("boolean") )
    field = add_boolean_field( name, node->GetChildren() );
  else if ( val == wxT("sprite") )
    field = add_sprite_field( name, node->GetChildren() );
  else if ( val == wxT("animation") )
    field = add_animation_field( name, node->GetChildren() );
  else if ( val == wxT("item_reference") )
    field = add_item_reference_field( name, node->GetChildren() );
  else
    throw xml_bad_value( wx_to_std_string(val) );

  field->set_required
    ( node->GetPropVal( wxT("required"), wxT("false") ) == wxT("true") );
  field->set_is_list
    ( node->GetPropVal( wxT("list"), wxT("false") ) == wxT("true") );

  item.add_field(name, *field);
  delete field;
} // item_class_xml_parser::read_field_type()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a field of type "integer" to the item.
 * \param node The node to explore.
 */
bf::type_field* bf::item_class_xml_parser::add_integer_field
( const std::string& name, const wxXmlNode* node ) const
{
  return
    create_field<int, true, true>(name, type_field::integer_field_type, node);
} // item_class_xml_parser::add_integer_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a field of type "u_integer" to the item.
 * \param node The node to explore.
 */
bf::type_field* bf::item_class_xml_parser::add_u_integer_field
( const std::string& name, const wxXmlNode* node ) const
{
  return create_field<unsigned int, true, true>
    (name, type_field::u_integer_field_type, node);
} // item_class_xml_parser::add_u_integer_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a field of type "real" to the item.
 * \param node The node to explore.
 */
bf::type_field* bf::item_class_xml_parser::add_real_field
( const std::string& name, const wxXmlNode* node ) const
{
  return
    create_field<double, true, true>(name, type_field::real_field_type, node);
} // item_class_xml_parser::add_real_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a field of type "string" to the item.
 * \param node The node to explore.
 */
bf::type_field* bf::item_class_xml_parser::add_string_field
( const std::string& name, const wxXmlNode* node ) const
{
  return
    create_field<int, true, false>(name, type_field::string_field_type, node);
} // item_class_xml_parser::add_string_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a field of type "boolean" to the item.
 * \param node The node to explore.
 */
bf::type_field* bf::item_class_xml_parser::add_boolean_field
( const std::string& name, const wxXmlNode* node ) const
{
  return
    create_field<int, false, false>(name, type_field::boolean_field_type, node);
} // item_class_xml_parser::add_boolean_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a field of type "sprite" to the item.
 * \param node The node to explore.
 */
bf::type_field* bf::item_class_xml_parser::add_sprite_field
( const std::string& name, const wxXmlNode* node ) const
{
  return
    create_field<int, false, false>(name, type_field::sprite_field_type, node);
} // item_class_xml_parser::add_sprite_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a field of type "animation" to the item.
 * \param node The node to explore.
 */
bf::type_field* bf::item_class_xml_parser::add_animation_field
( const std::string& name, const wxXmlNode* node ) const
{
  return create_field<int, false, false>
    (name, type_field::animation_field_type, node);
} // item_class_xml_parser::add_animation_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a field of type "item_reference" to the item.
 * \param node The node to explore.
 */
bf::type_field* bf::item_class_xml_parser::add_item_reference_field
( const std::string& name, const wxXmlNode* node ) const
{
  return create_field<int, false, false>
    (name, type_field::item_reference_field_type, node);
} // item_class_xml_parser::add_animation_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Read the "field" property of a node "before".
 * \param node The node to explore.
 */
std::string
bf::item_class_xml_parser::read_before( const wxXmlNode* node ) const
{
  wxString val;

  if ( !node->GetPropVal( wxT("field"), &val ) )
    throw xml_missing_property("field");

  return wx_to_std_string(val);
} // item_class_xml_parser::read_before()

/*----------------------------------------------------------------------------*/
/**
 * \brief Read the elements of a set of values.
 * \param node The node to explore.
 */
void bf::item_class_xml_parser::read_set
( const wxXmlNode* node, std::list<std::string>& set ) const
{
  for ( node=node->GetChildren(); node!=NULL; node = node->GetNext() )
    if ( node->GetName() == wxT("element") )
      {
        wxString val;

        if ( !node->GetPropVal( wxT("value"), &val ) )
          throw xml_missing_property("value");

        set.push_back( wx_to_std_string(val) );
      }
    else
      throw xml_bad_node( wx_to_std_string(node->GetName()) );
} // item_class_xml_parser::read_set()
