/*
  Plee The Bear - Level editor

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/code/item_class_pool_frame.cpp
 * \brief Implementation of the bf::item_class_pool_frame class.
 * \author Julien Jorge
 */
#include "bf/item_class_pool_frame.hpp"

#include "bf/item_class_pool.hpp"
#include "bf/wx_facilities.hpp"
#include "bf/ingame_view_frame.hpp"

#include <list>
#include <wx/dnd.h>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param parent Pointer to the owner.
 */
bf::item_class_pool_frame::item_class_pool_frame( wxFrame* parent )
  : wxFrame( parent, wxID_ANY, _("Item classes") ), m_tree(NULL)
{
  SetSizer( new wxBoxSizer(wxVERTICAL) );

  create_tree();
} // item_class_pool_frame::item_class_pool_frame()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the layer class for which we want the items.
 * \param layer_name The name of the layer class.
 */
void bf::item_class_pool_frame::set_layer( const wxString& layer_name )
{
  Layout();
} // item_class_pool_frame::set_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the windows layout of the program.
 * \param layout The layout.
 */
void bf::item_class_pool_frame::set_window_layout( windows_layout& layout )
{
  m_windows_layout = &layout;
} // item_class_pool_frame::set_window_layout()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create a wxTreeCtrl and add it in the frame.
 */
wxTreeCtrl* bf::item_class_pool_frame::add_tree()
{
  wxTreeCtrl* result = new wxTreeCtrl(this);

  GetSizer()->Add( result, 1, wxEXPAND );
  GetSizer()->Hide( result );

  result->Connect( wxEVT_MOTION,
                   wxMouseEventHandler(item_class_pool_frame::on_mouse_move),
                   NULL, this );
  result->Connect( wxEVT_LEFT_DCLICK,
                   wxMouseEventHandler(item_class_pool_frame::on_double_click),
                   NULL, this );
  result->Connect( wxEVT_KEY_UP,
                   wxKeyEventHandler(item_class_pool_frame::on_key_up),
                   NULL, this );

  return result;
} // item_class_pool_frame::add_tree()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the tree of classes for each layer.
 */
void bf::item_class_pool_frame::create_tree()
{
  m_tree = add_tree();
  m_tree->AddRoot( _("Items") );
  
  tree_builder tb;
  
  create_categories_tree(tb);
  tb.create_wxTree( *m_tree );
  m_tree->Expand( m_tree->GetRootItem() );
  
  GetSizer()->Show( m_tree );
} // item_class_pool_frame::create_tree()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the tree of classes.
 * \param tb (out) The tree builder.
 */
void bf::item_class_pool_frame::create_categories_tree( tree_builder& tb ) const
{
  const item_class_pool& class_pool = item_class_pool::get_instance();

  item_class_pool::const_iterator it = class_pool.begin();
  const item_class_pool::const_iterator eit = class_pool.end();

  for ( ; it!=eit; ++it )
    if ( it->get_category() != "-abstract-" )
      {
        const std::string cat( it->get_category() );

        if ( cat.empty() )
          tb.add_entries( it->get_class_name(), '/' );
        else
          tb.add_entries( cat + '/' + it->get_class_name(), '/' );
      }
} // item_class_pool_frame::create_categories_tree()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create an item.
 * \param all_children Tell if we expand the whole subtree (if selected item is
 *        not a leaf).
 */
void bf::item_class_pool_frame::select_class(bool all_children)
{
  wxTreeItemId item = m_tree->GetSelection();

  if ( item.IsOk() )
    if ( m_tree->ItemHasChildren(item) )
      {
        if ( m_tree->IsExpanded(item) )
          {
            if ( all_children )
              m_tree->CollapseAllChildren(item);
            else
              m_tree->Collapse(item);
          }
        else 
          {
            if ( all_children )
              m_tree->ExpandAllChildren(item);
            else
              m_tree->Expand(item);
          }
      }
    else if ( m_windows_layout->get_current_level_view() != NULL )
      {
        std::string class_name =
          wx_to_std_string(m_tree->GetItemText(item));
        
        m_windows_layout->get_current_level_view()->get_ingame_view()->add_item
          (class_name);
      }
} // item_class_pool_frame::select_class()

/*----------------------------------------------------------------------------*/
/**
 * \brief Procedure called when closing the window.
 * \param event This event occured.
 */
void bf::item_class_pool_frame::on_close(wxCloseEvent& event)
{
  if ( event.CanVeto() )
    {
      Hide();
      event.Veto();
    }
} // item_class_pool_frame::on_close()

/*----------------------------------------------------------------------------*/
/**
 * \brief Procedure called when the mouse moves over a tree.
 * \param event This event occured.
 */
void bf::item_class_pool_frame::on_mouse_move(wxMouseEvent& event)
{
  if ( event.LeftIsDown() )
    {
      wxTreeItemId item = m_tree->GetSelection();

      if ( item.IsOk() )
        if ( !m_tree->ItemHasChildren(item) )
          {
            wxTextDataObject class_name(m_tree->GetItemText(item));
            wxDropSource drag_source( this );

            drag_source.SetData( class_name );
            drag_source.DoDragDrop( TRUE );
          }
    }
  else
    event.Skip();
} // item_class_pool_frame::on_mouse_move()

/*----------------------------------------------------------------------------*/
/**
 * \brief Procedure called when a double click occurs on a tree.
 * \param event This event occured.
 */
void bf::item_class_pool_frame::on_double_click(wxMouseEvent& event)
{
  select_class(event.ShiftDown());
} // item_class_pool_frame::on_double_click()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user pressed a keybord key.
 * \param event The keyboard event that occured.
 */
void bf::item_class_pool_frame::on_key_up(wxKeyEvent& event)
{
  switch( event.GetKeyCode() )
    {
    case WXK_RETURN:
    case WXK_NUMPAD_ENTER:
      select_class( event.ShiftDown() );
      break;
    default:
      event.Skip();
    }
} // item_class_pool_frame::on_key_up()

/*----------------------------------------------------------------------------*/
BEGIN_EVENT_TABLE(bf::item_class_pool_frame, wxFrame)
  EVT_CLOSE( bf::item_class_pool_frame::on_close )
END_EVENT_TABLE()
