/*
  Plee The Bear - Level editor

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/code/item_class.cpp
 * \brief Implementation of the bf::item_class class.
 * \author Julien Jorge
 */
#include "bf/item_class.hpp"
#include "bf/item_class_pool.hpp"

#include <claw/logger.hpp>
#include <claw/assert.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bf::item_class::item_class()
{

} // item_class::item_class()

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy constructor.
 * \param that The item to copy from.
 */
bf::item_class::item_class( const item_class& that )
{
  copy(that);
} // item_class::item_class()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bf::item_class::~item_class()
{
  clear();
} // item_class::~item_class()

/*----------------------------------------------------------------------------*/
/**
 * \brief Assignment.
 * \param that The item to copy from.
 */
bf::item_class& bf::item_class::operator=( const item_class& that )
{
  clear();
  copy(that);

  return *this;
} // item_class::operator=()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a field to this item.
 * \param name The name of the field.
 * \param field Pointer on the field to add.
 * \remark \a field will be deleted by the destructor.
 */
void
bf::item_class::add_field( const std::string& name, const type_field& field )
{
  if ( m_field.find(name) != m_field.end() )
    claw::logger << claw::log_error << "Field '" << name << "' already exists."
                 << claw::lendl;
  else
    m_field[name] = field.clone();
} // item_class::add_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the name of the class.
 * \param class_name The name of the class.
 */
void bf::item_class::set_class_name( const std::string& class_name )
{
  m_class_name = class_name;
} // item_class::set_class_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the category of the class.
 * \param category The category of the class.
 */
void bf::item_class::set_category( const std::string& category )
{
  m_category = category;
} // item_class::set_category()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the color used for displaying the item.
 * \param color The color to use.
 */
void bf::item_class::set_color( const std::string& color )
{
  m_color = color;
} // item_class::set_color()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item can be fixed.
 * \param fixed The value to set.
 */
void bf::item_class::set_fixable( bool fixable )
{
  m_fixable = fixable;
} // item_class::set_fixable()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a class from which this one inherits.
 * \param super_class The name of the super class.
 */
void bf::item_class::add_super_class
( const std::string& super_class )
{
  m_super_classes.push_back(super_class);
} // item_class::add_super_class()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the description of a field, given its name.
 * \param field_name The name of the field.
 */
const bf::type_field&
bf::item_class::get_field( const std::string& field_name ) const
{
  const type_field* result = search_field(field_name);

  CLAW_ASSERT
    ( result != NULL,
      "None of the class in the hierarchy have a field with this name." );

  return *result;
} // item_class::get_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the description of a field, given its name.
 * \param field_name The name of the field.
 */
const bf::type_field*
bf::item_class::search_field( const std::string& field_name ) const
{
  field_map_type::const_iterator it = m_field.find(field_name);
  const bf::type_field* result = NULL;

  if ( it != m_field.end() )
    result = it->second;
  else
    {
      std::list<std::string>::const_iterator it_p;
      const item_class_pool& pool = item_class_pool::get_instance();

      for ( it_p=m_super_classes.begin();
            (result == NULL) && it_p!=m_super_classes.end(); ++it_p )
        result = pool.get_item_class(*it_p).search_field(field_name);
    }

  return result;
} // item_class::search_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the name of the class.
 */
const std::string& bf::item_class::get_class_name() const
{
  return m_class_name;
} // item_class::get_class_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the category of the class.
 */
const std::string& bf::item_class::get_category() const
{
  return m_category;
} // item_class::get_category()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the color used for displaying the item.
 */
const std::string& bf::item_class::get_color() const
{
  return m_color;
} // item_class::get_color()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item can be fixable.
 */
bool bf::item_class::get_fixable() const
{
  bool result = m_fixable;
  
  std::list<std::string>::const_iterator it;
  for ( it = m_super_classes.begin(); 
        ( it != m_super_classes.end() ) && result; ++it)
    result = result && 
      item_class_pool::get_instance().get_item_class(*it).get_fixable();

  return result;
} // item_class::get_fixable()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the classes from which this one inherits.
 */
const std::list<std::string>& bf::item_class::get_super_classes() const
{
  return m_super_classes;
} // item_class::get_super_classes()

/*----------------------------------------------------------------------------*/
/**
 * \brief Find the whole class hierarchy of the class.
 * \param hierarchy (out) The classes from which \a class_name inherit.
 */
void bf::item_class::find_hierarchy
( std::list<std::string>& hierarchy ) const
{
  const item_class_pool& pool = item_class_pool::get_instance();

  std::list<std::string> pending_classes;
  pending_classes.push_back( m_class_name );

  while( !pending_classes.empty() )
    {
      std::string current = pending_classes.front();

      pending_classes.pop_front();
      hierarchy.push_front( current );

      std::list<std::string>::const_iterator it;

      for ( it=pool.get_item_class(current).get_super_classes().begin();
            it!=pool.get_item_class(current).get_super_classes().end(); ++it )
        pending_classes.push_back(*it);
    }
} // item_class::find_hierachy()


/*----------------------------------------------------------------------------*/
/**
 * \brief Test the unicity of each field.
 * Return True if the class is valid.
 * \param error_msg The message of the error.
 */
bool bf::item_class::field_unicity_test(std::string& error_msg) const
{
  bool result = true;
  std::set<std::string> fields;
  const item_class_pool& pool = item_class_pool::get_instance();
  
  std::list<std::string>::iterator super_class;
  std::list<std::string> hierarchy;
  find_hierarchy( hierarchy );

  for ( super_class = hierarchy.begin(); 
        ( super_class != hierarchy.end() ) && result; super_class++)
    if ( pool.has_item_class(*super_class) )
      {
        field_iterator it;
        
        const item_class& current = pool.get_item_class(*super_class);
        for ( it = current.field_begin(); 
              ( it != current.field_end() ) && result; it++ )
          if ( fields.find(it->get_name()) != fields.end() )
            {
              result = false;
              error_msg = "the field '" + it->get_name() + 
                "' is already defined in the super class '"
                + current.get_class_name() + "'";
            }
          else
            fields.insert(it->get_name());
      }  
  
  return result;
} // item_class::field_unicity_test()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if this class, or one of the parent classes, has a field with a
 *        given name.
 * \param name The name of the field we are looking for.
 */
bool bf::item_class::has_field( const std::string& name ) const
{
  bool result = false;
  field_map_type::const_iterator it = m_field.find( name );

  if ( it != m_field.end() )
    result = true;
  else if ( !m_super_classes.empty() )
    {
      std::list<std::string>::const_iterator it;

      for ( it = m_super_classes.begin(); 
            !result && (it != m_super_classes.end()); ++it)
        {
          const item_class& parent =
            item_class_pool::get_instance().get_item_class(*it);
          result = parent.has_field( name );
        }
    }

  return result;
} // item_class::has_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if this class, or one of the parent classes, has a field of a
 *        given type with a given name.
 * \param name The name of the field we are looking for.
 * \param t The desired type.
 */
bool bf::item_class::has_field
( const std::string& name, type_field::field_type t ) const
{
  bool result = false;
  field_map_type::const_iterator it = m_field.find( name );

  if ( it != m_field.end() )
    result = it->second->get_field_type() == t;
  else if ( !m_super_classes.empty() )
    {
      std::list<std::string>::const_iterator it;

      for ( it = m_super_classes.begin();
            !result && (it != m_super_classes.end()); ++it)
        {
          const item_class& parent =
            item_class_pool::get_instance().get_item_class(*it);
          result = parent.has_field( name, t );
        }
    }

  return result;
} // item_class::has_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove all fields.
 */
void bf::item_class::clear()
{
  field_map_type::const_iterator it;

  for (it=m_field.begin(); it!=m_field.end(); ++it)
    delete it->second;

  m_field.clear();
} // item_class::clear()

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy all fields from an other item class.
 * \param that The item to copy from.
 */
void bf::item_class::copy( const item_class& that )
{
  m_class_name     = that.m_class_name;
  m_category       = that.m_category;
  m_color          = that.m_color;
  m_fixable        = that.m_fixable;

  std::list<std::string>::const_iterator it1;
  /*
    for (it1=that.m_layers.begin(); it1!=that.m_layers.end(); ++it1)
    m_layers.push_back(*it1);
  */
  for (it1=that.m_super_classes.begin(); 
       it1!=that.m_super_classes.end(); ++it1)
    m_super_classes.push_back(*it1);
    
  field_map_type::const_iterator it2;

  for (it2=that.m_field.begin(); it2!=that.m_field.end(); ++it2)
    m_field[it2->first] = it2->second->clone();
} // item_class::copy()

