/*
    Plee The Bear - Level editor

    Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/code/ingame_view_frame.cpp
 * \brief Implementation of the bf::ingame_view_frame class.
 * \author Julien Jorge
 */
#include "bf/ingame_view_frame.hpp"

#include "bf/grid_properties_frame.hpp"
#include "bf/layer_list_frame.hpp"
#include "bf/level_properties_frame.hpp"
#include "bf/main_frame.hpp"
#include "bf/properties_frame.hpp"
#include "bf/wx_facilities.hpp"
#include "bf/level_file_xml_reader.hpp"

#include <fstream>
#include <wx/filename.h>
#include <wx/toolbar.h>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param layout The windows of the program.
 * \param lvl The level.
 * \param level_file The path of the file from which \a lvl was read.
 * The level will be deleted in the destructor.
 */
bf::ingame_view_frame::ingame_view_frame
( windows_layout& layout, level* lvl, const wxString& level_file )
  : wxFrame( &layout.get_main_frame(), wxID_ANY, _("New level") ),
    m_layout(layout), m_level_file(level_file), m_changed(false)
{
  create_controls(layout, lvl);
  m_layout.add_level_view(*this);
  set_changed(false);
} // ingame_view_frame::ingame_view_frame()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the value/range of the scrollbars.
 */
void bf::ingame_view_frame::adjust_scrollbars()
{
  if ( !m_ingame_view->empty() )
    {
      m_h_scrollbar->SetScrollbar
        ( m_ingame_view->get_view_position().x, m_ingame_view->GetSize().x,
          m_ingame_view->get_active_layer().get_width(),
	  (3 * m_ingame_view->GetSize().x) / 4, true );
      m_v_scrollbar->SetScrollbar
        ( m_ingame_view->get_view_position().y, m_ingame_view->GetSize().y,
          m_ingame_view->get_active_layer().get_height(), 
	  (3 * m_ingame_view->GetSize().y) / 4, true );
    }
} // ingame_view_frame::adjust_scrollbars()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the level is changed or not.
 * \param b Tell if the level is changed or not.
 */
void bf::ingame_view_frame::set_changed( bool b )
{
  if ( !b )
    {
      if ( m_level_file.empty() )
        SetTitle( _("New level") );
      else
        SetTitle( wxFileName(m_level_file).GetName() );
    }
  else if ( !m_changed )
    SetTitle( GetTitle() + wxT("*") );

  m_changed = b;
} // ingame_view_frame::set_changed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the level has changes that are not saved.
 */
bool bf::ingame_view_frame::is_changed() const
{
  return m_changed;
} // ingame_view_frame::is_changed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the level.
 */
bool bf::ingame_view_frame::save()
{
  bool result = false;

  if ( m_level_file.empty() )
    result = save_as();
  else
    result = effective_save();

  return result;
} // ingame_view_frame::save()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the level and ask the file to the user.
 */
bool bf::ingame_view_frame::save_as()
{
  bool result = false;
  wxFileDialog dlg( this, _T("Choose a file"), _(""), m_level_file,
                    _T("Level files (*.lvl)|*.lvl"),
                    wxFD_SAVE | wxFD_OVERWRITE_PROMPT );

  if ( dlg.ShowModal() == wxID_OK )
    {
      m_level_file = dlg.GetPath();
      result = effective_save();
    }

  return result;
} // ingame_view_frame::save_as()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the ingame view.
 */
bf::ingame_view* bf::ingame_view_frame::get_ingame_view()
{
  return m_ingame_view;
} // ingame_view_frame::get_ingame_view()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the ingame view.
 */
const bf::ingame_view* bf::ingame_view_frame::get_ingame_view() const
{
  return m_ingame_view;
} // ingame_view_frame::get_ingame_view()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the horizontal scrollbar.
 */
wxScrollBar* bf::ingame_view_frame::get_h_scrollbar()
{
  return m_h_scrollbar;
} // ingame_view_frame::get_h_scrollbar()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the vertical scrollbar.
 */
wxScrollBar* bf::ingame_view_frame::get_v_scrollbar()
{
  return m_v_scrollbar;
} // ingame_view_frame::get_v_scrollbar()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the level without asking anything.
 */
bool bf::ingame_view_frame::effective_save()
{
  bool result = false;

  std::string std_path( wx_to_std_string(m_level_file) );
  std::ofstream f( std_path.c_str() );

  if (f)
    {
      m_ingame_view->save(f);
      result = true;
      m_changed = false;
      SetTitle( wxFileName(m_level_file).GetName() );
    }
  else
    {
      wxMessageDialog dlg
        ( this, _("Error"), _("Can't open the level file."), wxOK );

      dlg.ShowModal();
    }

  return result;
} // ingame_view_frame::effective_save()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls of the window.
 * \param layout The windows of the program.
 * \param lvl The level.
 */
void
bf::ingame_view_frame::create_controls( windows_layout& layout, level* lvl )
{
  create_member_controls(layout, lvl);
  create_sizer_controls();
  create_accelerators();
  create_menu();

  const int bar_sizes[5] = { -1, 80, 80, 100, 100 };

  wxStatusBar* bar = new wxStatusBar(this, wxID_ANY);
  bar->SetFieldsCount(5, bar_sizes);
  SetStatusBar(bar);
} // ingame_view_frame::create_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls for which we keep a reference.
 * \param layout The windows of the program.
 * \param lvl The level.
 */
void bf::ingame_view_frame::create_member_controls
( windows_layout& layout, level* lvl )
{
  m_ingame_view = new ingame_view(*this, lvl, layout);
  m_h_scrollbar = new wxScrollBar(this, wxID_ANY);
  m_v_scrollbar = new wxScrollBar
    ( this, wxID_ANY, wxDefaultPosition, wxDefaultSize, wxSB_VERTICAL );
} // ingame_view_frame::create_member_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls and add them in sizers.
 */
void bf::ingame_view_frame::create_sizer_controls()
{
  wxBoxSizer* sizer = new wxBoxSizer(wxVERTICAL);
	sizer->Add(m_v_scrollbar, 1, wxEXPAND);

  wxBoxSizer* s_sizer = new wxBoxSizer(wxHORIZONTAL);
  s_sizer->Add(m_ingame_view, 1, wxEXPAND);
  s_sizer->Add(sizer, 0, wxEXPAND);

  sizer = new wxBoxSizer(wxVERTICAL);
  sizer->Add(s_sizer, 1, wxEXPAND);

  s_sizer = new wxBoxSizer(wxHORIZONTAL);
	s_sizer->Add(m_h_scrollbar, 1, wxALIGN_BOTTOM);
  s_sizer->Add
    ( new wxWindow
      ( this, wxID_ANY, wxDefaultPosition,
        wxSize( m_v_scrollbar->GetSize().x, m_h_scrollbar->GetSize().y ) ),
      0
      );

  sizer->Add(s_sizer, 0, wxEXPAND);

  SetSizer(sizer);
  SetAutoLayout(true);
} // ingame_view_frame::create_sizer_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the popup menu.
 */
void bf::ingame_view_frame::create_menu()
{
  wxMenuBar* bar = new wxMenuBar();

  m_popup_menu.Append
    ( wxID_ANY, _("&Level"), create_level_menu(), _("Options of the level.") );
  bar->Append( create_level_menu(), _("&Level") );

  m_popup_menu.Append
    ( wxID_ANY, _("&Edit"), create_edit_menu(), _("Edit options.") );
  bar->Append( create_edit_menu(), _("&Edit") );

  m_popup_menu.Append
    ( wxID_ANY, _("&View"), create_view_menu(), _("Options of the view.") );
  bar->Append( create_view_menu(), _("&View") );

  SetMenuBar(bar);
} // ingame_view_frame::create_menu()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the level menu.
 */
wxMenu* bf::ingame_view_frame::create_level_menu() const
{
  wxMenu* result = new wxMenu();

  result->Append( wxID_SAVE, _("&Save"), _("Save the current level.") );
  result->Append
    ( wxID_SAVEAS, _("Save &as...\tCTRL+SHIFT+S"),
      _("Save the current level under a new name.") );
  result->Append
    ( wxID_REVERT_TO_SAVED, _("&Revert to saved"),
      _("Re-open the level file.") );
  result->AppendSeparator();
  result->Append
    ( IDM_CHECK_LEVEL, _("Check the &validty of the level\tF8"), 
      _("Check validity of the level."));
  result->Append
    ( IDM_COMPILE_LEVEL, _("C&ompile the level\tF9"), _("Compile the level."));
  result->AppendSeparator();
  result->Append
    ( wxID_PROPERTIES, _("&Properties"), _("Properties of the level.") );
  result->AppendSeparator();
  result->Append( wxID_CLOSE, _("&Close"), _("Close the level.") );

  return result;
} // ingame_view_frame::create_level_menu()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the edit menu.
 */
wxMenu* bf::ingame_view_frame::create_edit_menu() const
{
  wxMenu* result = new wxMenu();

  result->Append
    ( wxID_UNDO, _("&Undo\tCTRL+Z"), _("Undo the last operation.") );
  result->Append
    ( wxID_REDO, _("&Redo\tCTRL+Y"),_("Redo the last undone operation.") );
  result->AppendSeparator();

  result->Append
    ( wxID_CUT, _("Cu&t"), _("Move the selection to the clipboard.") );
  result->Append
    ( wxID_COPY, _("&Copy"), _("Copy the selection to the clipboard.") );
  result->Append
    ( wxID_PASTE, _("Paste"), _("Paste data from the clipboard.") );
  result->Append
    ( wxID_DELETE, _("&Delete\tDEL"), _("Delete the selection.") );
  result->AppendSeparator();

  result->Append
    ( wxID_SELECTALL, _("&Select all\tCTRL+A"),
      _("Select all items in the current layer.") );
  result->Append
    ( IDM_SELECT_NONE, _("&Select none\tCTRL+SHIFT+A"),
      _("Clear the selection.") );

  return result;
} // ingame_view_frame::create_edit_menu()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the view menu.
 */
wxMenu* bf::ingame_view_frame::create_view_menu() const
{
  wxMenu* result = new wxMenu();

  wxMenu* sub_menu = new wxMenu();
  sub_menu->AppendCheckItem
    ( IDM_GRID_VISIBLE, _("&Visible\tCtrl+G"),
      _("Toggle the visibility of the grid.") );
  sub_menu->AppendCheckItem
    ( IDM_GRID_MAGNETISM, _("&Magnetism\tCtrl+M"),
      _("Toggle the magnetism of the grid.") );
  sub_menu->AppendSeparator();
  sub_menu->Append
    ( IDM_SHOW_GRID_PROPERTIES, _("&Configure..."),
      _("Configuration of the grid.") );

  result->Append
    ( wxID_ANY, _("Gri&d"), sub_menu, _("Options of the grid.") );

  sub_menu = new wxMenu();
  sub_menu->AppendCheckItem
    ( wxID_ANY, _("Show"),
      _("Toggle the visibility of the margin.") );
  sub_menu->AppendSeparator();
  sub_menu->Append
    ( wxID_ANY, _("&Configure..."),
      _("Edit the size of the margin.") );

  result->Append
    ( wxID_ANY, _("&Layer margin"), sub_menu,
      _("Options of the layer margins.") );
  result->AppendCheckItem
    ( wxID_ANY, _("&Wireframe\tW"), _("Display items in wireframe mode.") );
  result->AppendCheckItem
    ( wxID_ANY, _("&Graphic\tG"), _("Display the sprites of the items.") );

  return result;
} // ingame_view_frame::create_view_menu()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the accelerators for the actions in the menus.
 */
void bf::ingame_view_frame::create_accelerators()
{
  wxAcceleratorEntry entries[] =
    {
      wxAcceleratorEntry( wxACCEL_CTRL, (int)'A', wxID_SELECTALL ),
      wxAcceleratorEntry
      ( wxACCEL_CTRL | wxACCEL_SHIFT, (int)'A', IDM_SELECT_NONE ),
      wxAcceleratorEntry( wxACCEL_CTRL, (int)'C', wxID_COPY ),
      wxAcceleratorEntry( wxACCEL_CTRL, (int)'G', IDM_GRID_VISIBLE ),
      wxAcceleratorEntry( wxACCEL_CTRL, (int)'M', IDM_GRID_MAGNETISM ),
      wxAcceleratorEntry( wxACCEL_CTRL, (int)'S', wxID_SAVE ),
      wxAcceleratorEntry( wxACCEL_CTRL | wxACCEL_SHIFT, (int)'S', wxID_SAVEAS ),
      wxAcceleratorEntry( wxACCEL_CTRL, (int)'V', wxID_PASTE ),
      wxAcceleratorEntry( wxACCEL_CTRL, (int)'X', wxID_CUT ),
      wxAcceleratorEntry( wxACCEL_CTRL, (int)'Y', wxID_REDO ),
      //wxAcceleratorEntry( wxACCEL_CTRL, (int)'Z', wxID_UNDO ),
      wxAcceleratorEntry( wxACCEL_NORMAL, WXK_DELETE, wxID_DELETE ),
    };

  wxAcceleratorTable accel
    ( sizeof(entries) / sizeof(wxAcceleratorEntry), entries );
  SetAcceleratorTable(accel);
} // ingame_view_frame::create_accelerators()

/*----------------------------------------------------------------------------*/
/**
 * \brief Toggle the visible status of the grid.
 */
void bf::ingame_view_frame::toggle_grid_visibility()
{
  m_ingame_view->show_grid( !m_ingame_view->grid_visible() );
  m_ingame_view->Refresh();
} // ingame_view_frame::toggle_grid_visibility()

/*----------------------------------------------------------------------------*/
/**
 * \brief Toggle the magnetism status of the grid.
 */
void bf::ingame_view_frame::toggle_grid_magnetism()
{
  m_ingame_view->get_grid().set_magnetism_active
    ( !m_ingame_view->get_grid().get_magnetism_active() );
} // ingame_view_frame::toggle_grid_magnetism()

/*----------------------------------------------------------------------------*/
/**
 * \brief Show the grid configuration windows.
 */
void bf::ingame_view_frame::show_grid_configuration()
{
  grid_properties_frame dlg( this, m_ingame_view->get_grid() );

  if ( dlg.ShowModal() == wxID_OK )
    {
      m_ingame_view->set_grid( dlg.get_grid() );
      m_ingame_view->Refresh();
    }
} // ingame_view_frame::show_grid_configuration()

/*----------------------------------------------------------------------------*/
/**
 * \brief Revert to the last save state.
 */
void bf::ingame_view_frame::revert_save()
{
  if ( ! m_ingame_view->revert_save() )
    {
      level_file_xml_reader reader;
      try
	{
	  level* lvl = reader.load(m_level_file);
	  m_ingame_view->put_level(lvl);
	  set_changed(false);
	}
	catch( std::exception& e )
	{
	  claw::logger << claw::log_error << e.what() << claw::lendl;
	}
    }
} // ingame_view_frame::revert_save()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check the validity of current level.
 * \return true if the level is ok.
 */
bool bf::ingame_view_frame::check_level()
{
  bool result = true;
  std::string error_msg;
  unsigned int layer_error = 0;
  item_instance* item = NULL;

  item = m_ingame_view->get_level().check(layer_error, error_msg);

  if ( item != NULL )
    {
      wxMessageDialog dlg
        ( this, std_to_wx_string(error_msg), _("Bad item"), wxICON_ERROR );

      dlg.ShowModal();
      m_ingame_view->select_item_and_layer(item,layer_error);
      result = false;
    }

  return result;
} // ingame_view_frame::check_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check the validity of current level.
 * \return true if the level is ok.
 */
void bf::ingame_view_frame::check_level_verbose()
{
  if ( check_level() )
    GetStatusBar()->SetStatusText( _("Check completed."), 0 );
} // ingame_view_frame::check_level_verbose()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile the level.
 */
void bf::ingame_view_frame::compile_level()
{
  Disable();
  Refresh();

  if ( check_level() )
    {
      bool do_it = true;

      if ( m_level_file.IsEmpty() )
        do_it = save_as();
      
      if ( do_it )
        compile_level_no_check();
    }

  Enable();
  GetStatusBar()->SetStatusText( _("Compilation done."), 0 );
} // ingame_view_frame::compile_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile the level without checking anything.
 */
void bf::ingame_view_frame::compile_level_no_check()
{
  std::string std_path( wx_to_std_string(m_level_file) );
  std::size_t pos = std_path.rfind(".lvl");

  if ( pos != std::string::npos )
    std_path = std_path.substr(0, pos);

  std_path += ".cl";

  std::ofstream f( std_path.c_str() );

  if (f)
    {
      compiled_file cf(f);
      m_ingame_view->get_level().compile(cf);
    }
  else
    {
      wxMessageDialog dlg
        ( this, _("Error"), _("Can't open the level file."), wxOK );

      dlg.ShowModal();
    }
} // ingame_view_frame::effective_save()

/*----------------------------------------------------------------------------*/
/**
 * \brief Event sent to a resized window.
 * \param event The event.
 */
void bf::ingame_view_frame::on_size(wxSizeEvent& event)
{
  adjust_scrollbars();
  m_ingame_view->set_view_position
    ( m_h_scrollbar->GetThumbPosition(), m_v_scrollbar->GetThumbPosition() );
  event.Skip();
} // ingame_view_frame::on_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw the content of the window.
 * \param event The paint event.
 */
void bf::ingame_view_frame::on_paint(wxPaintEvent& event)
{
  m_ingame_view->render();
} // ingame_view_frame::on_paint()

/*----------------------------------------------------------------------------*/
/**
 * \brief Fill the windows with the properties of this level.
 * \param event The paint event.
 */
void bf::ingame_view_frame::on_activate(wxActivateEvent& event)
{
  if ( m_layout.set_current_level_view(*this) )
    {
      m_ingame_view->SetFocus();
      m_ingame_view->update_layout();
    }
} // ingame_view_frame::on_activate()

/*----------------------------------------------------------------------------*/
/**
 * \brief Change the position of the view.
 * \param event The scroll event.
 */
void bf::ingame_view_frame::on_scroll(wxScrollEvent& event)
{
  m_ingame_view->set_view_position
    ( m_h_scrollbar->GetThumbPosition(), m_v_scrollbar->GetThumbPosition() );
} // ingame_view_frame::on_activate()

/*----------------------------------------------------------------------------*/
/**
 * \brief Procedure called when closing the window.
 * \param event This event occured.
 */
void bf::ingame_view_frame::on_close(wxCloseEvent& event)
{
  bool ok = true;

  if ( m_changed )
    {
      wxMessageDialog dlg
        ( this,
          _T("This level has changes that are not saved. Save now?"),
          _T("Level is not saved."), wxYES_NO );
      
      if ( dlg.ShowModal() == wxID_YES )
        ok = save();
    }

  if ( ok || !event.CanVeto() )
    {
      m_layout.remove_level_view(*this);
      m_layout.get_properties_frame().set_item(NULL);
      m_layout.get_layer_list_frame().set_level_view(NULL);

      Destroy();
    }
  else
    event.Veto();
} // layer_list_frame::on_close()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user wants the context menu.
 * \param event The event.
 */
void bf::ingame_view_frame::on_context_menu(wxContextMenuEvent& event)
{
  update_menu( m_popup_menu );
  update_menu( *GetMenuBar() );

  if ( event.GetPosition() == wxDefaultPosition )
    PopupMenu( &m_popup_menu, event.GetPosition() );
  else // event.GetPosition() is the mouse position on the screen
    PopupMenu( &m_popup_menu, ScreenToClient(event.GetPosition()) );
} // ingame_view_frame::on_context_menu()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the menu entry "Level/Properties".
 * \param event This event occured.
 */
void bf::ingame_view_frame::on_level_properties(wxCommandEvent& event)
{
  level_properties_frame dlg(this);
  dlg.init_from( m_ingame_view->get_level() );

  if ( dlg.ShowModal() == wxID_OK )
    {
      m_ingame_view->start_change();
      m_ingame_view->get_level().set
        ( dlg.get_width(), dlg.get_height(), dlg.get_camera_width(),
          dlg.get_camera_height(), dlg.get_music() );
    }
} // ingame_view_frame::on_level_properties()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the menu "Undo".
 * \param event This event occured.
 */
void bf::ingame_view_frame::on_undo(wxCommandEvent& event)
{
  m_ingame_view->undo();
} // ingame_view_frame::on_undo()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the menu "Redo".
 * \param event This event occured.
 */
void bf::ingame_view_frame::on_redo(wxCommandEvent& event)
{
  m_ingame_view->redo();
} // ingame_view_frame::on_redo()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the menu "Cut".
 * \param event This event occured.
 */
void bf::ingame_view_frame::on_cut(wxCommandEvent& event)
{
  m_ingame_view->cut_to_clipboard();
} // ingame_view_frame::on_cut()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the menu "Copy".
 * \param event This event occured.
 */
void bf::ingame_view_frame::on_copy(wxCommandEvent& event)
{
  m_ingame_view->copy_to_clipboard();
} // ingame_view_frame::on_copy()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the menu "Paste".
 * \param event This event occured.
 */
void bf::ingame_view_frame::on_paste(wxCommandEvent& event)
{
  m_ingame_view->paste_from_clipboard();
} // ingame_view_frame::on_paste()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the menu "Delete".
 * \param event This event occured.
 */
void bf::ingame_view_frame::on_delete(wxCommandEvent& event)
{
  m_ingame_view->delete_selection();
} // ingame_view_frame::on_delete()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the menu "Select all".
 * \param event This event occured.
 */
void bf::ingame_view_frame::on_select_all(wxCommandEvent& event)
{
  m_ingame_view->select_all();
} // ingame_view_frame::on_select_all()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the menu "Select none".
 * \param event This event occured.
 */
void bf::ingame_view_frame::on_select_none(wxCommandEvent& event)
{
  m_ingame_view->clear_selection();
} // ingame_view_frame::on_select_none()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the menu "Revert save".
 * \param event This event occured.
 */
void bf::ingame_view_frame::on_revert_save(wxCommandEvent& event)
{
  revert_save();
} // ingame_view_frame::on_revert_save()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the menu "Check level".
 * \param event This event occured.
 */
void bf::ingame_view_frame::on_check_level(wxCommandEvent& event)
{
  check_level_verbose();
} // ingame_view_frame::on_check_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the menu "Compile level".
 * \param event This event occured.
 */
void bf::ingame_view_frame::on_compile_level(wxCommandEvent& event)
{
  compile_level();
} // ingame_view_frame::on_compile_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the menu "Close".
 * \param event This event occured.
 */
void bf::ingame_view_frame::on_menu_close(wxCommandEvent& event)
{
  Close();
} // ingame_view_frame::on_menu_close()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the menu "Configure".
 * \param event This event occured.
 */
void bf::ingame_view_frame::on_show_grid_properties(wxCommandEvent& event)
{
  show_grid_configuration();
} // ingame_view_frame::on_show_grid_properties()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the menu "Magnetism".
 * \param event This event occured.
 */
void bf::ingame_view_frame::on_grid_magnetism(wxCommandEvent& event)
{
  toggle_grid_magnetism();
} // ingame_view_frame::on_grid_magnetism()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the menu "Visible".
 * \param event This event occured.
 */
void bf::ingame_view_frame::on_grid_visible(wxCommandEvent& event)
{
  toggle_grid_visibility();
} // ingame_view_frame::on_grid_visible()

/*----------------------------------------------------------------------------*/
/**
 * \brief A menu entry is highlighted.
 * \param event This event occured.
 */
void bf::ingame_view_frame::on_menu_highlight(wxMenuEvent& event)
{
  GetStatusBar()->SetStatusText
    ( m_popup_menu.GetHelpString(event.GetMenuId()), 0 );
} // ingame_view_frame::on_menu_highlight()

/*----------------------------------------------------------------------------*/
/**
 * \brief A menu is opened.
 * \param event The event.
 */
void bf::ingame_view_frame::on_open_menu(wxMenuEvent& event)
{
  update_menu( m_popup_menu );
  update_menu( *GetMenuBar() );
} // ingame_view_frame::on_menu_open()

/*----------------------------------------------------------------------------*/
BEGIN_EVENT_TABLE(bf::ingame_view_frame, wxFrame)
  EVT_SIZE( bf::ingame_view_frame::on_size )
  EVT_PAINT( bf::ingame_view_frame::on_paint )
  EVT_ACTIVATE( bf::ingame_view_frame::on_activate )
  EVT_SCROLL( bf::ingame_view_frame::on_scroll )
  EVT_CLOSE( bf::ingame_view_frame::on_close )
  EVT_CONTEXT_MENU( bf::ingame_view_frame::on_context_menu )
  EVT_MENU( wxID_PROPERTIES, bf::ingame_view_frame::on_level_properties )
  EVT_MENU( wxID_UNDO, bf::ingame_view_frame::on_undo )
  EVT_MENU( wxID_REDO, bf::ingame_view_frame::on_redo )
  EVT_MENU( wxID_CUT, bf::ingame_view_frame::on_cut )
  EVT_MENU( wxID_COPY, bf::ingame_view_frame::on_copy )
  EVT_MENU( wxID_PASTE, bf::ingame_view_frame::on_paste )
  EVT_MENU( wxID_DELETE, bf::ingame_view_frame::on_delete )
  EVT_MENU( wxID_SELECTALL, bf::ingame_view_frame::on_select_all )
  EVT_MENU( IDM_SELECT_NONE, bf::ingame_view_frame::on_select_none )
  EVT_MENU(  wxID_REVERT_TO_SAVED,
	     bf::ingame_view_frame::on_revert_save )
  EVT_MENU(  bf::ingame_view_frame::IDM_CHECK_LEVEL,
	     bf::ingame_view_frame::on_check_level )
  EVT_MENU(  bf::ingame_view_frame::IDM_COMPILE_LEVEL,
	     bf::ingame_view_frame::on_compile_level )
  EVT_MENU( wxID_CLOSE, bf::ingame_view_frame::on_menu_close )
  EVT_MENU( bf::ingame_view_frame::IDM_SHOW_GRID_PROPERTIES,
            bf::ingame_view_frame::on_show_grid_properties )
  EVT_MENU( bf::ingame_view_frame::IDM_GRID_VISIBLE,
            bf::ingame_view_frame::on_grid_visible )
  EVT_MENU( bf::ingame_view_frame::IDM_GRID_MAGNETISM,
            bf::ingame_view_frame::on_grid_magnetism )
  EVT_MENU_HIGHLIGHT_ALL( bf::ingame_view_frame::on_menu_highlight )
  EVT_MENU_OPEN( bf::ingame_view_frame::on_open_menu )
END_EVENT_TABLE()
