/*
    Plee The Bear - Level editor

    Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/animation_edit.cpp
 * \brief Implementation of the bf::animation_edit class.
 * \author Sebastien Angibaud
 */
#include "bf/animation_edit.hpp"
#include "bf/wx_facilities.hpp"

#include <limits>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param parent The window owning this window.
 * \param animation The initial animation.
 */
bf::animation_edit::animation_edit( wxWindow& parent, const animation& anim )
  : wxPanel(&parent, wxID_ANY), base_edit<animation>(anim),
    m_animation(anim)
{
  create_controls();
  value_updated();
  Fit();
} // animation_edit::animation_edit()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check if the displayed value is correct and, if it is, set the
 *        value according to the display.
 */
bool bf::animation_edit::validate()
{
  animation anim;

  anim.set_frames(m_animation.get_frames());
  anim.set_flip_x( m_flip_x_box->IsChecked() );
  anim.set_flip_y( m_flip_y_box->IsChecked() );
  anim.set_loops( m_loops_spin->GetValue() );
  anim.set_loop_back( m_loop_back_box->IsChecked() );
  anim.set_first_index( m_first_index_spin->GetValue());
  anim.set_last_index( m_last_index_spin->GetValue());
  anim.set_alpha( m_alpha_spin->GetValue() );
    
  set_value(anim);

  return true;
} // animation_edit::validate()

/*----------------------------------------------------------------------------*/
/**
 * \brief Method called after changing the value by set_value().
 */
void bf::animation_edit::value_updated()
{
  animation anim = get_value();

  m_flip_x_box->SetValue( anim.get_flip_x() );
  m_flip_y_box->SetValue( anim.get_flip_y() );
  m_loop_back_box->SetValue( anim.get_loop_back() );
  m_loops_spin->SetValue( anim.get_loops() );
  m_alpha_spin->SetValue( anim.get_alpha() );
  m_first_index_spin->SetValue( anim.get_first_index() );
  m_last_index_spin->SetValue( anim.get_last_index() );
  fill_controls();
} // animation_edit::value_updated()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update spin control.
 */
void bf::animation_edit::update_spin_ctrl()
{
  if (m_sprite_list->GetCount() == 0)
    m_first_index_spin->SetRange(0,0);
  else
    m_first_index_spin->SetRange(0, m_sprite_list->GetCount()-1);
  
  m_last_index_spin->SetRange( m_first_index_spin->GetValue(),
                               m_sprite_list->GetCount()-1);  
} // animation_edit::update_spin_ctrl()

/*----------------------------------------------------------------------------*/
/**
 * \brief Fill the controls with the values of the animation.
 */
void bf::animation_edit::fill_controls()
{
  int index = m_sprite_list->GetSelection();
  
  m_sprite_list->Clear();
  m_time_list->Clear();
  
  std::list<animation::frame>::const_iterator it;

  for (it=m_animation.get_frames().begin(); 
       it!=m_animation.get_frames().end(); ++it)
    {
      m_sprite_list->Append
        (human_readable<sprite>::convert(it->get_sprite()) );
      m_time_list->Append
        (human_readable<double>::convert(it->get_duration()) );
    }
     
  m_sprite_list->SetSelection(index);
  m_time_list->SetSelection(index);
  
  update_spin_ctrl();
} // animation_edit::fill_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls of the window.
 */
void bf::animation_edit::create_controls()
{
  m_sprite_list = new wxListBox( this, wxID_ANY );
  m_time_list = new wxListBox( this, wxID_ANY );
  m_flip_x_box = new wxCheckBox( this, wxID_ANY, _("Flip X") );
  m_flip_y_box = new wxCheckBox( this, wxID_ANY, _("Flip Y") );
  m_loop_back_box = new wxCheckBox( this, wxID_ANY, _("Loop_back") );

  m_loops_spin = 
    new spin_ctrl<unsigned int>( this, wxID_ANY,wxDefaultPosition, 
				 wxDefaultSize, wxEXPAND,0,
				 std::numeric_limits<unsigned int>::max(), 1 );
  
  m_first_index_spin = new wxSpinCtrl( this, IDC_BOUND_INDEX_CHANGE ); 
  m_last_index_spin = new wxSpinCtrl( this, IDC_BOUND_INDEX_CHANGE );
  
  m_alpha_spin = 
    new spin_ctrl<double>( this, wxID_ANY,wxDefaultPosition,
			   wxDefaultSize, wxEXPAND,0,1,1,0.01 );

  create_sizer_controls();
  create_connect();
  fill_controls();
} // animation_edit::create_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls and add them in sizers.
 */
void bf::animation_edit::create_sizer_controls()
{
  wxBoxSizer* sizer = new wxBoxSizer( wxVERTICAL );
 
  // the sizer for the buttons
  wxBoxSizer* s_h_sizer = new wxBoxSizer( wxHORIZONTAL );

  // the sizer for the buttons
  wxBoxSizer* s_sizer = new wxBoxSizer( wxVERTICAL );
  
  s_sizer->Add( new wxButton(this, wxID_NEW), 1, wxEXPAND | wxALL, 0 );
  s_sizer->Add( new wxButton(this, wxID_COPY), 1, wxEXPAND | wxALL, 0 );
  s_sizer->Add( new wxButton(this, wxID_EDIT), 1, wxEXPAND | wxALL, 0 );
  s_sizer->Add( new wxButton(this, wxID_DELETE), 1, wxEXPAND | wxALL, 0 );
  s_sizer->Add( new wxButton(this, wxID_UP), 1, wxEXPAND | wxALL, 0 );
  s_sizer->Add( new wxButton(this, wxID_DOWN), 1, wxEXPAND | wxALL, 0 );
  
  s_h_sizer->Add( m_sprite_list, 4, wxEXPAND );
  s_h_sizer->Add( m_time_list, 1, wxEXPAND );
  s_h_sizer->Add( s_sizer, 1, wxEXPAND );
  
  sizer->Add( s_h_sizer, 1, wxEXPAND );

  // options
  sizer->Add( create_loops_sizer(), 0, wxEXPAND );
  sizer->AddSpacer(5);
  sizer->Add( create_options_sizer(), 0, wxEXPAND );

  SetSizer(sizer);
} // animation_edit::create_sizer_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create connection of controls.
 */
void bf::animation_edit::create_connect()
{
  this->Connect( m_sprite_list->GetId(), wxEVT_COMMAND_LISTBOX_SELECTED,
                 wxCommandEventHandler(animation_edit::on_sprite_select) ); 

  this->Connect( m_sprite_list->GetId(), wxEVT_COMMAND_LISTBOX_DOUBLECLICKED,
                 wxCommandEventHandler(animation_edit::on_sprite_edit) );

  this->Connect( m_time_list->GetId(), wxEVT_COMMAND_LISTBOX_SELECTED,
                 wxCommandEventHandler(animation_edit::on_time_select) ); 

  this->Connect( m_time_list->GetId(), wxEVT_COMMAND_LISTBOX_DOUBLECLICKED,
                 wxCommandEventHandler(animation_edit::on_time_edit) );
} // animation_edit::create_connect()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls and the sizer for the options part.
 */
wxSizer* bf::animation_edit::create_options_sizer()
{
  wxBoxSizer* result = new wxBoxSizer( wxHORIZONTAL );

  // flip
  wxStaticBoxSizer* s_sizer = 
    new wxStaticBoxSizer( wxHORIZONTAL, this, _("Appearance"));
  s_sizer->Add( m_flip_x_box, 1, wxEXPAND, 0 );
  s_sizer->Add( m_flip_y_box, 1, wxEXPAND, 0 );
  s_sizer->Add
    ( new wxStaticText(this, wxID_ANY, _("Transparency:")),
      0, wxALIGN_CENTRE_VERTICAL | wxALL, 5 );
  s_sizer->Add( m_alpha_spin, 0, wxALL, 0 );
  result->Add( s_sizer, 1, wxEXPAND, 0 );

  return result;
} // animation_edit::create_options_sizer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls and the sizer for the loops part.
 */
wxSizer* bf::animation_edit::create_loops_sizer()
{
  wxBoxSizer* result = new wxBoxSizer( wxHORIZONTAL );

  wxStaticBoxSizer* s_sizer = 
    new wxStaticBoxSizer( wxVERTICAL, this, _("Loops") );
  
  wxBoxSizer* s_h_sizer = new wxBoxSizer( wxHORIZONTAL );
  
  //number
  wxBoxSizer* s_v_sizer = new wxBoxSizer( wxVERTICAL );
  s_v_sizer->Add
    ( new wxStaticText(this, wxID_ANY, _("Number:")),
      0, wxALIGN_CENTRE_VERTICAL, 0 );
  s_v_sizer->Add( m_loops_spin, 0, wxALIGN_CENTRE_VERTICAL , 0 );
  s_h_sizer->Add( s_v_sizer, 1, wxALIGN_CENTRE_VERTICAL , 0 );  
  
  
  //First index
  s_v_sizer = new wxBoxSizer( wxVERTICAL );
  s_v_sizer->Add
    ( new wxStaticText(this, wxID_ANY, _("First index:")),
      0, wxALIGN_CENTRE_VERTICAL, 0 );
  s_v_sizer->Add( m_first_index_spin, 0, wxALIGN_CENTRE_VERTICAL, 0 );
  s_h_sizer->Add( s_v_sizer, 1, wxALIGN_CENTRE_VERTICAL , 0 ); 

  //Last index
  s_v_sizer = new wxBoxSizer( wxVERTICAL );
  s_v_sizer->Add
    ( new wxStaticText(this, wxID_ANY, _("Last index:")),
      0, wxALIGN_CENTRE_VERTICAL | wxALL, 0 );
  s_v_sizer->Add( m_last_index_spin, 0, wxALIGN_CENTRE_VERTICAL, 0 );
  s_h_sizer->Add( s_v_sizer, 1, wxALIGN_CENTRE_VERTICAL , 0 ); 

  //loop back
  s_h_sizer->Add( m_loop_back_box, 1, wxALIGN_CENTRE_VERTICAL , 0 );
  
  s_sizer->Add( s_h_sizer, 1, wxEXPAND , 0 );  
  
  result->Add( s_sizer, 1, wxEXPAND , 0 );  

  return result;
} // animation_edit::create_loops_sizer()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user clicked on the "Edit" button.
 * \param event The event.
 */
void bf::animation_edit::edit_sprite( unsigned int index )
{
  sprite spr = m_animation[index].get_sprite();
  value_editor_dialog<sprite_edit,sprite> dlg(*m_parent, _("Sprite"), spr);
  dlg.set_value(spr);

  if ( dlg.ShowModal() == wxID_OK )
    {
      m_animation[index].set_sprite(dlg.get_value());
      fill_controls();
    }
} // animation_edit::edit_sprite()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user clicked on the "Edit" button.
 * \param event The event.
 */
void bf::animation_edit::edit_time( unsigned int index )
{
  value_editor_dialog<free_edit<real_type>, real_type> dlg
    (*m_parent, wxT("Duration"), m_animation[index].get_duration());

  if ( dlg.ShowModal() == wxID_OK )
    {
      m_animation[index].set_duration(dlg.get_value().get_value());
      fill_controls();
    }
} // animation_edit::edit_time()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user clicked on the "Up" button.
 * \param event The event.
 */
void
bf::animation_edit::on_up( wxCommandEvent& event )
{
  int index = m_sprite_list->GetSelection();

  if ( index != wxNOT_FOUND )
    if ( index > 0 )
      {
        std::list<animation::frame>::iterator prec;
        prec = m_animation.get_frames().begin();

        std::advance(prec, index-1);

        std::list<animation::frame>::iterator it(prec);
        ++it;

        std::swap(*it, *prec);
        m_sprite_list->SetSelection(index-1);

        fill_controls();
      }
} // animation_edit::on_up()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user clicked on the "Down" button.
 * \param event The event.
 */
void bf::animation_edit::on_down( wxCommandEvent& event )
{
  int index = m_sprite_list->GetSelection();

  if ( index != wxNOT_FOUND )
    if ( (unsigned int)index + 1 < m_sprite_list->GetCount() )
      {
        std::list<animation::frame>::iterator it;
        it = m_animation.get_frames().begin();

        std::advance(it, index);

        std::list<animation::frame>::iterator succ(it);
        ++succ;

        std::swap(*it, *succ);
        m_sprite_list->SetSelection(index+1);

        fill_controls();
      }
} // animation_edit::on_down()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user clicked on the "New" button.
 * \param event The event.
 */
void
bf::animation_edit::on_new( wxCommandEvent& event )
{
  sprite spr;
  value_editor_dialog<sprite_edit,sprite> dlg(*m_parent, _("Sprite"), spr);
    
  if (dlg.ShowModal() == wxID_OK )
    {
      animation::frame f;
      f.set_sprite(dlg.get_value());
      m_animation.get_frames().push_back(f);
      fill_controls();
    }
} // animation_edit::on_new()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user clicked on the "Copy" button.
 * \param event The event.
 */
void bf::animation_edit::on_copy( wxCommandEvent& event )
{
  int index = m_sprite_list->GetSelection();

  if ( index != wxNOT_FOUND )
    {
      m_animation.get_frames().push_back(m_animation[index]);
      fill_controls();
    }
} // animation_edit::on_copy()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user clicked on the "Edit" button.
 * \param event The event.
 */
void bf::animation_edit::on_sprite_edit( wxCommandEvent& event )
{
  int index = m_sprite_list->GetSelection();

  if ( index != wxNOT_FOUND )
    edit_sprite(index);
} // animation_edit::on_sprite_edit()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user clicked on the "Edit" button.
 * \param event The event.
 */
void bf::animation_edit::on_time_edit( wxCommandEvent& event )
{
  int index = m_time_list->GetSelection();

  if ( index != wxNOT_FOUND )
    edit_time(index);
} // animation_edit::on_time_edit()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user select a sprite in the list.
 * \param event The event.
 */
void bf::animation_edit::on_sprite_select( wxCommandEvent& event )
{
  int index = m_sprite_list->GetSelection();
  m_time_list->SetSelection(index);
  fill_controls();
} // animation_edit::on_sprite_select()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user select a time in the list.
 * \param event The event.
 */
void bf::animation_edit::on_time_select( wxCommandEvent& event )
{
  int index = m_time_list->GetSelection();
  m_sprite_list->SetSelection(index);
  fill_controls();
} // animation_edit::on_time_select()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user clicked on the "Delete" button.
 * \param event The event.
 */
void bf::animation_edit::on_delete( wxCommandEvent& event )
{
  int index = m_sprite_list->GetSelection();

  if ( index != wxNOT_FOUND )
    {
      std::list<animation::frame>::iterator it;
      it = m_animation.get_frames().begin();

      std::advance(it, index);
      m_animation.get_frames().erase(it);

      if ( !m_animation.get_frames().empty() )
        if ( (unsigned int)index == m_animation.get_frames().size() )
          m_sprite_list->SetSelection(index-1);

      fill_controls();
    }
} // animation_edit::on_delete()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user change the first or the last index.
 * \param event The event.
 */
void bf::animation_edit::on_bound_index_change( wxSpinEvent& event )
{
  update_spin_ctrl();
} // animation_edit::on_bound_index_change()

/*----------------------------------------------------------------------------*/
BEGIN_EVENT_TABLE(bf::animation_edit, wxPanel)
  EVT_BUTTON( wxID_UP, bf::animation_edit::on_up )
  EVT_BUTTON( wxID_DOWN, bf::animation_edit::on_down )
  EVT_BUTTON( wxID_NEW, bf::animation_edit::on_new )
  EVT_BUTTON( wxID_COPY, bf::animation_edit::on_copy )
  EVT_BUTTON( wxID_EDIT, bf::animation_edit::on_sprite_edit )
  EVT_BUTTON( wxID_DELETE, bf::animation_edit::on_delete )
  EVT_SPINCTRL( bf::animation_edit::IDC_BOUND_INDEX_CHANGE,
                bf::animation_edit::on_bound_index_change )
END_EVENT_TABLE()

