/*
    Plee The Bear - Level compiler

    Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file animation.cpp
 * \brief Implementation of the bf::animation class.
 * \author Julien Jorge
 */
#include "bf/animation.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Default constructor.
 */
bf::animation::frame::frame()
  : m_sprite(), m_duration(1)
{

} // animation::frame::frame()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the sprite.
 * \param spr The new sprite.
 */
void bf::animation::frame::set_sprite(const sprite& spr)
{
  m_sprite = spr;
} // animation::frame::set_sprite()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the duration.
 * \param time The new duration.
 */
void bf::animation::frame::set_duration(double duration)
{
  m_duration = duration;
} // animation::frame::set_duration()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the sprite.
 * \param spr The new sprite.
 */
const bf::sprite& bf::animation::frame::get_sprite() const
{
  return m_sprite;
} // animation::frame::get_sprite()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the duration.
 * \param spr The new duration.
 */
double bf::animation::frame::get_duration() const
{
  return m_duration;
} // animation::frame::get_duration()




/*----------------------------------------------------------------------------*/
/**
 * \brief Default constructor.
 */
bf::animation::animation()
  : m_flip_x(false), m_flip_y(false), m_alpha(1), 
    m_loops(0), m_loop_back(false),
    m_first_index(0), m_last_index(0)
{

} // animation::animation()

/*----------------------------------------------------------------------------*/
/**
 * \brief Default constructor by copy.
 */
bf::animation::animation(const animation& anim)
  : m_flip_x(anim.get_flip_x()), m_flip_y(anim.get_flip_y()), 
    m_alpha(anim.get_alpha()), 
    m_loops(anim.get_loops()), m_loop_back(anim.get_loop_back()),
    m_first_index(anim.get_first_index()), m_last_index(anim.get_last_index())
{
  set_frames(anim.get_frames());
} // animation::animation()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the frames of the animation.
 * \param frames The new frames.
 */
void
bf::animation::set_frames( const std::list<frame>& frames )
{
  std::list<frame>::const_iterator it;

  for ( it = frames.begin(); it != frames.end(); it ++ )
    m_frames.push_back( *it );
} // animation::set_frames()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the horizontal flip attribute of the image.
 * \param b The new value.
 */
void bf::animation::set_flip_x( bool b )
{
  m_flip_x = b;
} // animation::set_flip_x()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the vertical flip attribute of the image.
 * \param b The new value.
 */
void bf::animation::set_flip_y( bool b )
{
  m_flip_y = b;
} // animation::set_flip_y()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the alpha transparency.
 * \param alpha The new value.
 */
void bf::animation::set_alpha( double alpha )
{
  m_alpha = alpha;
} // animation::set_alpha()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the number of times the animation will be played.
 * \param loops The new value.
 */
void bf::animation::set_loops( unsigned int loops )
{
  m_loops = loops;
} // animation::set_loops()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the "play reverse at end" attribute of the animation.
 * \param b The new value.
 */
void bf::animation::set_loop_back( bool b )
{
  m_loop_back = b;
} // animation::set_loop_back()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the index of the first frame of the loops.
 * \param loops The new value.
 */
void bf::animation::set_first_index( unsigned int index )
{
  m_first_index = index;
} // animation::set_first_index()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the index of the last frame of the loops.
 * \param loops The new value.
 */
void bf::animation::set_last_index( unsigned int index )
{
  m_last_index = index;
} // animation::set_last_index()


/*----------------------------------------------------------------------------*/
/**
 * \brief Return the flip_x value.
 */
bool bf::animation::get_flip_x() const
{
  return m_flip_x;
} // animation::get_flip_x()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the flip_y value.
 */
bool bf::animation::get_flip_y() const
{
  return m_flip_y;
} // animation::get_flip_y()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the alpha value.
 */
double bf::animation::get_alpha() const
{
  return m_alpha;
} // animation::get_alpha()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the loops value.
 */
unsigned int bf::animation::get_loops() const
{
  return m_loops;
} // animation::get_loops()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the loop_back value.
 */
bool bf::animation::get_loop_back() const
{
  return m_loop_back;
} // animation::get_loop_back()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the index of the first frame for loops.
 */
unsigned int bf::animation::get_first_index() const
{
  return m_first_index;
} // animation::get_first_index()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the index of the last frame for loops.
 */
unsigned int bf::animation::get_last_index() const
{
  return m_last_index;
} // animation::get_last_index()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the frame.
 */
const std::list<bf::animation::frame>& bf::animation::get_frames() const
{
  return m_frames;
} // animation::get_frames()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the frame.
 */
std::list<bf::animation::frame>& bf::animation::get_frames()
{
  return m_frames;
} // animation::get_frames()

/*----------------------------------------------------------------------------*/
/**
 * \brief Indicates if the list of frames is empty.
 */
bool bf::animation::empty() const
{
  return m_frames.empty();
} // animation::empty()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the number of frames.
 */
unsigned int bf::animation::nb_frames() const
{
  return m_frames.size();
} // animation::nb_frames()

/*----------------------------------------------------------------------------*/
/**
 * \brief Access to a frame by its index.
 * \param index The index of the frame.
 */
bf::animation::frame& bf::animation::operator[](int index)
{
  std::list<frame>::iterator it = m_frames.begin();
  std::advance(it, index);

  return *it;
} // animation::operator[]()

/*----------------------------------------------------------------------------*/
/**
 * \brief Access to a frame by its index.
 * \param index The index of the frame.
 */
const bf::animation::frame& bf::animation::operator[](int index) const
{
  std::list<frame>::const_iterator it = m_frames.begin();
  std::advance(it, index);

  return *it;
} // animation::operator[]()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile the animation.
 * \param f The stream in which we write the compiled animation.
 */
void bf::animation::compile( compiled_file& f ) const
{
  std::list<frame>::const_iterator it;

  f << m_frames.size();

  for ( it=m_frames.begin(); it!=m_frames.end(); ++it)
    {
      f << it->get_duration();
      it->get_sprite().compile(f);
    }

  f << m_flip_x << m_flip_y << m_alpha << m_loops << m_loop_back
    << m_first_index << m_last_index;
} // animation::compile()
