/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file action_layer.cpp
 * \brief Implementation of the bear::action_layer class.
 * \author Julien Jorge
 */
#include "generic_items/layer/action_layer.hpp"

#include "engine/collision_event/collision_event_stop_player.hpp"
#include "engine/collision_event/collision_event_kill_player.hpp"
#include "engine/export.hpp"
#include "engine/game.hpp"

#include "universe/alignment/align_left.hpp"
#include "universe/alignment/align_right.hpp"
#include "universe/alignment/align_bottom.hpp"
#include "universe/alignment/align_top_left.hpp"
#include "universe/alignment/align_top_right.hpp"
#include "universe/alignment/align_bottom_left.hpp"
#include "universe/alignment/align_bottom_right.hpp"

LAYER_EXPORT( action_layer, bear )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param size The size of the layer/world.
 */
bear::action_layer::action_layer( const universe::size_box_type& size )
  : layer(size), m_world(size)
{
  create_margins();
} // action_layer::action_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Call the "start" method of all items.
 */
void bear::action_layer::start()
{
  m_world.start();
} // action_layer::start()

/*----------------------------------------------------------------------------*/
/**
 * \brief Progress all items in the active area.
 * \param active_area The area containing active items.
 * \param elapsed_time Elapsed time since the last call.
 */
void bear::action_layer::progress
( const region_type& active_area, universe::time_type elapsed_time )
{
  m_world.progress_entities( active_area, elapsed_time );
} // action_layer::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the sprites of the items in the visible area.
 * \param visuals (out) The sprites in the visible area, and their positions.
 * \param visible_area The visible part of the layer.
 */
void bear::action_layer::get_visual
( std::list<engine::scene_visual>& visuals,
  const universe::rectangle_type& visible_area ) const
{
  m_world.get_visual( visuals, visible_area );
} // action_layer::get_visual()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add an item in the layer.
 * \param that The item to add.
 */
void bear::action_layer::add_item( engine::base_item& that )
{
  if ( that.is_fixed() )
    static_item( that );
  else
    mobile_item( that );
} // action_layer::add_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the margins around the layer.
 */
void bear::action_layer::create_margins()
{
  const claw::math::coordinate_2d<unsigned int> screen_size
    ( engine::game::get_instance().get_screen_size() );

  create_corner_margins();
  create_left_margin( screen_size );
  create_right_margin( screen_size );
  create_top_margin(  screen_size );
  create_bottom_margin( screen_size );
} // action_layer::create_margins()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the margins on the corners of the level.
 */
void bear::action_layer::create_corner_margins()
{
  const unsigned int margin_size = engine::level::s_level_margin;

  new_margin_block
    < engine::collision_event_stop_player<universe::align_top_left> >
    ( 0, 0, margin_size, margin_size );

  new_margin_block
    < engine::collision_event_stop_player<universe::align_top_right> >
    ( get_size().x - margin_size, 0, margin_size, margin_size );

  new_margin_block
    < engine::collision_event_stop_player<universe::align_bottom_left> >
    ( 0, get_size().y - margin_size, margin_size, margin_size );

  new_margin_block
    < engine::collision_event_stop_player<universe::align_bottom_right> >
    ( get_size().x - margin_size, get_size().y - margin_size, margin_size,
      margin_size );
} // action_layer::create_corner_margins()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the margins on the left of the level.
 * \param screen_size The size of the screen.
 */
void bear::action_layer::create_left_margin
( const claw::math::coordinate_2d<unsigned int>& screen_size )
{
  create_vertical_margin
    < engine::collision_event_stop_player<universe::align_right> >
    ( screen_size.y, 0 );
} // action_layer::create_left_margin()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the margins on the right of the level.
 * \param screen_size The size of the screen.
 */
void bear::action_layer::create_right_margin
( const claw::math::coordinate_2d<unsigned int>& screen_size )
{
  create_vertical_margin
    < engine::collision_event_stop_player<universe::align_left> >
    ( screen_size.y, get_size().x - engine::level::s_level_margin );
} // action_layer::create_right_margin()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the margins on the top the level.
 * \param screen_size The size of the screen.
 */
void bear::action_layer::create_top_margin
( const claw::math::coordinate_2d<unsigned int>& screen_size )
{
  create_horizontal_margin
    < engine::collision_event_stop_player<universe::align_bottom> >
    ( screen_size.x, 0 );
} // action_layer::create_top_margin()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the margins on the bottom of the level.
 * \param screen_size The size of the screen.
 */
void bear::action_layer::create_bottom_margin
( const claw::math::coordinate_2d<unsigned int>& screen_size )
{
  create_horizontal_margin
    < engine::collision_event_kill_player>
    ( screen_size.x, get_size().y - engine::level::s_level_margin );
} // action_layer::create_bottom_margin()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a static item in the layer.
 * \param that The item to add.
 */
void bear::action_layer::static_item( engine::base_item& that )
{
  CLAW_PRECOND( that.is_valid() );
  CLAW_PRECOND( that.get_mass() == 0 );

  m_world.add_static( &that );
} // action_layer::static_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add an entity in the layer.
 * \param that The entity to add.
 */
void bear::action_layer::mobile_item( engine::base_item& that )
{
  CLAW_PRECOND( that.is_valid() );
  m_world.register_item( &that );
} // action_layer::mobile_item()
