/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file platform_with_two_elastics.cpp
 * \brief Implementation of the bear::platform_with_two_elastics class.
 * \author Julien Jorge
 */
#include "generic_items/platform_with_two_elastics.hpp"

#include <claw/assert.hpp>
#include "engine/world.hpp"
#include "universe/link/elastic_link.hpp"

#include "engine/export.hpp"

BASE_ITEM_EXPORT( platform_with_two_elastics, bear )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::platform_with_two_elastics::platform_with_two_elastics()
  : m_link_sprite(NULL), m_link_length(0), m_link_strength(0),
    m_platform_sprite(NULL), m_platform(NULL), m_link_steps(0)
{

} // platform_with_two_elastics::platform_with_two_elastics()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destrutor.
 */
bear::platform_with_two_elastics::~platform_with_two_elastics()
{
  if (m_link_sprite)
    delete m_link_sprite;
} // platform_with_two_elastics::~platform_with_two_elastics()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type <real>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::platform_with_two_elastics::set_real_field
( const std::string& name, double value )
{
  bool ok = true;

  if (name == "strength")
    m_link_strength = value;
  else
    ok = super::set_real_field( name, value );

  return ok;
} // platform_with_two_elastics::set_real_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type <unsigned integer>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::platform_with_two_elastics::set_u_integer_field
( const std::string& name, unsigned int value )
{
  bool ok = true;

  if ( name == "length" )
    m_link_length = value;
  else if ( name == "width" )
    set_width( value );
  else
    ok = super::set_u_integer_field( name, value );

  return ok;
} // platform_with_two_elastics::set_u_integer_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type visual::sprite.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::platform_with_two_elastics::set_sprite_field
( const std::string& name, visual::sprite* value )
{
  bool ok = true;

  if (name == "link")
    m_link_sprite = value;
  else if ( name == "platform" )
    m_platform_sprite = value;
  else
    ok = super::set_sprite_field( name, value );

  return ok;
} // platform_with_two_elastics::set_sprite_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is correctly initialized.
 */
bool bear::platform_with_two_elastics::is_valid() const
{
  return m_platform_sprite && m_link_strength && super::is_valid();
} // platform_with_two_elastics::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the item.
 */
void bear::platform_with_two_elastics::start()
{
  engine::world* w = get_owner();
  m_platform = new falling_block;

  m_platform->set_string_field("block_type", "center_center_wall");
  m_platform->set_sprite_field("sprite", m_platform_sprite);
  m_platform_sprite = NULL;

  set_position();

  CLAW_ASSERT( m_platform->is_valid(),
               "The platform isn't correctly initialized" );

  w->register_item( m_platform );
  
  create_links();

  m_platform->start();
} // platform_with_two_elastics::start()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the sprite representing the item.
 * \param visuals (out) The sprites of the item, and their positions.
 */
void bear::platform_with_two_elastics::get_visual
( std::list<engine::scene_visual>& visuals ) const
{
  if (m_link_sprite)
    {
      render_left_link( visuals );
      render_right_link( visuals );
    }
} // platform_with_two_elastics::get_visual()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the position of the platform.
 */
void bear::platform_with_two_elastics::set_position()
{
  universe::position_type pos;
  universe::coordinate_type delta;
  pos.y = get_position().y;

  if (m_platform->get_size().x > get_size().x)
    {
      delta = m_platform->get_size().x - get_size().x;
      pos.x = get_position().x - delta / 2;
    }
  else
    {
      delta = get_size().x - m_platform->get_size().x;
      pos.x = get_position().x + delta / 2;
    }

  if (m_link_sprite)
    m_link_steps = (unsigned int)(delta / (2 * m_link_sprite->width() ));

  m_platform->set_position(pos);
} // platform_with_two_elastics::set_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the elastic links.
 */
void bear::platform_with_two_elastics::create_links() const
{
  engine::base_item* left_hook = new engine::base_item;
  engine::base_item* right_hook = new engine::base_item;

  engine::world* w = get_owner();

  left_hook->set_center_of_mass( get_position() );
  right_hook->set_center_of_mass
    ( universe::position_type
      ( get_position().x + get_width() - right_hook->get_width(),
        get_position().y ) );

  CLAW_ASSERT( left_hook->is_valid(), "left hook is not valid." );
  CLAW_ASSERT( right_hook->is_valid(), "right hook is not valid." );

  w->add_static( left_hook );
  w->add_static( right_hook );

  if (m_link_length)
    {
      new universe::elastic_link(*left_hook, *m_platform, m_link_strength,
				 m_link_length);
      new universe::elastic_link(*m_platform, *right_hook, m_link_strength,
				 m_link_length);
    }
  else
    {
      new universe::elastic_link(*left_hook, *m_platform, m_link_strength);
      new universe::elastic_link(*m_platform, *right_hook, m_link_strength);
    }
} // platform_with_two_elastics::create_links()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the sprites representing the left link.
 * \param visuals (out) The sprites of the item, and their positions.
 */
void bear::platform_with_two_elastics::render_left_link
( std::list<engine::scene_visual>& visuals ) const
{
  CLAW_PRECOND( m_link_sprite );

  universe::position_type attach, target;

  attach = get_position();
  target.set( m_platform->get_position().x - m_link_sprite->width() / 2,
              m_platform->get_position().y + m_platform->get_size().y / 2 );

  render_link( visuals, attach, target );
} // platform_with_two_elastics::render_left_link()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the sprites representing the right link.
 * \param visuals (out) The sprites of the item, and their positions.
 */
void bear::platform_with_two_elastics::render_right_link
( std::list<engine::scene_visual>& visuals ) const
{
  CLAW_PRECOND( m_link_sprite );

  universe::position_type target;

  target.set( m_platform->get_right() + m_link_sprite->width() / 2,
              m_platform->get_position().y + m_platform->get_size().y / 2 );

  render_link( visuals, get_top_right(), target );
} // platform_with_two_elastics::render_right_link()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the sprites representing a link.
 * \param visuals (out) The sprites of the item, and their positions.
 * \param src Coordinates of the first position.
 * \param target Coordinates of the position to reach.
 */
void bear::platform_with_two_elastics::render_link
( std::list<engine::scene_visual>& visuals, const universe::position_type& src,
  const universe::position_type& target ) const
{
  CLAW_PRECOND( m_link_sprite );

  universe::position_type dir( target - src );
  const universe::position_type initial_pos
    ( src - m_link_sprite->get_size() / 2 );
  claw::math::coordinate_2d<unsigned int> initial_pos_visual;
  
  for (unsigned int i=0; i<m_link_steps; ++i)
    {
      universe::position_type pos = initial_pos + (dir * i) / (m_link_steps-1);
      visuals.push_front( engine::scene_visual( pos, *m_link_sprite ) );
    }

  visuals.push_front
    ( engine::scene_visual( initial_pos + dir, *m_link_sprite ) );
} // platform_with_two_elastics::render_link()

