/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file change_friction_item.cpp
 * \brief Implementation of the ptb::change_friction_item class.
 * \author Sbastien Angibaud
 */
#include "generic_items/change_friction_item.hpp"
#include "generic_items/decorative_item.hpp"

#include "engine/world.hpp"

#include "engine/export.hpp"

BASE_ITEM_EXPORT( change_friction_item, bear )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::change_friction_item::change_friction_item()
  : m_top_friction(1), m_bottom_friction(1),
    m_top_animation(NULL), m_bottom_animation(NULL) 
{
  set_can_move_items(false);
  set_global(true);
  set_phantom(true);
} // change_friction_item::change_friction_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::change_friction_item::~change_friction_item()
{
  if ( m_top_animation )
    delete m_top_animation;
  
  if ( m_bottom_animation )
    delete m_bottom_animation;
} // change_friction_item::change_friction_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c <real>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::change_friction_item::set_real_field
( const std::string& name,double value )
{
  bool result;

  if ( name == "top_friction" )
    m_top_friction = value;
  else if ( name == "bottom_friction" )
      m_bottom_friction = value;
  else
    result = super::set_real_field( name, value );

  return result;
} // change_friction_item::set_u_integer_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c visual::animation.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::change_friction_item::set_animation_field
( const std::string& name, visual::animation* value )
{
  bool ok = true;

  if (name == "top_animation")
    m_top_animation = value;
  else if (name == "bottom_animation")
    m_bottom_animation = value;
  else
    ok = super::set_animation_field(name,value);

  return ok;
} // change_friction_item::set_animation_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief A collision occured, the strongest item has the control.
 * \param that The other item of the collision.
 * \param old_self The state to consider for the current item.
 * \param old_that The state to consider for the item "that".
 */
void bear::change_friction_item::hit
( engine::base_item& that, const physical_item_state& old_self,
  const physical_item_state& old_that )
{
  if ( old_that.get_bottom() <= get_bottom() ) 
    {
      if ( that.get_bottom() > get_bottom() ) 
        {
          claw::math::coordinate_2d<int> pos;
          pos.x = (int)that.get_bottom_middle().x;
          pos.y = (int)that.get_bottom_middle().y;

          // not about generic_items
          /*if ( that.get_mass() <= 1 ) 
            get_level_globals().get_sound("splash1").play( pos );
          else
          get_level_globals().get_sound("splash2").play( pos );*/
          
          that.set_environment_friction(m_bottom_friction);
          create_animation(m_top_animation, that.get_bottom_middle());
        }
    }
  else if ( that.get_bottom() <= get_bottom() ) 
    {
      that.set_environment_friction(m_top_friction);
      create_animation(m_bottom_animation, that.get_bottom_middle());
    }
} // change_friction_item::hit()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create an animation.
 * \param anim The animation
 */
void bear::change_friction_item::create_animation
( visual::animation* anim, universe::position_type pos )
{
  if ( anim )
    {
      engine::world* w = get_owner();

      engine::base_item* new_item = new decorative_item;
      
      visual::animation* new_anim = new visual::animation( *anim );
      
      new_anim->set_loops(1);
      
      new_item->set_animation_field("animation", new_anim);
      new_item->set_real_field("pos_x",pos.x-(anim->get_max_size().x/2));
      new_item->set_real_field("pos_y",get_bottom()-(anim->get_max_size().y));
      new_item->set_bool_field("kill_when_finished", true);
      
      w->register_item( new_item );

      CLAW_ASSERT( new_item->is_valid(),
        "change_friction item isn't correctly initialized" );
      new_item->start();
    }
} // bear::change_friction_item::create_animation()
