/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file base_block.tpp
 * \brief Implementation of the bear::base_block class.
 * \author Sebastien Angibaud
 */
#include <claw/logger.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Contructor.
 */
template<class Base, class EventGenerator>
bear::base_block<Base, EventGenerator>::base_block()
  : m_block_type(undefined_block)
{
  this->set_mass(0);
} // base_block::base_block()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type std::string.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 *
 * Valid values for the \a name and \a value parameters are :
 *  - "block_type", the type of the block (see header file)
 *  - anything supported by the bear::renderable_item class
 */
template<class Base, class EventGenerator>
bool bear::base_block<Base, EventGenerator>::set_string_field
( const std::string& name, const std::string& value )
{
  bool ok = true;

  if (name == "block_type")
    ok = set_type(value);
  else
    ok = super::set_string_field(name, value);

  return ok;
} // base_block::set_string_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is correctly initialized.
 */
template<class Base, class EventGenerator>
bool bear::base_block<Base, EventGenerator>::is_valid() const
{
  return this->block_type_is_valid() && super::is_valid();
} // base_block::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the collision events for the type left_top.
 */
template<class Base, class EventGenerator>
void
bear::base_block<Base, EventGenerator>::set_collision_events_left_top_wall()
{
  set_collision_event
    <universe::zone::top_left_zone, universe::zone::top_left_zone>();
  set_collision_event
    <universe::zone::top_zone, universe::zone::top_zone>();
  set_collision_event
    <universe::zone::top_right_zone, universe::zone::top_zone>();
  set_collision_event
    <universe::zone::middle_left_zone,
    universe::zone::middle_left_zone>();
  set_collision_event
    <universe::zone::bottom_left_zone, universe::zone::middle_left_zone>();
} // base_block::set_collision_events_left_top_wall()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the collision events for the type right_top .
 */
template<class Base, class EventGenerator>
void
bear::base_block<Base, EventGenerator>::set_collision_events_right_top_wall()
{
  set_collision_event
    <universe::zone::top_left_zone, universe::zone::top_zone>();
  set_collision_event
    <universe::zone::top_zone, universe::zone::top_zone>();
  set_collision_event
    <universe::zone::top_right_zone, universe::zone::top_right_zone>();
  set_collision_event
    <universe::zone::middle_right_zone, universe::zone::middle_right_zone>();
  set_collision_event
    <universe::zone::bottom_right_zone, universe::zone::middle_right_zone>();
} // base_block::set_collision_events_right_top_wall()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the collision events for the type left_center.
 */
template<class Base, class EventGenerator>
void
bear::base_block<Base,
                   EventGenerator>::set_collision_events_left_center_wall()
{
  set_collision_event
    <universe::zone::top_left_zone, universe::zone::middle_left_zone>();
  set_collision_event
    <universe::zone::middle_left_zone, universe::zone::middle_left_zone>();
  set_collision_event
    <universe::zone::bottom_left_zone, universe::zone::middle_left_zone>();
} // base_block::set_collision_events_left_center_wall()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the collision events for the type center_center.
 */
template<class Base, class EventGenerator>
void
bear::base_block<Base,
                   EventGenerator>::set_collision_events_center_center_wall()
{
  set_collision_event
    <universe::zone::top_left_zone, universe::zone::top_left_zone>();
  set_collision_event
    <universe::zone::top_zone, universe::zone::top_zone>();
  set_collision_event
    <universe::zone::top_right_zone, universe::zone::top_right_zone>();
  set_collision_event
    <universe::zone::middle_left_zone, universe::zone::middle_left_zone>();
  set_collision_event
    <universe::zone::middle_right_zone, universe::zone::middle_right_zone>();
  set_collision_event
    <universe::zone::bottom_left_zone, universe::zone::bottom_left_zone>();
  set_collision_event
    <universe::zone::bottom_zone, universe::zone::bottom_zone>();
  set_collision_event
    <universe::zone::bottom_right_zone, universe::zone::bottom_right_zone>();
} // base_block::set_collision_events_center_center_wall()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the collision events for the type right_center.
 */
template<class Base, class EventGenerator>
void
bear::base_block<Base,
                   EventGenerator>::set_collision_events_right_center_wall()
{
  set_collision_event
    <universe::zone::top_right_zone, universe::zone::middle_right_zone>();
  set_collision_event
    <universe::zone::middle_right_zone, universe::zone::middle_right_zone>();
  set_collision_event
    <universe::zone::bottom_right_zone, universe::zone::middle_right_zone>();
} // base_block::set_collision_events_right_center_wall()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the collision events for the type left_bottom.
 */
template<class Base, class EventGenerator>
void
bear::base_block<Base,
                   EventGenerator>::set_collision_events_left_bottom_wall()
{
  set_collision_event
    <universe::zone::top_left_zone, universe::zone::middle_left_zone>();
  set_collision_event
    <universe::zone::middle_left_zone, universe::zone::middle_left_zone>();
  set_collision_event
    <universe::zone::bottom_left_zone, universe::zone::bottom_left_zone>();
  set_collision_event
    <universe::zone::bottom_zone, universe::zone::bottom_zone>();
  set_collision_event
    <universe::zone::bottom_right_zone, universe::zone::bottom_zone>();
} // base_block::set_collision_events_left_bottom_wall()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the collision events for the type right_bottom.
 */
template<class Base, class EventGenerator>
void
bear::base_block<Base,
                   EventGenerator>::set_collision_events_right_bottom_wall()
{
  set_collision_event
    <universe::zone::top_right_zone, universe::zone::middle_right_zone>();
  set_collision_event
    <universe::zone::middle_right_zone, universe::zone::middle_right_zone>();
  set_collision_event
    <universe::zone::bottom_left_zone, universe::zone::bottom_zone>();
  set_collision_event
    <universe::zone::bottom_zone, universe::zone::bottom_zone>();
  set_collision_event
    <universe::zone::bottom_right_zone, universe::zone::bottom_right_zone>();
} // base_block::set_collision_events_right_bottom_wall()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the collision events for the type center_bottom.
 */
template<class Base, class EventGenerator>
void
bear::base_block<Base,  EventGenerator>::set_collision_events_center_ceiling()
{
  set_collision_event
    <universe::zone::bottom_left_zone, universe::zone::bottom_zone>();
  set_collision_event
    <universe::zone::bottom_zone, universe::zone::bottom_zone>();
  set_collision_event
    <universe::zone::bottom_right_zone, universe::zone::bottom_zone>();
} // base_block::set_collision_events_center_ceiling()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the collision events for the type left_ground.
 */
template<class Base, class EventGenerator>
void
bear::base_block<Base, EventGenerator>::set_collision_events_left_ground()
{
  set_collision_event
    <universe::zone::top_left_zone, universe::zone::top_left_zone>();
  set_collision_event
    <universe::zone::top_zone, universe::zone::top_zone>();
  set_collision_event
    <universe::zone::top_right_zone, universe::zone::top_zone>();
} // base_block::set_collision_events_left_ground()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the collision events for the type center_top.
 */
template<class Base, class EventGenerator>
void
bear::base_block<Base, EventGenerator>::set_collision_events_center_ground()
{
  set_collision_event
    <universe::zone::top_left_zone, universe::zone::top_zone>();
  set_collision_event
    <universe::zone::top_zone, universe::zone::top_zone>();
  set_collision_event
    <universe::zone::top_right_zone, universe::zone::top_zone>();
} // base_block::set_collision_events_center_top()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the collision events for the type right_ground.
 */
template<class Base, class EventGenerator>
void
bear::base_block<Base, EventGenerator>::set_collision_events_right_ground()
{
  set_collision_event
    <universe::zone::top_left_zone, universe::zone::top_zone>();
  set_collision_event
    <universe::zone::top_zone, universe::zone::top_zone>();
  set_collision_event
    <universe::zone::top_right_zone, universe::zone::top_right_zone>();
} // base_block::set_collision_events_right_ground()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the collision events for the type shifted_up_ground.
 */
template<class Base, class EventGenerator>
void
bear::base_block<Base,
		   EventGenerator>::set_collision_events_shifted_up_ground()
{
  set_collision_event
    <universe::zone::top_left_zone, universe::zone::top_zone>();
  set_collision_event
    <universe::zone::top_zone, universe::zone::top_zone>();
  set_collision_event
    <universe::zone::top_right_zone, universe::zone::top_zone>();
  set_collision_event
    <universe::zone::middle_left_zone, universe::zone::top_zone>();
  set_collision_event
    <universe::zone::middle_right_zone, universe::zone::top_zone>();
} // base_block::set_collision_events_shifted_up_ground()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the collision events for the type left_rod.
 */
template<class Base, class EventGenerator>
void bear::base_block<Base, EventGenerator>::set_collision_events_left_rod()
{
  set_collision_event
    <universe::zone::top_left_zone, universe::zone::top_left_zone>();
  set_collision_event
    <universe::zone::top_zone, universe::zone::top_zone>();
  set_collision_event
    <universe::zone::top_right_zone, universe::zone::top_zone>();
  set_collision_event
    <universe::zone::middle_left_zone, universe::zone::middle_left_zone>();
  set_collision_event
    <universe::zone::bottom_left_zone, universe::zone::bottom_left_zone>();
  set_collision_event
    <universe::zone::bottom_zone, universe::zone::bottom_zone>();
  set_collision_event
    <universe::zone::bottom_right_zone, universe::zone::bottom_zone>();
} // base_block::set_collision_events_left_rod()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the collision events for the type right rod.
 */
template<class Base, class EventGenerator>
void bear::base_block<Base, EventGenerator>::set_collision_events_right_rod()
{
  set_collision_event
    <universe::zone::top_left_zone, universe::zone::top_zone>();
  set_collision_event
    <universe::zone::top_zone, universe::zone::top_zone>();
  set_collision_event
    <universe::zone::top_right_zone, universe::zone::top_right_zone>();
  set_collision_event
    <universe::zone::middle_right_zone, universe::zone::middle_right_zone>();
  set_collision_event
    <universe::zone::bottom_left_zone, universe::zone::bottom_zone>();
  set_collision_event
    <universe::zone::bottom_zone, universe::zone::bottom_zone>();
  set_collision_event
    <universe::zone::bottom_right_zone, universe::zone::bottom_right_zone>();
} // base_block::set_collision_events_right_rod()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the collision events for the type top_rod.
 */
template<class Base, class EventGenerator>
void bear::base_block<Base, EventGenerator>::set_collision_events_top_rod()
{
  set_collision_event
    <universe::zone::top_left_zone, universe::zone::top_left_zone>();
  set_collision_event
    <universe::zone::top_zone, universe::zone::top_zone>();
  set_collision_event
    <universe::zone::top_right_zone, universe::zone::top_right_zone>();
  set_collision_event
    <universe::zone::middle_left_zone, universe::zone::middle_left_zone>();
  set_collision_event
    <universe::zone::middle_right_zone, universe::zone::middle_right_zone>();
  set_collision_event
    <universe::zone::bottom_left_zone, universe::zone::middle_left_zone>();
  set_collision_event
    <universe::zone::bottom_right_zone, universe::zone::middle_right_zone>();
} // base_block::set_collision_events_top_rod()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the collision events for the type bottom_rod.
 */
template<class Base, class EventGenerator>
void bear::base_block<Base, EventGenerator>::set_collision_events_bottom_rod()
{
  set_collision_event
    <universe::zone::top_left_zone, universe::zone::middle_left_zone>();
  set_collision_event
    <universe::zone::top_right_zone, universe::zone::middle_right_zone>();
  set_collision_event
    <universe::zone::middle_left_zone, universe::zone::middle_left_zone>();
  set_collision_event
    <universe::zone::middle_right_zone, universe::zone::middle_right_zone>();
  set_collision_event
    <universe::zone::bottom_left_zone, universe::zone::bottom_left_zone>();
  set_collision_event
    <universe::zone::bottom_zone, universe::zone::bottom_zone>();
  set_collision_event
    <universe::zone::bottom_right_zone, universe::zone::bottom_right_zone>();
} // base_block::set_collision_events_bottom_rod()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the collision events for the type vertical_rod.
 */
template<class Base, class EventGenerator>
void
bear::base_block<Base, EventGenerator>::set_collision_events_vertical_rod()
{
  set_collision_event
    <universe::zone::top_left_zone, universe::zone::middle_left_zone>();
  set_collision_event
    <universe::zone::top_right_zone, universe::zone::middle_right_zone>();
  set_collision_event
    <universe::zone::middle_left_zone, universe::zone::middle_left_zone>();
  set_collision_event
    <universe::zone::middle_right_zone, universe::zone::middle_right_zone>();
  set_collision_event
    <universe::zone::bottom_left_zone, universe::zone::middle_left_zone>();
  set_collision_event
    <universe::zone::bottom_right_zone, universe::zone::middle_right_zone>();
} // base_block::set_collision_events_()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the collision events for the type horizontal_rod.
 */
template<class Base, class EventGenerator>
void
bear::base_block<Base, EventGenerator>::set_collision_events_horizontal_rod()
{
  set_collision_event
    <universe::zone::top_left_zone, universe::zone::top_zone>();
  set_collision_event
    <universe::zone::top_zone, universe::zone::top_zone>();
  set_collision_event
    <universe::zone::top_right_zone, universe::zone::top_zone>();
  set_collision_event
    <universe::zone::bottom_left_zone, universe::zone::bottom_zone>();
  set_collision_event
    <universe::zone::bottom_zone, universe::zone::bottom_zone>();
  set_collision_event
    <universe::zone::bottom_right_zone, universe::zone::bottom_zone>();
} // base_block::set_collision_events_horizontal_rod()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item have a correct type of block.
 */
template<class Base, class EventGenerator>
bool bear::base_block<Base, EventGenerator>::block_type_is_valid() const
{
  return m_block_type != undefined_block;
} // base_block::block_type_is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the type of the block.
 * \param value The type if the block.
 * \return false if the field "value" is unknow, true otherwise.
 */
template<class Base, class EventGenerator>
bool
bear::base_block<Base, EventGenerator>::set_type( const std::string& value )
{
  bool ok = true;

  if ( value == "left_top_wall")
    m_block_type = left_top_wall;
  else if ( value == "right_top_wall")
    m_block_type = right_top_wall;
  else if ( value == "left_center_wall")
    m_block_type = left_center_wall;
  else if ( value == "center_center_wall")
    m_block_type = center_center_wall;
  else if ( value == "right_center_wall")
    m_block_type = right_center_wall;
  else if ( value == "left_bottom_wall")
    m_block_type = left_bottom_wall;
  else if ( value == "right_bottom_wall")
    m_block_type = right_bottom_wall;

  else if ( value == "center_ceiling")
    m_block_type = center_ceiling;

  else if ( value == "left_ground")
    m_block_type = left_ground;
  else if ( value == "center_ground" )
    m_block_type = center_ground;
  else if ( value == "right_ground")
    m_block_type = right_ground;
  else if ( value == "shifted_up_ground")
    m_block_type = shifted_up_ground;

  else if ( value == "left_rod")
    m_block_type = left_rod;
  else if ( value == "right_rod")
    m_block_type = right_rod;
  else if ( value == "top_rod")
    m_block_type = top_rod;
  else if ( value == "bottom_rod")
    m_block_type = bottom_rod;
  else if ( value == "vertical_rod")
    m_block_type = vertical_rod;
  else if ( value == "horizontal_rod")
    m_block_type = horizontal_rod;
  else
    {
      claw::logger << claw::log_warning
                   << "base_block::set_type(): unknow block type '" << value
                   << "'." << claw::lendl;
      ok = false;
    }

  if ( ok )
    set_collision_events_table();

  return ok;
} // base_block::set_type()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the possible collision events.
 * \pre m_block_type != undefined_block
 */
template<class Base, class EventGenerator>
void bear::base_block<Base, EventGenerator>::set_collision_events_table()
{
  CLAW_PRECOND( m_block_type != undefined_block );

  switch( m_block_type )
    {
    case left_top_wall:       set_collision_events_left_top_wall();      break;
    case right_top_wall:      set_collision_events_right_top_wall();     break;
    case left_center_wall:    set_collision_events_left_center_wall();   break;
    case center_center_wall:  set_collision_events_center_center_wall(); break;
    case right_center_wall:   set_collision_events_right_center_wall();  break;
    case left_bottom_wall:    set_collision_events_left_bottom_wall();   break;
    case right_bottom_wall:   set_collision_events_right_bottom_wall();  break;

    case center_ceiling:      set_collision_events_center_ceiling();     break;

    case left_ground:         set_collision_events_left_ground();        break;
    case center_ground:       set_collision_events_center_ground();      break;
    case right_ground:        set_collision_events_right_ground();       break;
    case shifted_up_ground:   set_collision_events_shifted_up_ground();  break;

    case left_rod:            set_collision_events_left_rod();           break;
    case right_rod:           set_collision_events_right_rod();          break;
    case top_rod:             set_collision_events_top_rod();            break;
    case bottom_rod:          set_collision_events_bottom_rod();         break;
    case vertical_rod:        set_collision_events_vertical_rod();       break;
    case horizontal_rod:      set_collision_events_horizontal_rod();     break;
    default:
      // undefined_block is checked in preconditions
      break;
    }
} // base_block::set_collision_events_table()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the collision event generated for zone=\a EventZone in the \a Zone
 *        zone.
 */
template<class Base, class EventGenerator>
template< bear::universe::zone::position Zone,
          bear::universe::zone::position EventZone >
void bear::base_block<Base, EventGenerator>::set_collision_event()
{
  EventGenerator generator;
  super::set_collision_event(Zone, generator.template get<EventZone>( *this ));
} // base_block::set_collision_event();
