/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file zoom_effect.cpp
 * \brief Implementation of the bear::visual::zoom_effect class.
 * \author Julien Jorge
 */
#include "visual/zoom_effect.hpp"
#include <claw/assert.hpp>

/*---------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param first_value Initial strength (speed).
 * \param last_value Last strength.
 * \param length Effect duration (itrations).
 * \pre 0 <= first_value <= 1 && 0 <= last_value <= 1
 */
bear::visual::zoom_effect::zoom_effect
( double first_value, double last_value,
  unsigned int length, const claw::math::coordinate_2d<unsigned int>& center )
  : progressive_screen_effect(first_value, last_value, length),
    m_center(center)
{
  CLAW_PRECOND( first_value >= 0 );
  CLAW_PRECOND( last_value >= 0 );
  CLAW_PRECOND( first_value <= 1 );
  CLAW_PRECOND( last_value <= 1 );
} // zoom_effect::fade_effect()

/*---------------------------------------------------------------------------*/
/**
 * \brief Apply the filter.
 * \param image The image to modify.
 * \param coeff Current filter coefficient.
 */
void bear::visual::zoom_effect::progressive_apply( claw::graphic::image& target,
                                             double coeff )
{
  // pixel in process
  claw::math::coordinate_2d<unsigned int> current_pixel;
  // source of the processing pixel
  claw::math::coordinate_2d<unsigned int> source_pixel;

  const unsigned int width = target.width();
  const unsigned int height = target.height();
  const claw::graphic::image reference( target );
  claw::graphic::pixel32* pixels;

  /*
    We will process the target in four parts to save sign comparisons.
    from the center pixel :
    - at the top left;
    - at the top right;
    - at the bottom left;
    - at the bottom right.
  */

  // at the top left : x - m_center.x < 0; y - m_center.y < 0
  for (current_pixel.y=0; current_pixel.y!=m_center.y; ++current_pixel.y)
    {
      pixels = target[current_pixel.y];

      for (current_pixel.x=0; current_pixel.x!=m_center.x; ++current_pixel.x)
        {
          source_pixel.set
            ( current_pixel.x
              + (unsigned int)((m_center.x - current_pixel.x) * coeff),
              current_pixel.y
              + (unsigned int)((m_center.y - current_pixel.y) * coeff) );

          average_of_pixel_line( reference, current_pixel, source_pixel,
                                 *pixels);
          ++pixels;
        }
    }

  // at the top right  : x - m_center.x >= 0; y - m_center.y < 0
  for (current_pixel.y=0; current_pixel.y!=m_center.y; ++current_pixel.y)
    {
      pixels = & target[current_pixel.y][m_center.x];

      for (current_pixel.x=m_center.x; current_pixel.x!=width;
           ++current_pixel.x)
        {
          source_pixel.set
            ( current_pixel.x
              - (unsigned int)((current_pixel.x - m_center.x) * coeff),
              current_pixel.y
              + (unsigned int)((m_center.y - current_pixel.y) * coeff) );

          average_of_pixel_line( reference, current_pixel, source_pixel,
                                 *pixels);
          ++pixels;
        }
    }

  // at the bottom left : x - m_center.x < 0; y - m_center.y >= 0
  for (current_pixel.y=m_center.y; current_pixel.y!=height; ++current_pixel.y)
    {
      pixels = target[current_pixel.y];

      for (current_pixel.x=0; current_pixel.x!=m_center.x; ++current_pixel.x)
        {
          source_pixel.set
            ( current_pixel.x
              + (unsigned int)((m_center.x - current_pixel.x) * coeff),
              current_pixel.y
              - (unsigned int)((current_pixel.y - m_center.y) * coeff) );

          average_of_pixel_line( reference, current_pixel, source_pixel,
                                 *pixels);
          ++pixels;
        }
    }

  // at the bottom right : x - m_center.x >= 0; y - m_center.y >= 0
  for (current_pixel.y=m_center.y; current_pixel.y!=height; ++current_pixel.y)
    {
      pixels = & target[current_pixel.y][m_center.x];

      for (current_pixel.x=m_center.x; current_pixel.x!=width;
           ++current_pixel.x)
        {
          source_pixel.set
            ( current_pixel.x
              - (unsigned int)((current_pixel.x - m_center.x) * coeff),
              current_pixel.y
              - (unsigned int)((current_pixel.y - m_center.y) * coeff) );

          average_of_pixel_line( reference, current_pixel, source_pixel,
                                 *pixels);
          ++pixels;
        }
    }
}

/*---------------------------------------------------------------------------*/
/**
 * \brief Calculate the average pixel of pixels between two points.
 * \param Reference Where we read pixels.
 * \param p1 The first point
 * \param p2 The second point
 * \param pixel (out) The average pixel.
 */
void bear::visual::zoom_effect::average_of_pixel_line
( const claw::graphic::image& reference,
  const claw::math::coordinate_2d<unsigned int>& p1,
  const claw::math::coordinate_2d<unsigned int>& p2,
  claw::graphic::pixel32& pixel ) const
{
  std::vector<unsigned int> colors(4, 0);
  unsigned int n; // Number of pixels between the two points

  n = sum_of_pixel_line( reference, p1, p2, colors );

  pixel.components.red   = colors[0] / n;
  pixel.components.green = colors[1] / n;
  pixel.components.blue  = colors[2] / n;
  pixel.components.alpha = colors[3] / n;
}

/*---------------------------------------------------------------------------*/
/**
 * \brief Calculate the sum of components of pixels between two points.
 * \param Reference Where we read pixels.
 * \param p1 The first point
 * \param p2 The second point
 * \param colors (in/out) Summed components.
 * \return The number of pixels between the two points.
 */
unsigned int bear::visual::zoom_effect::sum_of_pixel_line
( const claw::graphic::image& reference,
  const claw::math::coordinate_2d<unsigned int>& p1,
  const claw::math::coordinate_2d<unsigned int>& p2,
  std::vector<unsigned int>& colors) const
{
  // Bresenham algorithm
  int inc_x, inc_y;
  int dx, dy;
  int x, y;
  int x1, y1, x2, y2;
  int eps = 0;

  x1 = p1.x; y1 = p1.y;
  x2 = p2.x; y2 = p2.y;

  // Increments and deltas
  if (x2 < x1)
    {
      dx = x1 - x2;
      inc_x = -1;
    }
  else
    {
      dx = x2 - x1;
      inc_x = 1;
    }

  if (y2 < y1)
    {
      dy = y1 - y2;
      inc_y = -1;
    }
  else
    {
      dy = y2 - y1;
      inc_y = 1;
    }

  // Draw line
  if ( dx >= dy ) // low slope
    {
      y = y1;
      x2 += inc_x;
      for ( x = (int)x1; x != (int)x2; x+=inc_x )
        {
          colors[0] += reference[y][x].components.red;
          colors[1] += reference[y][x].components.green;
          colors[2] += reference[y][x].components.blue;
          colors[3] += reference[y][x].components.alpha;

          eps += dy;
          if ( 2 * eps >= dx )
            {
              y += inc_y;
              eps -= dx;
            }
        }

      return dx + 1;
    }
  else // high slope
    {
      x = x1;
      y2 += inc_y;
      for ( y = (int)y1; y != (int)y2; y+=inc_y )
        {
          colors[0] += reference[y][x].components.red;
          colors[1] += reference[y][x].components.green;
          colors[2] += reference[y][x].components.blue;
          colors[3] += reference[y][x].components.alpha;

          eps += dx;
          if ( 2 * eps >= dy )
            {
              x+=inc_x;
              eps -= dy;
            }
        }

      return dy + 1;
    }
}
