/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file quake_effect.cpp
 * \brief Implementation of the bear::visual::quake_effect class.
 * \author Julien Jorge
 */
#include "visual/quake_effect.hpp"
#include <claw/assert.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param first_value Initial move.
 * \param last_value Last move.
 * \param length Effect duration (itrations).
 * \pre first_value > 0 && last_value > 0
 */
bear::visual::quake_effect::quake_effect
( double first_value, double last_value, unsigned int length)
  : progressive_screen_effect(first_value, last_value, length)
{
  CLAW_PRECOND(first_value > 0);
  CLAW_PRECOND(last_value > 0);
} // quake_effect::quake_effect()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the filter.
 * \param image The image to modify.
 * \param coeff Current filter coefficient.
 */
void bear::visual::quake_effect::progressive_apply
( claw::graphic::image& target, k double coeff )
{
  int i_coeff = (int)(coeff + 0.5);
  int x, y, dx, dy, min_x, max_x, min_y, max_y;
  claw::math::coordinate_2d<int> delta( rand() % i_coeff - i_coeff / 2,
                                        rand() % i_coeff - i_coeff / 2 );

  // set the direction the strarting and the ending pixels
  if (delta.x < 0)
    {
      dx = 1;
      min_x = 0;
      max_x = target.width() + delta.x;
    }
  else
    {
      dx = -1;
      min_x = target.width() - 1;
      max_x = delta.x - 1;
    }

  if (delta.y < 0)
    {
      dy = 1;
      min_y = 0;
      max_y = target.height() + delta.y;
    }
  else
    {
      dy = -1;
      min_y = target.height() - 1;
      max_y = delta.y - 1;
    }

  // Then copy the picture on itself.
  if ( (min_y*dy < max_y*dy) && (min_x*dx < max_x*dx) )
    for (y=min_y; y!=max_y; y+=dy)
      {
        claw::graphic::pixel32* dest   = & target[y][min_x];
        claw::graphic::pixel32* source =
          & target[y - delta.y][min_x - delta.x];

        for (x=min_x; x!=max_x; x+=dx, source+=dx, dest+=dx)
          *dest = *source;
      }
} // quake_effect::progressive_apply()
