/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file pixel_effect.cpp
 * \brief Implementation of the bear::visual::pixel_effect class.
 * \author Julien Jorge.
 */
#include "visual/pixel_effect.hpp"
#include <claw/assert.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param first_value Initial length of the box' edges.
 * \param last_value Last length of the box' edges.
 * \param length Effect duration (itrations).
 * \pre first_value >= 1 && last_value >= 1
 */
bear::visual::pixel_effect::pixel_effect( double first_value, double last_value,
                                    unsigned int length)
  : progressive_screen_effect(first_value, last_value, length)
{
  CLAW_PRECOND(first_value >= 1);
  CLAW_PRECOND(last_value >= 1);
} // pixel_effect::pixel_effect() [constructor]

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the filter.
 * \param image The image to modify.
 * \param coeff Current filter coefficient : the length of a pixel "edge".
 */
void bear::visual::pixel_effect::progressive_apply
( claw::graphic::image& target, double coeff )
{
  unsigned int edge_length = (unsigned int)(coeff + 0.5);

  if (edge_length > 1)
    {
      claw::math::rectangle<unsigned int> area;
      unsigned int right_bound, bottom_bound;
      claw::graphic::pixel32 pixel;

      area.width = area.height = edge_length;
      right_bound  = target.width()  - target.width()  % edge_length;
      bottom_bound = target.height() - target.height() % edge_length;

      // first, full sized area
      for (area.position.y=0; area.position.y!=bottom_bound;
           area.position.y+=edge_length)
        for (area.position.x=0; area.position.x!=right_bound;
             area.position.x+=edge_length)
          {
            pixel = average_pixel( target, area );
            fill_area( target, area, pixel );
          }

      // bottom strip
      area.height = target.height() % edge_length;
      if (area.height)
        for (area.position.x=0; area.position.x!=right_bound;
             area.position.x+=edge_length)
          {
            pixel = average_pixel( target, area );
            fill_area( target, area, pixel );
          }

      // right_strip
      area.height = edge_length;
      area.width = target.width()  % edge_length;
      if (area.width)
        for (area.position.y=0; area.position.y!=bottom_bound;
             area.position.y+=edge_length)
          {
            pixel = average_pixel( target, area );
            fill_area( target, area, pixel );
          }

      // bottom-right corner
      area.height = target.height() % edge_length;
      if ( area.height && area.width )
        {
          pixel = average_pixel( target, area );
          fill_area( target, area, pixel );
        }
    }
} // pixel_effect::progressive_apply()

/*----------------------------------------------------------------------------*/
/**
 * \brief Calculate the average pixel of a part of an image.
 * \param refenrece The image in which we read pixels.
 * \param area The part of the image, to read.
 */
claw::graphic::pixel32
bear::visual::pixel_effect::average_pixel
( const claw::graphic::image& reference,
  const claw::math::rectangle<unsigned int>& area ) const
{
  unsigned int r, g, b, a;
  claw::graphic::pixel32 result;

  unsigned int x, x_bound;
  unsigned int y, y_bound;
  const claw::graphic::pixel32* pixels;

  r = g = b = a = 0;

  x_bound = area.right();
  y_bound = area.bottom();

  for (y=area.position.y; y<=y_bound; ++y)
    {
      pixels = & reference[y][area.position.x];

      for (x=area.position.x; x<=x_bound; ++x, ++pixels)
        {
          r += pixels->components.red;
          g += pixels->components.green;
          b += pixels->components.blue;
          a += pixels->components.alpha;
        }
    }

  result.components.red   = r / area.area();
  result.components.green = g / area.area();
  result.components.blue  = b / area.area();
  result.components.alpha = a / area.area();

  return result;
} // pixel_effect::average_pixel()

/*----------------------------------------------------------------------------*/
/**
 * \brief Fill a part of an image with an unique color.
 * \param target The image to fill.
 * \param area The part of the image, to fill.
 * \param color Filling color.
 */
void bear::visual::pixel_effect::fill_area
( claw::graphic::image& target,
  const claw::math::rectangle<unsigned int>& area,
  const claw::graphic::pixel32& color ) const
{
  unsigned int x, x_bound;
  unsigned int y, y_bound;
  claw::graphic::pixel32* pixels;

  x_bound = area.right();
  y_bound = area.bottom();

  for (y=area.position.y; y<=y_bound; ++y)
    {
      pixels = & target[y][area.position.x];

      for (x=area.position.x; x<=x_bound; ++x, ++pixels)
        *pixels = color;
    }
} // bear::visual::pixel_effect::fill_area()
