/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file fade_effect.cpp
 * \brief Implementation of the bear::visual::fade_effect class.
 * \author Julien Jorge.
 */
#include "visual/fade_effect.hpp"
#include <claw/assert.hpp>

/*---------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param first_value Initial pixels intensity (percentage).
 * \param last_value Last pixel_intensity.
 * \param length Effect duration (itrations).
 * \pre 0 <= first_value <= 1 && 0 <= last_value <= 1
 */
bear::visual::fade_effect::fade_effect( double first_value, double last_value,
                                  unsigned int length )
  : progressive_screen_effect(first_value, last_value, length)
{
  CLAW_PRECOND( 0 <= first_value );
  CLAW_PRECOND( 0 <= last_value );
  CLAW_PRECOND( first_value <= 1 );
  CLAW_PRECOND( last_value <= 1 );
} // fade_effect::fade_effect() [constructor]

/*---------------------------------------------------------------------------*/
/**
 * \brief Apply the filter.
 * \param image The image to modify.
 * \param coeff Current filter coefficient.
 */
void bear::visual::fade_effect::progressive_apply( claw::graphic::image& target,
                                             double coeff )
{
  claw::graphic::image::iterator pixel = target.begin();
  const unsigned int pixels_count = target.width() * target.height();

  for (unsigned int i=0; i!=pixels_count; ++i, ++pixel)
    {
      pixel->components.red =
        (unsigned char)((double)pixel->components.red * coeff );
      pixel->components.green =
        (unsigned char)((double)pixel->components.green * coeff );
      pixel->components.blue =
        (unsigned char)((double)pixel->components.blue * coeff );
    }

} // fade_effect::progressive_apply()
