/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file animation.cpp
 * \brief Implementation of the bear::visual::animation class.
 * \author Julien Jorge
 */
#include "visual/animation.hpp"
#include <claw/assert.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param images Names of the frames images.
 * \pre images.size() > 0
 * \post loops == 0, loop_back == false
 * \remark The sprites pointers passed as arguments will be deleted in the
 *         destructor.
 */
bear::visual::animation::animation
( const bear::visual::animation::frame_list& images )
  : m_sprites(images), m_index(0), m_loops(0), m_loop_back(false),
    m_forward(true), m_play_count(0)
{
  CLAW_PRECOND(images.size() > 0);
} // animation::animation()

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy constructor.
 * \param that The animation to copy from.
 */
bear::visual::animation::animation( const bear::visual::animation& that )
  : m_index(that.m_index), m_loops(that.m_loops), m_loop_back(that.m_loop_back),
    m_forward(that.m_forward), m_play_count(that.m_play_count)
{
  unsigned int i;

  m_sprites.resize( that.m_sprites.size() );

  try
    {
      for (i = 0; i != m_sprites.size(); ++i)
        m_sprites[i] = new sprite( *that.m_sprites[i] );
    }
  catch (...)
    {
      while (i!=0)
        {
          --i;
          delete m_sprites[i];
        }
      throw;
    }
} // animation::animation() [copy constructor]

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::visual::animation::~animation()
{
  std::vector<sprite*>::iterator it;

  for (it = m_sprites.begin(); it!=m_sprites.end(); ++it)
    delete *it;
} // animation::~animation()

/*----------------------------------------------------------------------------*/
/**
 * \brief Come back to first frame.
 */
void bear::visual::animation::reset()
{
  m_index = 0;
  m_forward = true;
  m_play_count = 0;
} // animation::reset()

/*----------------------------------------------------------------------------*/
/**
 * \brief Go to the next frame.
 */
void bear::visual::animation::next()
{
  if ( !is_finished() )
    {
      if (m_forward)
        next_forward();
      else
        next_backward();
    }
} // animation::next()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the current frame.
 */
const bear::visual::sprite& bear::visual::animation::get_sprite() const
{
  return *(m_sprites[m_index]);
} // animation::get_sprite()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the biggest size of the frames.
 */
claw::math::coordinate_2d<unsigned int>
bear::visual::animation::get_max_size() const
{
  unsigned int width=0, height=0;

  for (unsigned int i=0; i!=m_sprites.size(); ++i)
    {
      if ( m_sprites[i]->get_size().x > width )
        width = m_sprites[i]->get_size().x;

      if ( m_sprites[i]->get_size().y > height )
        height = m_sprites[i]->get_size().y;
    }

  return claw::math::coordinate_2d<unsigned int>( width, height );
} // animation::get_max_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set loop back mode.
 */
void bear::visual::animation::set_loop_back( bool loop_back )
{
  m_loop_back = loop_back;
} // animation::set_loop_back()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the number of times the animation should be read.
 */
void bear::visual::animation::set_loops( unsigned int loops )
{
  m_play_count = 0;
  m_loops = loops;
} // animation::set_loops()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get animation's translucy.
 */
double bear::visual::animation::get_alpha_blend() const
{
  return m_sprites[0]->get_alpha_blend();
} // animation::get_alpha_blend()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set animation's translucy.
 */
void  bear::visual::animation::set_alpha_blend( double alpha_blend )
{
  std::vector<sprite*>::iterator it;

  for (it = m_sprites.begin(); it != m_sprites.end(); ++it)
    (*it)->set_alpha_blend(alpha_blend);
} // animation::set_alpha_blend()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set vertical inversion.
 * \param b The new value for the inversion.
 */
void bear::visual::animation::flip( bool b )
{
  std::vector<sprite*>::iterator it;

  for (it = m_sprites.begin(); it != m_sprites.end(); ++it)
    (*it)->flip(b);
} // animation::flip()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set horizontal inversion.
 * \param b The new value for the inversion.
 */
void bear::visual::animation::mirror( bool b )
{
  std::vector<sprite*>::iterator it;

  for (it = m_sprites.begin(); it != m_sprites.end(); ++it)
    (*it)->mirror(b);
} // animation::mirror()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get vertical inversion.
 */
bool bear::visual::animation::is_flipped() const
{
  return m_sprites[0]->is_flipped();
} // animation::is_flipped()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get horizontal inversion.
 */
bool bear::visual::animation::is_mirrored() const
{
  return m_sprites[0]->is_mirrored();
} // animation::is_mirrored()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the final status.
 */
bool bear::visual::animation::is_finished() const
{
  return (m_play_count == m_loops) && m_loops;
} // animation::is_finished()

/*----------------------------------------------------------------------------*/
/**
 * \brief Go to the next frame, forward playing.
 */
void bear::visual::animation::next_forward()
{
  ++m_index;

  if ( m_index == m_sprites.size() ) // we're after the end
    {
      if ( m_loop_back ) // should we play reverse ?
        {
          m_forward = false;

          if (m_index >= 2)
            m_index -= 2;
          else
            m_index = 0;
        }
      else // come back directly to the first frame
        {
          ++m_play_count;

           // the last time we play the animation, we'll stay on the last frame
          if ( m_play_count == m_loops )
            --m_index;
          else
            m_index = 0;
        }
    }
} // animation::next_forward()

/*----------------------------------------------------------------------------*/
/**
 * \brief Go to the next frame, backward playing.
 */
void bear::visual::animation::next_backward()
{
  if ( m_index == 0 ) // we're at the begining
    {
      ++m_play_count;

      if ( m_sprites.size() > 1 )
        {
          m_forward = true;

          if ( (m_play_count != m_loops) || !m_loops )
            ++m_index;
        }
    }
  else
    --m_index;
} // animation::next_backward()
