/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file universe/world.hpp
 * \brief This is the representation of the world.
 * \author Julien Jorge.
 */
#ifndef __UNIVERSE_WORLD_HPP__
#define __UNIVERSE_WORLD_HPP__

#include "concept/item_container.hpp"
#include "concept/static_map.hpp"
#include "concept/region.hpp"
#include "universe/physic_rules.hpp"
#include "universe/base_entity.hpp"
#include "universe/class_export.hpp"

#include <vector>

namespace bear
{
  namespace universe
  {
    /**
     * \brief This is the representation of the world.
     *
     * The world is made of static surfaces (round, walls, ...), living items
     * (heros, enemies, ... ) and everything is governed by physical rules.
     * \author Julien Jorge.
     */
    class UNIVERSE_EXPORT world:
      public concept::item_container<base_entity*>
    {
    public:
      /**
       * \brief Class given to concept::static_map, as traits for base_entity
       *        pointers.
       */
      class base_entity_traits
      {
      public:
        claw::math::rectangle<unsigned int>
        get_bounding_box( base_entity* const& item ) const;
      }; // class base_entity_traits

      /** \brief Structure used for representing a region (a part) of the
          world. */
      typedef concept::region<rectangle_type> region_type;

      /** \brief The type of the map containing static items. */
      typedef
      concept::static_map<base_entity*, base_entity_traits> item_map;

    protected:
      typedef std::set<base_entity*> item_set;

    public:
      world( const size_box_type& size );

      void progress_entities
      ( const region_type& regions, time_type elapsed_time );

      void add_static(base_entity* who);

      const size_box_type& get_size() const;
      void print_stats() const;

    protected:
      void list_active_items
      ( item_set& items, const region_type& regions ) const;
      void pick_items( item_set& items, const position_type& pos ) const;

    private:
      void search_interesting_items
      ( const region_type& regions,  item_set& static_items,
        item_set& living_items, item_set& global_items);
    
      void stabilize_dependent_items
      (item_set& static_items, item_set& living_items, item_set& global_items);

      void add_dependent_items
      ( item_set& static_items, item_set& living_items, 
        item_set& global_items,
        const std::set<physical_item_state*>& dependent_item );
    
      void progress_items
      ( const item_set& items, time_type elapsed_time ) const;

      void active_region_traffic
      ( const region_type& regions, const item_set& items );

      void add_in_physic( const item_set& items );
      void add_in_physic_globals( const item_set& items );

      void list_static_items
      ( const region_type& regions, item_set& items ) const;

      bool item_in_regions
      ( const base_entity& item, const region_type& regions ) const;

      void add( base_entity* const& who );
      void remove( base_entity* const& who );

    private:
      /** \brief Size of the parts of m_static_surfaces. */
      static const unsigned int c_map_compression;

      /** \brief The living entities. Can be added and deleted any time. */
      item_set m_entities;

      /** \brief The static surfaces of the world. */
      item_map m_static_surfaces;

      /** \brief Physic rules. */
      physic_rules m_physic;

      /** \brief The size of the world. */
      size_box_type m_size;

      /** \brief Entity in the last active region. */
      item_set m_last_interesting_items;

    }; // class world
  } // namespace universe
} // namespace bear

#endif // __UNIVERSE_WORLD_HPP__
