/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file universe/code/world.cpp
 * \brief Implementation of the bear::universe::world class.
 * \author Julien Jorge
 */
#include "universe/world.hpp"

/*----------------------------------------------------------------------------*/
const unsigned int bear::universe::world::c_map_compression = 256;

/*----------------------------------------------------------------------------*/
claw::math::rectangle<unsigned int>
bear::universe::world::base_entity_traits::get_bounding_box
( base_entity* const & item ) const
{
  CLAW_PRECOND( item != NULL );

  return item->get_bounding_box().cast_value_type_to<unsigned int>();
} // world::base_entity_traits::get_bounding_box()




/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param size Size of the world.
 */
bear::universe::world::world( const size_box_type& size )
  : m_static_surfaces( (unsigned int)size.x + 1, (unsigned int)size.y + 1,
                       c_map_compression ),
    m_size(size)
{
  m_last_interesting_items.clear();
} // world::world()

/*----------------------------------------------------------------------------*/
/**
 * \brief Call the progress() method on entities in the active region, then
 *        apply physic rules.
 * \param regions The active regions.
 * \param elapsed_time Elapsed time since the last call of this method.
 */
void bear::universe::world::progress_entities
( const region_type& regions, time_type elapsed_time )
{
  item_set static_items;
  item_set living_items;
  item_set global_items;

  lock();

  search_interesting_items(regions, static_items, living_items, global_items);

  progress_items(static_items, elapsed_time);
  progress_items(living_items, elapsed_time);
  progress_items(global_items, elapsed_time);

  active_region_traffic( regions, living_items );

  m_physic.begin_listing();

  add_in_physic(static_items);
  add_in_physic(living_items);
  add_in_physic(global_items);

  m_physic.end_listing();
  m_physic.progress( elapsed_time );

  unlock();
} // world::progress_entities()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a static item in the world.
 * \param who The item to add.
 */
void bear::universe::world::add_static(base_entity* who)
{
  CLAW_PRECOND( who != NULL );
  CLAW_PRECOND( !locked() );

  who->fix();
  m_static_surfaces.insert( who );
} // world::add_static()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the size of the world.
 */
const bear::universe::size_box_type& bear::universe::world::get_size() const
{
  return m_size;
} // world::get_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Print some statistics.
 */
void bear::universe::world::print_stats() const
{
  unsigned int min, max;
  double avg;

  m_static_surfaces.cells_load(min, max, avg);

  claw::logger << claw::log_verbose << "World's size is " << m_size.x << ", "
               << m_size.y << '\n'
               << "Cells' size is " << c_map_compression << '\n'
               << "The loading is (min, max, avg) (" << min << '\t' << max
               << '\t' << avg << ")\n"
               << m_static_surfaces.empty_cells() << " cells are empty\n"
               << "There are " << m_entities.size() << " entities."
               << claw::lendl;
} // world::print_stats()

/*----------------------------------------------------------------------------*/
/**
 * \brief List items and entities which are in the active region.
 * \param items (out) The interesting items.
 * \param regions A list of regions in which we take the items.
 */
void bear::universe::world::list_active_items
( item_set& items, const region_type& regions ) const
{
  item_set::const_iterator it;

  list_static_items( regions, items );

  for ( it=m_entities.begin(); it!=m_entities.end(); ++it )
    if ( item_in_regions(**it, regions) )
      items.insert( *it );
} // world::list_active_items()

/*----------------------------------------------------------------------------*/
/**
 * \brief Pick items overlapping a given position.
 * \param items (out) The interesting items.
 * \param pos The position where the items are searched.
 */
void bear::universe::world::pick_items
( item_set& items, const position_type& pos ) const
{
  rectangle_type region
    ( pos.x, pos.y, 2 * c_map_compression, pos.y - c_map_compression );

  if ( pos.x >= c_map_compression )
    region.position.x -= c_map_compression;
  else
    region.position.x = 0;

  if ( pos.y >= c_map_compression )
    region.position.y -= c_map_compression;
  else
    region.position.y = 0;

  item_set items_found;
  item_set::const_iterator it;
  region_type r;
  r.push_front(region);

  list_active_items( items_found, r );

  for ( it=items_found.begin(); it!=items_found.end(); ++it )
    {
      bear::universe::rectangle_type box = (*it)->get_bounding_box();

      if ( box.includes(pos) )
        items.insert( *it );
    }
} // world::pick_items()

/*----------------------------------------------------------------------------*/
/**
 * \brief Search all interesting items in the active region
 *  and add dependent items.
 * \param regions The active regions.
 * \param static_items List of static items.
 * \param living_items List of moving and non global items.
 * \param global_items List of global items.
 */
void bear::universe::world::search_interesting_items
( const region_type& regions, item_set& static_items, item_set& living_items,
  item_set& global_items)
{
  item_set::const_iterator it;

  list_static_items( regions, static_items );
  
  for ( it=m_entities.begin(); it!=m_entities.end(); ++it )
    if ( item_in_regions(**it, regions) )
      living_items.insert(*it);
    else if ( (*it)->is_global() )
      global_items.insert(*it);   
    
  stabilize_dependent_items(static_items, living_items, global_items);
} // world::search_interesting_items()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add and stabilize the lists of items.
 * \param static_items List of static items.
 * \param living_items List of moving and non global items.
 * \param global_items List of global items.
 */
void bear::universe::world::stabilize_dependent_items
( item_set& static_items, item_set& living_items, item_set& global_items )
{
  unsigned int new_size =
    static_items.size() + global_items.size() + living_items.size();  
  unsigned int last_size;
  item_set::const_iterator it;
  
  do
    {
      std::set<physical_item_state*> dependent_items;

      for ( it=living_items.begin(); it!=living_items.end(); ++it )
        (*it)->get_dependent_items(dependent_items);

      for ( it=global_items.begin(); it!=global_items.end(); ++it )
        (*it)->get_dependent_items(dependent_items);

      for ( it=static_items.begin(); it!=static_items.end(); ++it )
        (*it)->get_dependent_items(dependent_items);

      if( !dependent_items.empty() )
        add_dependent_items
          ( static_items, living_items, global_items, dependent_items );

      last_size = new_size;
      new_size =
        static_items.size() + global_items.size() + living_items.size();
    }
  while (new_size != last_size);
} // world::stabilize_dependent_items()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add and stabilize the lists of items.
 * \param static_items List of static items.
 * \param living_items List of moving and non global items.
 * \param global_items List of global items.
 * \param dependent_items Set of dependent_items.
 */
void bear::universe::world::add_dependent_items
( item_set& static_items, item_set& living_items, item_set& global_items,
  const std::set<physical_item_state*>& dependent_items )
{
  std::set<physical_item_state*>::iterator it;

  for( it=dependent_items.begin(); it!=dependent_items.end(); ++it )
    {
      base_entity* item = static_cast<base_entity*>(*it);
      
      if ( item->is_fixed() )
        static_items.insert(item);
      else if ( item->is_global() )
        global_items.insert(item);
      else
        living_items.insert(item);
    }
} // world::add_dependent_items()

/*----------------------------------------------------------------------------*/
/**
 * \brief Call the progress() method on some items.
 * \param items The items on which we call the method.
 * \param elapsed_time Elapsed time since the last call.
 */
void bear::universe::world::progress_items
( const item_set& items, time_type elapsed_time ) const
{
  item_set::const_iterator it;

  for( it=items.begin(); it!=items.end(); ++it )
    (*it)->progress( elapsed_time );
} // world::progress_items()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell items if they leave the active region.
 * \param regions A list of regions in which we take the items.
 * \param items The items to verify.
 */
void bear::universe::world::active_region_traffic
( const region_type& regions, const item_set& items )
{
  item_set::const_iterator it;

  for( it=m_last_interesting_items.begin(); it!=m_last_interesting_items.end();
       ++it )
    if ( m_entities.find(*it) != m_entities.end() )
      if ( items.find(*it) == items.end() )
        (*it)->left_active_region();
        
  m_last_interesting_items = items;
} // world::active_region_traffic()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add some items in the physic_rules instance.
 * \param static_item List of static items.
 */
void bear::universe::world::add_in_physic( const item_set& static_items)
{
  item_set::const_iterator it;

  for (it=static_items.begin(); it!=static_items.end(); ++it)
    m_physic.add_item( **it );
} // world::add_items_in_physic()

/*----------------------------------------------------------------------------*/
/**
 * \brief Call the progress() method on all the entities in the active region
 *        and add them to the physic_rules instance.
 * \param regions A list of regions in which we take the items.
 * \param elapsed_time Elapsed time since the last call.
 */
void bear::universe::world::add_in_physic_globals( const item_set& items )
{
  item_set::const_iterator it;

  for( it=items.begin(); it!=items.end(); ++it )
    m_physic.add_global_item( **it );
} // world::add_items_in_physic_globals()

/*----------------------------------------------------------------------------*/
/**
 * \brief List static items which are in the active region.
 * \param items (out) The interesting items.
 * \param regions A list of regions in which we take the items.
 */
void bear::universe::world::list_static_items
( const region_type& regions, item_set& items ) const
{
  region_type::const_iterator it;

  for (it=regions.begin(); it!=regions.end(); ++it)
    m_static_surfaces.get_area( it->cast_value_type_to<unsigned int>(), items );
} // world::list_static_items()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if an item is in a region.
 * \param item The item to check.
 * \param regions A list of regions in which search the item.
 */
bool bear::universe::world::item_in_regions
( const base_entity& item, const region_type& regions ) const
{
  return regions.intersects( item.get_bounding_box() );
} // world::item_in_regions()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add an entity in the world.
 */
void bear::universe::world::add(base_entity* const& who)
{
  m_entities.insert( who );
} // world::add()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove an entity from the world.
 */
void bear::universe::world::remove(base_entity* const& who)
{
  if ( m_entities.find(who) != m_entities.end() ) 
    m_entities.erase(who);
} // world::remove()

