/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file physical_item.cpp
 * \brief Implementation of the bear::universe::physical_item class.
 * \author Julien Jorge
 */

#include <algorithm>

#include "universe/physical_item.hpp"
#include "universe/forced_movement.hpp"
#include "universe/collision_event/collision_event.hpp"
#include "universe/link/base_link.hpp"

#include <claw/assert.hpp>
#include <claw/functional.hpp>
#include <claw/logger.hpp>

/*----------------------------------------------------------------------------*/
const double
bear::universe::physical_item::c_acceleration_epsilon = 0.0001; // 0.001;
const double bear::universe::physical_item::c_speed_epsilon = 0.1; // 0.6

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::universe::physical_item::physical_item()
  : m_forced_movement(NULL),
    m_collision_event( zone::cardinality, new collision_event )
{

} // physical_item::physical_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy constructor.
 * \param that The item to copy from.
 * \remark Links and forced movement are not copied.
 */
bear::universe::physical_item::physical_item( const physical_item& that )
  : m_forced_movement(NULL), m_collision_event( that.m_collision_event )
{

} // physical_item::physical_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::universe::physical_item::~physical_item()
{
  clear_forced_movement();
  remove_all_links();
} // physical_item::~physical_item() [destructor]

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the forced movement applied to this item, if any.
 * \pre this->have_force_movement()
 */
bear::universe::forced_movement&
bear::universe::physical_item::get_forced_movement()
{
  CLAW_PRECOND( have_forced_movement() );

  return *m_forced_movement;
} // physical_item::get_forced_movement()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply a forced movement to the item.
 * \param m The movement to apply.
 * \remark If this object already have a movement, il will be released.
 * \remark m must be dynamically allocated. Current item will deal with
 *         realeasing, so don't worry about that.
 */
void bear::universe::physical_item::set_forced_movement( forced_movement& m )
{
  clear_forced_movement();

  if ( is_fixed() )
    claw::logger << claw::log_warning
                 << "physical_item::set_forced_movement(): setting a "
                 << "forced movement but the item is fixed." << claw::lendl;

  set_speed( speed_type(0, 0) );
  set_acceleration( force_type(0, 0) );
  m_forced_movement = &m;
} // physical_item::set_forced_movement()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check if the item have a forced movement.
 */
bool bear::universe::physical_item::have_forced_movement() const
{
  return m_forced_movement != NULL;
} // physical_item::have_forced_movement()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove the forced movement, if any.
 */
void bear::universe::physical_item::clear_forced_movement()
{
  if (m_forced_movement)
    {
      delete m_forced_movement;
      m_forced_movement = NULL;
    }
} // physical_item::clear_forced_movement()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the movement of the item.
 * \param elapsed_time Elasped time since the last progress.
 */
void bear::universe::physical_item::move( time_type elapsed_time )
{
  default_move(elapsed_time);
} // physical_item::move()

/*----------------------------------------------------------------------------*/
/**
 * \brief When a collision between two items occurs, hit() is called from the
 *        strongest item.
 * \param that The other item of the collision.
 * \param old_self The state to consider for the the current item.
 * \param old_that The state to consider for the item "that".
 */
void bear::universe::physical_item::collision
( physical_item& that, const physical_item_state& old_self,
  const physical_item_state& old_that )
{
  default_collision_event( that, old_self, old_that );
} // physical_item::collision()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a link in this item.
 * \param link The link to add.
 * \pre The caller is bear::universe::base_link.
 * \pre The link is not already in the item's list.
 * \remark The item keeps a reference on the link.
 */
void bear::universe::physical_item::add_link( base_link& link )
{
  CLAW_PRECOND
    ( std::find(m_links.begin(), m_links.end(), &link) == m_links.end() );

  m_links.push_front(&link);
} // physical_item::add_link()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove a link from this items.
 * \param link The link to remove.
 * \pre The caller is bear::universe::base_link.
 * \pre The link is in the item's list.
 */
void bear::universe::physical_item::remove_link( base_link& link )
{
  CLAW_PRECOND
    ( std::find(m_links.begin(), m_links.end(), &link) != m_links.end() );

  m_links.erase( std::find(m_links.begin(), m_links.end(), &link) );
} // physical_item::remove_link()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator at the begining of the list of links.
 */
bear::universe::physical_item::const_link_iterator
bear::universe::physical_item::links_begin() const
{
  return m_links.begin();
} // physical_item::links_begin()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator at the end of the list of links.
 */
bear::universe::physical_item::const_link_iterator
bear::universe::physical_item::links_end() const
{
  return m_links.end();
} // physical_item::links_end()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a dependent item.
 * Parameters are not commented, S�bastien ?
 */
void
bear::universe::physical_item::add_dependent_item( physical_item_state& item)
{
  m_dependent_items.insert(&item);
} // physical_item::add_dependent_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove a dependent item.
 * Parameters are not commented, S�bastien ?
 */
void bear::universe::physical_item::remove_dependent_item
( physical_item_state& item )
{
  m_dependent_items.erase(&item);
} // physical_item::remove_dependent_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the list of dependent items.
 * Parameters are not commented, S�bastien ?
 */
void bear::universe::physical_item::get_dependent_items
( std::set<physical_item_state*>& items_set ) const
{
  items_set.insert( m_dependent_items.begin(), m_dependent_items.end() );
} // physical_item::get_dependent_items()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the default movement of the item (forced_movement or
 *        acceleration and speed).
 * \param elapsed_time Elasped time since the last progress.
 */
void bear::universe::physical_item::default_move( time_type elapsed_time )
{
  if ( have_forced_movement() )
    m_forced_movement->next_position();
  else
    update_position(elapsed_time);
} // physical_item::default_move()

/*----------------------------------------------------------------------------*/
/**
 * \brief Change the method of alignment when a collision occurs.
 * \param z The zone to change.
 * \param event The new event associated with this zone.
 * \remark The parameter "event" must point to a dynamically allocated instance.
 *         Moreover, you do not have to take care about its destruction;
 *         bear::universe::physical_item will do it for you.
 */
void bear::universe::physical_item::set_collision_event
( zone::position z, collision_event* event )
{
  CLAW_PRECOND( event != NULL );

  //delete m_collision_event[z];
  m_collision_event[z] = event;
} // physical_item::set_collision_event()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the default event when a collision occurs between two items.
 * \param that (in/out) The item to align.
 * \param old_self The state to consider for the current item.
 * \param old_that The state to consider for the item "that".
 */
void bear::universe::physical_item::default_collision_event
( physical_item& that, const physical_item_state& old_self,
  const physical_item_state& old_that )
{
  collision_info info( old_self, old_that, *this, that );

  m_collision_event[ info.get_collision_side() ]->execute( info, *this, that );
} // physical_item::default_collision_event()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove all the links from this items.
 */
void bear::universe::physical_item::remove_all_links()
{
  // The base_link::unlink() method must call physical_item::remove_link() so
  // we don't have to remove it from here.
  while ( !m_links.empty() )
    {
      base_link* link = m_links.front();
      delete link;
    }
} // physical_item::remove_all_links()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the position, the speed and the acceleration of the item.
 * \param elapsed_time Elasped time since the last progress.
 */
void bear::universe::physical_item::update_position( time_type elapsed_time )
{
  force_type a( get_acceleration() + get_gravity() );
  position_type pos( get_position() );

  speed_type speed;
  speed =  ( a + get_speed() ) * get_environment_friction() * get_friction();
  set_speed( speed );

  pos += get_speed() + a / 2;

  set_position(pos);

  adjust_cinetic();
} // physical_item::update_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Calculate the new acceleration of the item.
 */
void bear::universe::physical_item::adjust_cinetic()
{
  force_type s( get_speed() );

  if ( (s.x < get_speed_epsilon().x) && (s.x > -get_speed_epsilon().x) )
    s.x = 0;
  if ( (s.y < get_speed_epsilon().y) && (s.y > -get_speed_epsilon().y) )
    s.y = 0;

  set_acceleration( force_type(0, 0) );
  set_speed( s );
} // physical_item::adjust_cinetic()
