/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file physic_rules.cpp
 * \brief Implementation of the bear::universe::physic_rules class.
 * \author Julien Jorge
 */
#include "universe/physic_rules.hpp"
#include "universe/forced_movement.hpp"
#include "universe/link/base_link.hpp"

#include <claw/assert.hpp>
#include <claw/avl.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor
 */
bear::universe::physic_rules::physic_rules()
  : m_in_progress(false)
{

} // physic_rules::physic_rules() [constructor]

/*----------------------------------------------------------------------------*/
/**
 * \brief Add an item for the next progress() call.
 * \param item The item to add.
 */
void bear::universe::physic_rules::add_item(physical_item& item)
{
  CLAW_PRECOND( !m_in_progress );

  item.clear_contacts();

  if ( (item.get_mass() == 0) && (!item.have_forced_movement())
       || item.is_fixed() )
    m_static_items.push_front( &item );
  else
    m_mobile_items.push_front( &item );
} // physic_rules::add_static_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a global item for the next progress() call.
 * \param item The item to add.
 */
void bear::universe::physic_rules::add_global_item(physical_item& item)
{
  CLAW_PRECOND( !m_in_progress );

  item.clear_contacts();
  m_global_items.push_front( &item );
} // physic_rules::add_global_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Start the listing of items for next progression (successive calls
 *        to ad_item()).
 */
void bear::universe::physic_rules::begin_listing()
{
  CLAW_PRECOND( !m_in_progress );

  m_static_items.clear();
  m_mobile_items.clear();
  m_global_items.clear();
} // physic_rules::begin_listing()

/*----------------------------------------------------------------------------*/
/**
 * \brief Start the listing of items for next progression.
 */
void bear::universe::physic_rules::end_listing()
{
  CLAW_PRECOND(!m_in_progress );
} // physic_rules::end_listing()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update positions of stored items and check collisions.
 * \param elapsed_time Elasped time since the last progress.
 */
void bear::universe::physic_rules::progress( time_type elapsed_time )
{
  CLAW_PRECOND( !m_in_progress );

  m_in_progress = true;

  add_items( m_static_items );
  add_items( m_mobile_items );
  add_items( m_global_items );

  progress_positions(elapsed_time);
  apply_links();
  detect_collisions();

  m_in_progress = false;
} // physic_rules::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update positions of stored items.
 * \param elapsed_time Elasped time since the last progress.
 */
void bear::universe::physic_rules::progress_positions( time_type elapsed_time )
{
  item_list::iterator it;

  for(it=m_mobile_items.begin(); it!=m_mobile_items.end(); ++it)
    (*it)->move( elapsed_time );

  for(it=m_global_items.begin(); it!=m_global_items.end(); ++it)
    (*it)->move( elapsed_time );
} // physic_rules::progress_positions()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the links forces between items.
 */
void bear::universe::physic_rules::apply_links() const
{
  claw::avl<base_link*> links;
  claw::avl<base_link*>::const_iterator it;
  item_list::const_iterator item;

  for (item=m_mobile_items.begin(); item!=m_mobile_items.end(); ++item)
    links.insert( (*item)->links_begin(), (*item)->links_end() );

  for (item=m_static_items.begin(); item!=m_static_items.end(); ++item)
    links.insert( (*item)->links_begin(), (*item)->links_end() );

  for (item=m_global_items.begin(); item!=m_global_items.end(); ++item)
    links.insert( (*item)->links_begin(), (*item)->links_end() );

  for( it=links.begin(); it!=links.end(); ++it )
    (*it)->adjust();
} // physic_rules::apply_links()
