/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file text_input.cpp
 * \brief Implementation of the bear::gui::text_input class.
 * \author Julien Jorge
 */
#include "gui/text_input.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param owner The component containing this component.
 * \param f The font used to draw the text.
 * \param cursor_color The color of the cursor.
 * \remark \a f will be deleted in the destructor.
 */
bear::gui::text_input::text_input( visual_component* owner, font_type f,
                             claw::graphic::pixel32 cursor_color )
  : visual_component(owner), m_cursor(0), m_first(0),
    m_last(0), m_line_length(0)
{
  m_static_text = new static_text(this, f);
  create_cursor( cursor_color );
} // text_input::text_input()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::gui::text_input::~text_input()
{
  delete m_cursor_sprite;
  delete m_cursor_image;
} // text_input::~text_input()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the text of the component.
 * \param text The new text.
 */
void bear::gui::text_input::set_text( const std::string& text )
{
  m_text = text;
  m_cursor = m_last = m_text.size();
  m_first = m_last - std::min( m_text.size(), m_line_length-1 );

  adjust_visible_part_of_text();
} // text_input::set_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the text of the component.
 */
const std::string& bear::gui::text_input::get_text() const
{
  return m_text;
} // text_input::get_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Insert a key in the text.
 * \param key The pressed key.
 */
bool bear::gui::text_input::on_key_press( input::keyboard::key_code key )
{
  bool result = true;

  if ( input::keyboard::is_symbol(key) )
    insert_character( input::keyboard::code_to_char(key) );
  else if ( !special_code( key ) )
    result = false;

  if (result)
    adjust_visible_part_of_text();

  return result;
} // text_input::on_key_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Force the height to be a little higher that the font size.
 */
void bear::gui::text_input::on_resized()
{
  const unsigned int default_height =
    m_static_text->get_font()->get_size().y + m_cursor_sprite->height();

  set_size( width(), default_height );
  m_static_text->set_size( width(), m_static_text->get_font()->get_size().y );

  m_line_length =
    m_static_text->width() / m_static_text->get_font()->get_size().x;
} // text_input::on_resized()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw the component on a screen.
 * \param screen The screen on which to draw the component.
 * \param pos The position of the component in the screen.
 */
void bear::gui::text_input::display
( visual::screen& screen,
  const claw::math::coordinate_2d<unsigned int>& pos ) const
{
  claw::math::coordinate_2d<unsigned int> sprite_pos(pos);

  sprite_pos.x +=
    (m_cursor - m_first) * m_static_text->get_font()->get_size().x;
  sprite_pos.y += height() - m_cursor_sprite->height();

  screen.render( sprite_pos, *m_cursor_sprite );
} // text_input::display()

/*----------------------------------------------------------------------------*/
/**
 * \brief Insert a character at cursor position.
 * \param key The character to insert.
 */
void bear::gui::text_input::insert_character( char key )
{
  m_text.insert( m_cursor, 1, key );

  if ( m_text.size() < m_line_length )
    ++m_last;

  move_right();
} // text_input::insert_character()

/*----------------------------------------------------------------------------*/
/**
 * \brief Process a special key code.
 * \param key The code of the key.
 * \return True if the code has been processed.
 */
bool bear::gui::text_input::special_code( input::keyboard::key_code key )
{
  bool result = true;

  if( key == input::keyboard::kc_delete )
    {
      if (m_cursor < m_text.size() )
        {
          m_text.erase(m_cursor, 1);

          if (m_last == m_text.size() + 1)
            --m_last;
        }
    }
  else if ( key == input::keyboard::kc_backspace )
    {
      if (m_cursor > 0)
        {
          m_text.erase(m_cursor - 1, 1);

          if (m_last == m_text.size())
            --m_last;

          move_left();
        }
    }
  else if ( key == input::keyboard::kc_left )
    move_left();
  else if ( key == input::keyboard::kc_right )
    move_right();
  else if ( key == input::keyboard::kc_home )
    {
      m_cursor = 0;
      adjust_text_by_left();
    }
  else if ( key == input::keyboard::kc_end )
    {
      m_cursor = m_text.size();
      adjust_text_by_right();
    }
  else
    result = false;

  return result;
} // text_input::special_code()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite of the cursor.
 * \param cursor_color The color of the cursor.
 */
void bear::gui::text_input::create_cursor( claw::graphic::pixel32 cursor_color )
{
  /* image size must be a power of two. */
  unsigned int x = 2;
  const unsigned int w = m_static_text->get_font()->get_size().x;

  while (x < w)
    x *= 2;

  claw::graphic::image image(x, 4);

  for (unsigned int y=0; y!=image.height(); ++y)
    for (x=0; x!=w; ++x)
      image[y][x] = cursor_color;

  claw::math::rectangle<unsigned int> clip_rect( 0, 0, w, image.height() );

  m_cursor_image = new visual::image( image );
  m_cursor_sprite = new visual::sprite( *m_cursor_image, clip_rect );
} // text_input::create_cursor()

/*----------------------------------------------------------------------------*/
/**
 * \brief Move the cursor one character to the left.
 */
void bear::gui::text_input::move_left()
{
  if (m_cursor > 0)
    {
      --m_cursor;

      adjust_text_by_left();
    }
} // text_input::move_left()

/*----------------------------------------------------------------------------*/
/**
 * \brief Move the cursor one character to the right.
 */
void bear::gui::text_input::move_right()
{
  if (m_cursor < m_text.size())
    {
      ++m_cursor;

      adjust_text_by_right();
    }
} // text_input::move_right()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the part of the visible text when the cursor is before the
 *        begining.
 */
void bear::gui::text_input::adjust_text_by_left()
{
  if (m_cursor < m_first)
    {
      m_first = m_cursor;
      m_last = m_first + std::min( m_text.size() - m_first, m_line_length-1 );
    }
} // text_input::adjust_text_by_left()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the part of the visible text when the cursor is after the end.
 */
void bear::gui::text_input::adjust_text_by_right()
{
  if (m_cursor > m_last)
    {
      m_last = m_cursor;
      m_first = m_last - std::min( m_text.size(), m_line_length-1 );
    }
} // text_input::adjust_text_by_right()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the visible text according to cursor's position.
 */
void bear::gui::text_input::adjust_visible_part_of_text()
{
  m_static_text->set_text( std::string(m_text, m_first, m_last - m_first) );
} // text_input::adjust_visible_part_of_text()
