/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file player.hpp
 * \brief The class describing the players.
 * \author Julien Jorge
 */
#ifndef __ENGINE_PLAYER_HPP__
#define __ENGINE_PLAYER_HPP__

#include "engine/class_export.hpp"
#include "engine/messageable_item.hpp"
#include "engine/model.hpp"
#include "universe/physical_item_state.hpp"

namespace bear
{
  namespace engine
  {
    /**
     * \brief The class describing the players.
     * \author Julien Jorge
     */
    class ENGINE_EXPORT player:
      public model<messageable_item<base_item> >
    {
    public:
      /** \brief The type of the parent class. */
      typedef model<messageable_item<base_item> > super;

      /** \brief  The minimum gap in y ordonate. */
      static const int s_min_gap_y = -250;

      /** \brief  The maximum gap in y ordonate. */
      static const int s_max_gap_y = 300;

      /** \brief  The gap of the spot in one iteration. */
      static const int s_gap_iteration_spot = 15;

      /**
       * \brief A message to get a pointer on the instance of a player.
       * \author Julien Jorge
       */
      class get_instance_message : public communication::message
      {
      public:
        get_instance_message();

        bool apply_to( communication::messageable& that );

        player* get_instance() const;

      private:
        /** \brief Pointer on the instance of the player. */
        player* m_player_instance;

      }; // class get_instance_message

      /** \brief The type of the codes for the actions. */
      typedef unsigned int action;

    public:
      player();

      bool is_player() const;
      universe::position_type hot_spot() const;
      unsigned int get_index() const;

      void start();
      bool set_u_integer_field( const std::string& name, unsigned int value );
      bool is_valid() const;

      virtual void start_action( action a );
      virtual void do_action( action a );
      virtual void stop_action( action a );

      static std::string player_name( unsigned int player_index );

      void add_spot_gap( const universe::position_type& gap );
      void balance_spot();

    private:
      void set_index( unsigned int index );

    public:
      /** \brief Action code for no action. */
      static const action action_null = 0;

      /** \brief Action code dying. */
      static const action action_die = action_null + 1;

      /** \brief Value of the last action code. */
      static const action last_action_code_value = action_die;

    private:
      /** \brief The index of this player. */
      unsigned int m_index;

      /** \brief Position of the hot spot, relative to the center of mass. */
      universe::position_type m_hot_spot_position;

    }; // class player
  } // namespace engine
} // namespace bear

#endif // __ENGINE_PLAYER_HPP__
