/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file game_description.hpp
 * \brief This class loads and store informations from a game description file.
 * \author Julien Jorge
 */
#ifndef __ENGINE_GAME_DESCRIPTION_HPP__
#define __ENGINE_GAME_DESCRIPTION_HPP__

#include <iostream>
#include <string>
#include <map>
#include <list>

#include <claw/math.hpp>

#include "engine/class_export.hpp"

namespace bear
{
  namespace engine
  {
    /**
     * \brief This class loads and store informations from a game description
     *        file.
     * \author Julien Jorge
     */
    class ENGINE_EXPORT game_description
    {
    public:
      typedef std::map<std::string, std::string> level_files_map;
      typedef std::list<std::string> string_list;

    public:
      game_description( std::istream& f );

      const level_files_map& level_files() const;
      const string_list& playable_levels() const;
      const std::string& start_level() const;
      const std::string& game_name() const;
      const claw::math::coordinate_2d<unsigned int>& camera_size() const;
      double active_area_ratio() const;
      const string_list& resources_path() const;
      const string_list& libraries() const;
      const std::string& base_layout_file_name() const;
      const std::string& language_file() const;

    private:
      void set_default_values();
      void read_file( std::istream& f );
      void get_next_line( std::istream& f, std::string& line ) const;
      void process_line( const std::string& line );

      void set_name( const std::string& value );
      void set_camera_width( const std::string& value );
      void set_camera_height( const std::string& value );
      void set_active_area_ratio( const std::string& value );
      void set_resources_path( const std::string& value );
      void set_library( const std::string& value );
      void set_base_layout_file_name( const std::string& value );
      void set_language_file( const std::string& value );
      void set_start_level( const std::string& value );
      void set_level( const std::string& value );
      void set_playable_level( const std::string& value );

    private:
      /** \brief Files associated to the levels. */
      level_files_map m_level_files;

      /** \brief The names of the playable levels. */
      string_list m_playable_levels;

      /** \brief The name of the first level to load. */
      std::string m_start_level;

      /** \brief The name of the game. */
      std::string m_game_name;

      /** \brief The size of the camera. */
      claw::math::coordinate_2d<unsigned int> m_camera_size;

      /** \brief The ratio of the active area compared to the screen size. */
      double m_active_area_ratio;

      /** \brief The paths to the forder containing the resources. */
      string_list m_resources_path;

      /** \brief The game libraries to link to. */
      string_list m_libraries;

      /** \brief The prefix for controller's layout file. */
      std::string m_base_layout_file_name;

      /** \brief The name of the file containing string resources. */
      std::string m_language_file;

    }; // class game_description
  } // namespace engine
} // namespace bear

#endif // __ENGINE_GAME_DESCRIPTION_HPP__
