/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file controller_layout.hpp
 * \brief This class makes the correspondence between the controllers keys and
 *        the actions of the player.
 * \author Julien Jorge
 */
#ifndef __ENGINE_CONTROLLER_LAYOUT_HPP__
#define __ENGINE_CONTROLLER_LAYOUT_HPP__

#include <map>

#include "engine/class_export.hpp"
#include "engine/player.hpp"
#include "input/keyboard.hpp"
#include "input/joystick_button.hpp"
#include "input/mouse.hpp"

namespace bear
{
  namespace engine
  {
    /**
     * \brief This class makes the correspondence between the controllers keys
     *        and the actions of the player.
     * \author Julien Jorge
     */
    class ENGINE_EXPORT controller_layout
    {
    private:
      /** \brief The type of the map to store the actions associated with the
          keyboard. */
      typedef std::map<input::keyboard::key_code, player::action> keyboard_map;

      /** \brief The type of the map to store the actions associated with the
          joysticks. */
      typedef std::map<input::joystick_button, player::action> joystick_map;

      /** \brief The type of the map to store the actions associated with the
          mouse. */
      typedef std::map<input::mouse::mouse_code, player::action> mouse_map;

    public:
      player::action& operator()( input::keyboard::key_code key );
      player::action operator()( input::keyboard::key_code key ) const;

      player::action& operator()( unsigned int joy,
                                  input::joystick::joy_code key );
      player::action operator()( unsigned int joy,
                                 input::joystick::joy_code key ) const;

      player::action& operator()( input::mouse::mouse_code button );
      player::action operator()( input::mouse::mouse_code button ) const;

      input::keyboard::key_code find_key( player::action a ) const;
      input::joystick_button find_joystick_button( player::action a ) const;
      input::mouse::mouse_code find_mouse( player::action a ) const;

      void escape_action_sequence( std::string& str ) const;

      void load( std::istream& f );
      void save( std::ostream& f ) const;

      void remove_key( input::keyboard::key_code key );
      void remove_joy( unsigned int joy, input::joystick::joy_code key );
      void remove_mouse( input::mouse::mouse_code button );

      bool empty() const;

      static std::string build_joystick_button_name
      ( input::joystick::joy_code button, unsigned int joy_index );

    private:
      unsigned int append_action_string
      ( std::string& result, const std::string& str,
        unsigned int current ) const;
      bool append_action_string( std::string& str, unsigned int action ) const;

    private:
      /** \brief Actions associated with the keyboard. */
      keyboard_map m_keyboard;

      /** \brief Actions associated with the joysticks. */
      joystick_map m_joystick;

      /** \brief Actions associated with the mouse. */
      mouse_map m_mouse;

    }; // class controller_layout
  } // namespace engine
} // namespace bear

#endif // __ENGINE_CONTROLLER_LAYOUT_HPP__
