/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file string_base.cpp
 * \brief Implementation of the bear::engine::string_base class.
 * \author Julien Jorge
 */
#include "engine/string_base.hpp"

#include <claw/string_algorithm.hpp>
#include <claw/logger.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the instance.
 */
bear::engine::string_base& bear::engine::string_base::get_instance()
{
  return super::get_instance();
} // string_base::get_instance()

/*----------------------------------------------------------------------------*/
/**
 * \brief Build the base by reading a stream (text file).
 * \param is The stream to build from.
 */
void bear::engine::string_base::load( std::istream& is )
{
  m_strings.clear();
  std::string line, name, content;

  while ( claw::text::getline( is, line ) )
    if ( line != "" )
      if ( line[0] != '#' )
        {
          std::string::size_type pos = line.find_first_of(':');

          if ( pos != std::string::npos )
            {
              name = line.substr(0, pos);
              content = line.substr(pos+1);

              claw::text::trim(name);
              claw::text::trim(content);

              escape(content);
              m_strings[name] = content;
            }
          else
            claw::logger << claw::log_warning
                         << "missing ':' in string resource '" << line << "'."
                         << claw::lendl;
        }
} // string_base::load()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a string.
 * \param name (in) The name of the string resource. (out) The string.
 */
void bear::engine::string_base::get_string( std::string& name ) const
{
  string_map::const_iterator it = m_strings.find(name);

  if ( it != m_strings.end() )
    name = it->second;
} // string_base::get_string()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert the escaped "\n", "\t", etc. substrings of a string into their
 *        equivalent in the C language.
 * \param str (in/out) The string to convert.
 */
void bear::engine::string_base::escape( std::string& str ) const
{
  std::string result;
  unsigned int ref = 0;
  unsigned int prev = 0;
  unsigned int current = 1;

  while ( current < str.size() )
    if ( str[prev] == '\\' )
      {
        switch( str[current] )
          {
          case 'n' : result += str.substr(ref, prev - ref) + '\n'; break;
          case 't' : result += str.substr(ref, prev - ref) + '\t'; break;
          case '\\' : result += str.substr(ref, prev - ref) + '\\'; break;
          default : result += str.substr(ref, current - ref + 1);
          }
        ref = current + 1;
        prev = ref;
        current = prev + 1;
      }
    else
      {
        ++prev;
        ++current;
      }

  if ( ref < str.size() )
    result += str.substr(ref);

  str = result;
} // string_base::escape()
