/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file player.cpp
 * \brief Implementation of the bear::engine::player class.
 * \author Julien Jorge
 */
#include "engine/player.hpp"

#include <sstream>
#include "engine/camera.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::engine::player::get_instance_message::get_instance_message()
  : m_player_instance(NULL)
{

} // get_instance_message::get_instance_message()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the message to a player.
 * \param that The player to apply the message to.
 */
bool bear::engine::player::get_instance_message::apply_to
( communication::messageable& that )
{
  CLAW_PRECOND( dynamic_cast<player*>(&that) != NULL );

  m_player_instance = (player*)&that;
  return true;
} // get_instance_message::apply_to()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the instance of the player.
 */
bear::engine::player*
bear::engine::player::get_instance_message::get_instance() const
{
  return m_player_instance;
} // get_instance_message::get_instance()




/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param name The name of the player.
 */
bear::engine::player::player()
  : m_index(0), m_hot_spot_position(0,0)
{
  set_mass(0.1);
} // player::player()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that this is a player.
 */
bool bear::engine::player::is_player() const
{
  return true;
} // player::is_player()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the coordinates of the point to target to (for camera or active
 *        region).
 */
bear::universe::position_type bear::engine::player::hot_spot() const
{
  return get_center_of_mass() + m_hot_spot_position;
} // player::hot_spot()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the index of this player.
 */
unsigned int bear::engine::player::get_index() const
{
  return m_index;
} // player::get_index()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialise the item.
 */
void bear::engine::player::start()
{
  super::start();
  set_model( "model/" + get_name() + ".cm" );
} // player::start()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type <unsigned integer>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::engine::player::set_u_integer_field
( const std::string& name, unsigned int value )
{
  bool result = true;

  if (name == "index")
    set_index(value);
  else
    result = super::set_u_integer_field(name, value);

  return result;
} // player::set_u_integer_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is correctly initialized.
 */
bool bear::engine::player::is_valid() const
{
  return (m_index != 0) && super::is_valid();
} // player::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell to the player to start an action.
 * \param a The action to start.
 */
void bear::engine::player::start_action( action a )
{
  claw::logger << claw::log_warning << "Action ignored (start): " << a
               << claw::lendl;
} // player::start_action()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell to the player to do an action.
 * \param a The action to do.
 */
void bear::engine::player::do_action( action a )
{
  claw::logger << claw::log_warning << "Action ignored: " << a << claw::lendl;
} // player::do_action()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell to the player to stop an action.
 * \param a The action to stop.
 */
void bear::engine::player::stop_action( action a )
{
  claw::logger << claw::log_warning << "Action ignored (stop): " << a
               << claw::lendl;
} // player::stop_action()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the name of the player associated to an index.
 * \param player_index The index of the player.
 */
std::string bear::engine::player::player_name( unsigned int player_index )
{
  std::ostringstream oss;
  oss << "player_" << player_index;

  return oss.str();
} // player::player_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the index of this player.
 * \param index The index of the player.
 */
void bear::engine::player::set_index( unsigned int index )
{
  m_index = index;
  set_string_field( "name", player_name(index) );
} // player::set_index()

/*----------------------------------------------------------------------------*/
/**
 * \brief Change the gap of the spot.
 * \param gap The vector of gap.
 */
void bear::engine::player::add_spot_gap( const universe::position_type& gap )
{
  m_hot_spot_position += gap;

  if ( m_hot_spot_position.y > s_max_gap_y )
    m_hot_spot_position.y = s_max_gap_y;
  else if ( m_hot_spot_position.y < s_min_gap_y )
    m_hot_spot_position.y = s_min_gap_y;
} // player::add_spot_gap()

/*----------------------------------------------------------------------------*/
/**
 * \brief Balance the gap of the spot.
 */
void bear::engine::player::balance_spot()
{
  if ( m_hot_spot_position.y > s_gap_iteration_spot)
    m_hot_spot_position.y -=  s_gap_iteration_spot;
  else if ( m_hot_spot_position.y > - s_gap_iteration_spot)
    m_hot_spot_position.y = 0;
  else
    m_hot_spot_position.y +=  s_gap_iteration_spot;
} // player::balance_spot()
