/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file level_globals.cpp
 * \brief Implementation of the bear::engine::level_globals class.
 * \author Julien Jorge
 */
#include "engine/level_globals.hpp"

#include <sstream>
#include <cassert>
#include <claw/logger.hpp>
#include "engine/resource_pool.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Load an image.
 * \param file_name The name of the file to load the image from.
 */
const bear::visual::image&
bear::engine::level_globals::load_image( const std::string& file_name )
{
  if ( !m_image_manager.exists(file_name) )
    {
      claw::logger << claw::log_verbose << "loading image '" << file_name
                   << "'." << claw::lendl;

      std::stringstream f;
      resource_pool::get_instance().get_file(file_name, f);

      if (f)
        m_image_manager.load_image(file_name, f);
      else
        claw::logger << claw::log_error << "can not open file '" << file_name
                     << "'." << claw::lendl;
    }

  return m_image_manager.get_image(file_name);
} // level_globals::load_image()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load a sound.
 * \param file_name The name of the file to load the sound from.
 */
bear::audio::sound&
bear::engine::level_globals::load_sound( const std::string& file_name )
{
  if ( !m_sound_manager.sound_exists(file_name) )
    {
      claw::logger << claw::log_verbose << "loading sound '" << file_name
                   << "'." << claw::lendl;

      std::stringstream f;
      resource_pool::get_instance().get_file(file_name, f);

      if (f)
        m_sound_manager.load_sound(file_name, f);
      else
        claw::logger << claw::log_error << "can not open file '" << file_name
                     << "'." << claw::lendl;
    }

  return m_sound_manager.get_sound(file_name);
} // level_globals::load_sound()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load a music.
 * \param file_name The name of the file to load the music from.
 */
bear::audio::music&
bear::engine::level_globals::load_music( const std::string& file_name )
{
  if ( !m_sound_manager.music_exists(file_name) )
    {
      claw::logger << claw::log_verbose << "loading music '" << file_name
                   << "'." << claw::lendl;

      std::stringstream f;
      resource_pool::get_instance().get_file(file_name, f);

      if (f)
        m_sound_manager.load_music(file_name, f);
      else
        claw::logger << claw::log_error << "can not open file '" << file_name
                     << "'." << claw::lendl;
    }

  return m_sound_manager.get_music(file_name);
} // level_globals::load_music()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an image.
 * \param name The name of the image to get.
 * \pre There is an image named \a name.
 */
const bear::visual::image&
bear::engine::level_globals::get_image( const std::string& name )
{
  if ( !m_image_manager.exists(name) )
    load_image(name);

  return m_image_manager.get_image(name);
} // level_globals::get_image()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a sound.
 * \param name The name of the sound to get.
 * \pre There is a sound named \a name.
 */
bear::audio::sound&
bear::engine::level_globals::get_sound( const std::string& name )
{
  if ( !m_sound_manager.sound_exists(name) )
    load_sound(name);

  return m_sound_manager.get_sound(name);
} // level_globals::get_sound()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a music.
 * \param name The name of the music to get.
 * \pre There is a music named \a name.
 */
bear::audio::music&
bear::engine::level_globals::get_music( const std::string& name )
{
  if ( !m_sound_manager.music_exists(name) )
    load_music(name);

  return m_sound_manager.get_music(name);
} // level_globals::get_music()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if there is an image with a given name.
 * \param name The name of the image to check.
 */
bool bear::engine::level_globals::image_exists( const std::string& name ) const
{
  return m_image_manager.exists(name)
    || resource_pool::get_instance().exists(name);
} // level_globals::image_exists()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if there is a sound with a given name.
 * \param name The name of the sound to check.
 */
bool bear::engine::level_globals::sound_exists( const std::string& name ) const
{
  return m_sound_manager.sound_exists(name)
    || resource_pool::get_instance().exists(name);
} // level_globals::sound_exists()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if there is an music with a given name.
 * \param name The name of the music to check.
 */
bool bear::engine::level_globals::music_exists( const std::string& name ) const
{
  return m_sound_manager.music_exists(name)
    || resource_pool::get_instance().exists(name);
} // level_globals::music_exists()

/*----------------------------------------------------------------------------*/
/**
 * \brief Register an item in the post office.
 * \param item The item to register.
 */
void bear::engine::level_globals::register_item
( communication::messageable& item )
{
  m_post_office.register_item( &item );
} // level_globals::register_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Release an item from the post office.
 * \param item The item to release.
 */
void bear::engine::level_globals::release_item
( communication::messageable& item )
{
  m_post_office.release_item( &item );
} // level_globals::release_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Send a message to an item via the post office.
 * \param name The name of the item to contact.
 * \param msg The message to send to this item.
 */
bool bear::engine::level_globals::send_message
( const std::string& target, communication::message& msg )
{
  return m_post_office.send_message( target, msg );
} // level_globals::send_message()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the position of the ears.
 * \param position The new position.
 */
void bear::engine::level_globals::set_ears_position
( const claw::math::coordinate_2d<int>& position )
{
  m_sound_manager.set_ears_position(position);
} // level_globals::set_ears_position()
