/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file item_with_decoration.cpp
 * \brief Implementation of the bear::engine::item_with_decoration class.
 * \author Sebastien Angibaud
 */

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type visual::sprite.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 *
 * Valid values for the \a name and \a value parameters are :
 *  - "sprite", visual::sprite*
 *  - anything supported by the parent class
 */
template<class Base>
bool bear::engine::item_with_decoration<Base>::set_sprite_field
( const std::string& name, visual::sprite* value )
{
  bool ok = true;

  if (name == "sprite")
    {
      m_sprite = value;

      if ( (this->get_size().x == 0) && (this->get_size().y == 0) )
        this->set_size( m_sprite->width(), m_sprite->height() );
	
      if ( m_animation != NULL )
	{
	delete m_animation;
	m_animation = NULL;
	} 
    }
  else
    ok = super::set_sprite_field(name, value);

  return ok;
} // item_with_decoration::set_sprite_field()


/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type visual::animation.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 *
 * Valid values for the \a name and \a value parameters are :
 *  - "animation", visual::animation*
 *  - anything supported by the parent class
 */
template<class Base>
bool bear::engine::item_with_decoration<Base>::set_animation_field
( const std::string& name, visual::animation* value )
{
  bool ok = true;

  if (name == "animation")
    {
      m_animation = value;
	
      if ( (this->get_size().x == 0) && (this->get_size().y == 0) )
        this->set_size( m_animation->get_max_size() );

      if ( m_sprite != NULL )
	{
	delete m_sprite;
	m_sprite = NULL;
	} 
    }
  else
    ok = super::set_animation_field(name, value);

  return ok;
} // item_with_decoration::set_animation_field()


/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is correctly initialized.
 */
template<class Base>
bool bear::engine::item_with_decoration<Base>::is_valid() const
{
  return ( (m_sprite!=NULL) || (m_animation!=NULL) ) && super::is_valid();
} // item_with_decoration::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the sprite representing the item.
 * \param visuals (out) The sprites of the item, and their positions.
 */
template<class Base>
void bear::engine::item_with_decoration<Base>::get_visual
( std::list<scene_visual>& visuals ) const
{
  if (m_sprite != NULL)
	visuals.push_front
    ( scene_visual( this->get_position() + this->get_gap(), 
                    *m_sprite, this->get_angle(), this->get_z_position() ) );

  if (m_animation != NULL)
	visuals.push_front
    ( scene_visual( this->get_position() + this->get_gap(), 
                    m_animation->get_sprite(),
                    this->get_angle(), this->get_z_position() ) );	
} // item_with_decoration::get_visual()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 * \param elapsed_time Elapsed time since the last call.
 */
template<class Base>
void bear::engine::item_with_decoration<Base>::progress
( universe::time_type elapsed_time )
{
  super::progress(elapsed_time);

  if ( m_animation != NULL ) 
    m_animation->next();
} // item_with_decoration::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do post creation actions.
 */
template<class Base>
void bear::engine::item_with_decoration<Base>::start()
{
  super::start();
		
  if ( m_animation != NULL )
    m_animation->reset();
  
} // item_with_decoration::start()
