/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file input_layout.cpp
 * \brief Implementation of the bear::engine::input_layout class.
 * \author Julien Jorge
 */
#include "engine/input_layout.hpp"
#include "engine/input_reader/input_local_reader.hpp"
#include "engine/input_reader/input_local_to_network_reader.hpp"
#include "engine/input_reader/input_network_reader.hpp"
#include "engine/input_reader/input_null_reader.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param rules The rules of the current game.
 */
bear::engine::input_layout::input_layout( const game_rules& rules )
  : m_input_player_1(NULL), m_input_player_2(NULL)
{
  create_input_player_1( rules );
  create_input_player_2( rules );
} // input_layout::input_layout()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::engine::input_layout::~input_layout()
{
  release_controllers();
} // input_layout::~input_layout()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update inputs.
 */
void bear::engine::input_layout::update()
{
  CLAW_PRECOND( m_input_player_1 );
  CLAW_PRECOND( m_input_player_2 );

  m_input_player_1->update();
  m_input_player_2->update();
} // input_layout::update()

/*----------------------------------------------------------------------------*/
/**
 * \brief Release the controllers.
 */
void bear::engine::input_layout::release_controllers()
{
  if ( m_input_player_1 )
    {
      delete m_input_player_1;
      m_input_player_1 = NULL;
    }

  if ( m_input_player_2 )
    {
      delete m_input_player_2;
      m_input_player_2 = NULL;
    }
} // input_layout::release_controllers()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the input reader for the first player, according to the game's
 *        rules.
 * \param rules The rules of the current game.
 */
void
bear::engine::input_layout::create_input_player_1( const game_rules& rules )
{
  if ( rules.get_mode() == game_rules::network_local_second )
    m_input_player_1 =
      new input_network_reader( 1, rules.get_distant_address(),
                                rules.get_distant_port() );
  else
    {
      input_base_local_reader* p;

      if ( rules.get_mode() == game_rules::local )
        p = new input_local_reader( 1, rules.get_layout(1) );
      else
        p = new input_local_to_network_reader( 1, rules.get_layout(1),
                                               rules.get_distant_address(),
                                               rules.get_distant_port() );

      m_input_player_1 = p;
    }
} // input_layout::create_input_player_1()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the input reader for the second player, according to the game's
 *        rules.
 * \param rules The rules of the current game.
 */
void
bear::engine::input_layout::create_input_player_2( const game_rules& rules )
{
  if ( rules.get_number_of_players() == 1 )
    m_input_player_2 = new input_null_reader(2);
  else if ( rules.get_mode() == game_rules::network_local_first )
    m_input_player_2 =
      new input_network_reader( 2, rules.get_distant_address(),
                                rules.get_distant_port() );
  else
    {
      input_base_local_reader* p;

      if ( rules.get_mode() == game_rules::local )
        p = new input_local_reader( 2, rules.get_layout(2) );
      else
        p =
          new input_local_to_network_reader( 2, rules.get_layout(2),
                                             rules.get_distant_address(),
                                             rules.get_distant_port() );

      m_input_player_2 = p;
    }
} // input_layout::create_input_player_2()

