/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file game_rules.hpp
 * \brief Implementation of the bear::engine::game_rules class.
 * \author Julien Jorge
 */
#include "engine/game_rules.hpp"

#include "engine/controller_layout.hpp"

#include <claw/assert.hpp>
#include <claw/logger.hpp>
#include <sstream>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor. Create a single player game.
 * \param base_layout_file_name The base name for controller's layout file. We
 *        will add to its name _i, where i is the controller index.
 */
bear::engine::game_rules::game_rules( const std::string& base_layout_file_name )
  : m_base_layout_file_name(base_layout_file_name), m_number_of_players(1),
    m_game_mode(local), m_layout(2), m_distant_address("127.0.0.1"),
    m_distant_port(30000)
{
  load_controller_layout(1);
  load_controller_layout(2);
} // game_rules::game_rules()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the number of players.
 */
unsigned char bear::engine::game_rules::get_number_of_players() const
{
  return m_number_of_players;
} // game_rules::get_number_of_players()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the number of players.
 * \param n The number of players.
 * \pre 1 <= \a n <= 2
 */
void bear::engine::game_rules::set_number_of_players(unsigned char n)
{
  CLAW_PRECOND( (n==1) || (n==2) );
  m_number_of_players = n;
} // game_rules::set_number_of_players()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the running mode of the game.
 */
bear::engine::game_rules::mode bear::engine::game_rules::get_mode() const
{
  return m_game_mode;
} // game_rules::get_mode()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the running mode of the game.
 * \param m The new mode.
 * \pre ((\a m == network_local_first) || (\a m == network_local_second))
 *      implies (get_number_of_players() == 2)
 */
void bear::engine::game_rules::set_mode( mode m )
{
  CLAW_PRECOND( !( (m != local) && (get_number_of_players() == 1) ) );
  m_game_mode = m;
} // game_rules::set_mode()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the layout of the controls for a player.
 * \param i Index of the player.
 * \pre 1 <= \a i <= 2
 */
const bear::engine::controller_layout&
bear::engine::game_rules::get_layout(unsigned char i) const
{
  CLAW_PRECOND( (i==1) || (i==2) );
  return m_layout[i-1];
} // game_rules::get_layout()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the layout of the controls for a player.
 * \param i Index of the player.
 * \param layout The new layout.
 * \pre 1 <= \a i <= 2
 */
void bear::engine::game_rules::set_layout
( unsigned char i, const controller_layout& layout )
{
  CLAW_PRECOND( (i==1) || (i==2) );
  m_layout[i-1] = layout;
} // game_rules::set_layout()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the address of a distant game.
 * \pre get_mode() != local
 */
const std::string& bear::engine::game_rules::get_distant_address() const
{
  CLAW_PRECOND( get_mode() != local );
  return m_distant_address;
} // game_rules::get_distant_address()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the address of a distant game.
 * \param address The new address.
 * \pre get_mode() != local
 */
void bear::engine::game_rules::set_distant_address( const std::string& address )
{
  CLAW_PRECOND( get_mode() != local );
  m_distant_address = address;
} // game_rules::set_distant_address()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the port of a distant game.
 * \pre get_mode() != local
 */
unsigned int bear::engine::game_rules::get_distant_port() const
{
  CLAW_PRECOND( get_mode() != local );
  return m_distant_port;
} // game_rules::get_distant_port()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the port of a distant game.
 * \param port The new port.
 * \pre get_mode() != local
 */
void bear::engine::game_rules::set_distant_port(unsigned int port)
{
  CLAW_PRECOND( get_mode() != local );
  m_distant_port = port;
} // game_rules::set_distant_port()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the layout of the controls for a player.
 * \param i Index of the player.
 * \pre 1 <= \a i <= 2
 */
void bear::engine::game_rules::save_controller_layout( unsigned char i ) const
{
  CLAW_PRECOND( (i==1) || (i==2) );

  std::ostringstream oss;
  oss << m_base_layout_file_name << "_" << (int)i;

  std::ofstream f( oss.str().c_str() );

  if ( !f )
    claw::logger << claw::log_warning << "Can't open controller's layout file '"
		 << oss.str() << "'." << claw::lendl;
  else
    {
      m_layout[i-1].save(f);
      f.close();
    }
} // game_rules::save_controller_layout()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the layout of the controls for a player.
 * \param i Index of the player.
 * \pre 1 <= \a i <= 2
 */
void bear::engine::game_rules::load_controller_layout( unsigned char i )
{
  CLAW_PRECOND( (i==1) || (i==2) );

  std::ostringstream oss;
  oss << m_base_layout_file_name << "_" << (int)i;

  std::ifstream f( oss.str().c_str() );

  if ( !f )
    claw::logger << claw::log_warning << "Can't open controller's layout file '"
		 << oss.str() << "'." << claw::lendl;
  else
    {
      m_layout[i-1].load(f);
      f.close();
    }
} // game_rules::load_controller_layout()

