/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file game_description.hpp
 * \brief Implementation of the bear::engine::game_description class.
 * \author Julien Jorge
 */
#include <string>
#include <algorithm>

#include "engine/game_description.hpp"

#include <claw/assert.hpp>
#include <claw/logger.hpp>
#include <claw/string_algorithm.hpp>

#include "engine/game.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param The file from which we read the secription.
 */
bear::engine::game_description::game_description( std::istream& f )
{
  set_default_values();
  read_file(f);
} // game_description()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the files associated to the levels.
 */
const bear::engine::game_description::level_files_map&
bear::engine::game_description::level_files() const
{
  return m_level_files;
} // game_description::level_files()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the names of the playable levels.
 */
const bear::engine::game_description::string_list&
bear::engine::game_description::playable_levels() const
{
  return m_playable_levels;
} // game_description::playable_levels()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the name of the first level to load.
 */
const std::string& bear::engine::game_description::start_level() const
{
  return m_start_level;
} // game_description::start_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the name of the game.
 */
const std::string& bear::engine::game_description::game_name() const
{
  return m_game_name;
} // game_description::game_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the size of the camera.
 */
const claw::math::coordinate_2d<unsigned int>&
bear::engine::game_description::camera_size() const
{
  return m_camera_size;
} // game_description::camera_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the ratio of the active area.
 */
double  bear::engine::game_description::active_area_ratio() const
{
  return m_active_area_ratio;
} // game_description::active_area_ratio()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the paths to the forder containing the resources.
 */
const bear::engine::game_description::string_list&
bear::engine::game_description::resources_path() const
{
  return m_resources_path;
} // game_description::resources_path()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the paths to the libraries to link to.
 */
const bear::engine::game_description::string_list&
bear::engine::game_description::libraries() const
{
  return m_libraries;
} // game_description::libraries()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the prefix for controller's layout file.
 */
const std::string& bear::engine::game_description::base_layout_file_name() const
{
  return m_base_layout_file_name;
} // game_description::base_layout_file_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the name of the file containing string resources.
 */
const std::string& bear::engine::game_description::language_file() const
{
  return m_language_file;
} // game_description::language_file()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the fields to their default values.
 */
void bear::engine::game_description::set_default_values()
{
  m_start_level = "start_level";
  m_game_name = "Unnamed game";
  m_camera_size.set(640, 480);
  m_active_area_ratio = 1;
  m_base_layout_file_name = "controller_layout";
} // game_description::set_default_values()

/*----------------------------------------------------------------------------*/
/**
 * \brief Read the content of the description file.
 * \param f The file from which we read.
 */
void bear::engine::game_description::read_file( std::istream& f )
{
  std::string line;

  get_next_line( f, line );

  while(line != "")
    {
      process_line( line );
      get_next_line( f, line );
    }
} // game_description::read_file()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the next not commented and not empty line from the file.
 * \param f The file from which we read.
 */
void bear::engine::game_description::get_next_line
( std::istream& f, std::string& line ) const
{
  bool stop = false;

  while ( !stop )
    {
      line = "";

      if ( claw::text::getline( f, line ) )
        {
          claw::text::trim( line, " \t" );

          if ( line!="" )
            stop = (line[0] != '#');
        }
      else
        stop = true;
    }
} // game_description::get_next_line()

/*----------------------------------------------------------------------------*/
/**
 * \brief Process the content of a line.
 * \param line The line to process.
 */
void bear::engine::game_description::process_line( const std::string& line )
{
  CLAW_PRECOND( line != "" );
  std::string::size_type pos = line.find_first_of('=');

  if ( pos != std::string::npos )
    {
      std::string key = line.substr(0, pos);
      std::string value = line.substr(pos+1, line.length() - pos - 1);

      claw::text::trim( key, " \t" );
      claw::text::trim( value, " \t" );

      std::transform( key.begin(), key.end(), key.begin(), static_cast<int (*)(int)>(tolower) );

      if ( key == "name" )
        set_name( value );
      else if ( key == "camera_width" )
        set_camera_width( value );
      else if ( key == "camera_height" )
        set_camera_height( value );
      else if ( key == "active_area_ratio" )
        set_active_area_ratio( value );
      else if ( key == "resource_path" )
        set_resources_path( value );
      else if ( key == "library" )
        set_library( value );
      else if ( key == "base_layout_file_name" )
        set_base_layout_file_name( value );
      else if ( key == "language_file" )
        set_language_file( value );
      else if ( key == "start_level" )
        set_start_level( value );
      else if ( key == "level" )
        set_level( value );
      else if ( key == "playable_level" )
        set_playable_level( value );
      else
        claw::logger << claw::log_warning << "game_description: Unknow key: '"
                     << key << "'." << claw::lendl;
    }
  else
    claw::logger << claw::log_warning << "game_description: Ignored line:\n"
                 << line << claw::lendl;
} // game_description()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the name of the game.
 * \param value The value of the field.
 */
void bear::engine::game_description::set_name( const std::string& value )
{
  m_game_name = value;
} // game_description::set_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the width of the camera/screen.
 * \param value The value of the field.
 */
void
bear::engine::game_description::set_camera_width( const std::string& value )
{
  if ( claw::text::is_of_type<unsigned int>(value) )
    {
      std::istringstream iss(value);
      iss >> m_camera_size.x;
    }
  else
    claw::logger << claw::log_warning
                 << "game_description:set_camera_width(): Not an unsigned"
                 << " integer: '" << value << "'." << claw::lendl;
} // game_description::set_camera_width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the height of the camera/screen.
 * \param value The value of the field.
 */
void
bear::engine::game_description::set_camera_height( const std::string& value )
{
  if ( claw::text::is_of_type<unsigned int>(value) )
    {
      std::istringstream iss(value);
      iss >> m_camera_size.y;
    }
  else
    claw::logger << claw::log_warning
                 << "game_description:set_camera_height(): Not an unsigned"
                 << " integer: '" << value << "'." << claw::lendl;
} // game_description::set_camera_height()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the ratio of the active_area.
 * \param value The value of the field.
 */
void bear::engine::game_description::set_active_area_ratio
( const std::string& value )
{
  if ( claw::text::is_of_type<double>(value) )
    {
      std::istringstream iss(value);
      iss >> m_active_area_ratio;
    }
  else
    claw::logger << claw::log_warning
                 << "game_description:set_active_area_width(): Not a double: '"
                 << value << "'." << claw::lendl;
} // game_description::set_active_area_width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a path to a folder containing the game resources.
 * \param value The value of the field.
 */
void
bear::engine::game_description::set_resources_path( const std::string& value )
{
  m_resources_path.push_back(value);
} // game_description::set_resources_path()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a library in the libraries_pool.
 * \param value The value of the field.
 */
void bear::engine::game_description::set_library( const std::string& value )
{
  m_libraries.push_back(value);
} // game_description::set_library()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the prefix for controller's layout file.
 * \param value The value of the field.
 */
void bear::engine::game_description::set_base_layout_file_name
( const std::string& value )
{
  m_base_layout_file_name = value;
} // game_description::set_base_layout_file_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the name of the file containing string resources.
 * \param value The value of the field.
 */
void
bear::engine::game_description::set_language_file( const std::string& value )
{
  m_language_file = value;
} // game_description::set_language_file()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the name of the first level.
 * \param value The value of the field.
 */
void bear::engine::game_description::set_start_level( const std::string& value )
{
  m_start_level = value;
} // game_description::set_start_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a level.
 * \param value The value of the field.
 */
void bear::engine::game_description::set_level( const std::string& value )
{
  std::string::size_type pos = value.find_first_of(" \t");

  if ( pos != std::string::npos )
    {
      std::string name = value.substr(0, pos);
      pos = value.find_first_not_of( " \t", pos );
      std::string file = value.substr(pos);

      if ( m_level_files.find(name) == m_level_files.end() )
        m_level_files[name] = file;
      else
        claw::logger << claw::log_warning
                     << "game_description::set_level() : Level '" << name
                     << "' already exists." << claw::lendl;
    }
  else
    claw::logger << claw::log_warning
                 << "game_description::set_level() : Ignored value '" << value
                 << "'." << claw::lendl;
} // game_description::set_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a level in the list of playable levels.
 * \param value The value of the field.
 */
void
bear::engine::game_description::set_playable_level( const std::string& value )
{
  m_playable_levels.push_back(value);
} // game_description::set_playable_level()
