/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file controller_layout.cpp
 * \brief Implementation of the bear::engine::controller_layout class.
 * \author Julien Jorge
 */
#include "engine/controller_layout.hpp"

#include "engine/string_base.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the action associated to a keyboard key.
 * \param key The code of the key.
 */
bear::engine::player::action&
bear::engine::controller_layout::operator()( input::keyboard::key_code key )
{
  return m_keyboard[key];
} // controller_layout::operator()( key_code )

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the action associated to a keyboard key.
 * \param key The code of the key.
 */
bear::engine::player::action
bear::engine::controller_layout::operator()
  ( input::keyboard::key_code key ) const
{
  keyboard_map::const_iterator it;

  it = m_keyboard.find(key);

  if ( it != m_keyboard.end() )
    return it->second;
  else
    return player::action_null;
} // controller_layout::operator()( key_code ) [const]

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the action associated to a key of a joystick.
 * \param joy The index of the joystick.
 * \param key The code of the key.
 */
bear::engine::player::action&
bear::engine::controller_layout::operator()
  ( unsigned int joy, input::joystick::joy_code key )
{
  return m_joystick[ input::joystick_button(joy, key) ];
} // controller_layout::operator()( unsigned int, joy_code )

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the action associated to a key of a joystick.
 * \param joy The index of the joystick.
 * \param key The code of the key.
 */
bear::engine::player::action
bear::engine::controller_layout::operator()
  ( unsigned int joy, input::joystick::joy_code key ) const
{
  joystick_map::const_iterator it;

  it = m_joystick.find( input::joystick_button(joy, key) );

  if ( it != m_joystick.end() )
    return it->second;
  else
    return player::action_null;
} // controller_layout::operator()( unsigned int, joy_code ) [const]

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the action associated to a mouse button.
 * \param button The code of the button.
 */
bear::engine::player::action&
bear::engine::controller_layout::operator()( input::mouse::mouse_code button )
{
  return m_mouse[button];
} // controller_layout::operator()( mouse_code )

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the action associated to a mouse button.
 * \param button The code of the button.
 */
bear::engine::player::action
bear::engine::controller_layout::operator()
  ( input::mouse::mouse_code button ) const
{
  mouse_map::const_iterator it;

  it = m_mouse.find(button);

  if ( it != m_mouse.end() )
    return it->second;
  else
    return player::action_null;
} // controller_layout::operator()( mouse_code ) [const]

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the keyboard key associated with an action.
 * \return input::keyboard::kc_not_a_key if no key is associated with the
 *         action.
 */
bear::input::keyboard::key_code
bear::engine::controller_layout::find_key( player::action a ) const
{
  input::keyboard::key_code result = input::keyboard::kc_not_a_key;

  keyboard_map::const_iterator it;

  for ( it=m_keyboard.begin();
        (it!=m_keyboard.end()) && (result == input::keyboard::kc_not_a_key);
        ++it )
    if ( it->second == a )
      result = it->first;

  return result;
} // controller_layout::find_key()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the joystick key and index associated with an action.
 * \return result.button == input::joystick::jc_invalid if no key is associated
 *         with the action.
 */
bear::input::joystick_button
bear::engine::controller_layout::find_joystick_button( player::action a ) const
{
  input::joystick_button result( 0, input::joystick::jc_invalid );

  joystick_map::const_iterator it;

  for ( it=m_joystick.begin();
        (it!=m_joystick.end()) && (result.button==input::joystick::jc_invalid);
        ++it )
    if ( it->second == a )
      result = it->first;

  return result;
} // controller_layout::find_joystick_button()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the mouse key and index associated with an action.
 * \return result == input::mouse::mc_invalid if no button is associated with
 *         the action.
 */
bear::input::mouse::mouse_code
bear::engine::controller_layout::find_mouse( player::action a ) const
{
  input::mouse::mouse_code result( input::mouse::mc_invalid );

  mouse_map::const_iterator it;

  for ( it=m_mouse.begin();
        (it!=m_mouse.end()) && (result == input::mouse::mc_invalid);
        ++it )
    if ( it->second == a )
      result = it->first;

  return result;
} // controller_layout::find_mouse()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert the escaped action sequence of a string.
 * \param str (in/out) The string to convert.
 */
void bear::engine::controller_layout::escape_action_sequence
( std::string& str ) const
{
  std::string result;
  unsigned int ref = 0;
  unsigned int prev = 0;
  unsigned int current = 1;

  while ( current < str.size() )
    if ( str[prev] == '%' )
      {
        switch( str[current] )
          {
          case 'a' :
            result += str.substr(ref, prev - ref);
            current = append_action_string(result, str, current);
            break;
          default : result += str.substr(ref, current - ref + 1);
          }
        ref = current + 1;
        prev = ref;
        current = prev + 1;
      }
    else
      {
        ++prev;
        ++current;
      }

  if ( ref < str.size() )
    result += str.substr(ref);

  str = result;
} // controller_layout::escape_action_sequence()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the layout from a stream.
 * \param f The stream from which we read.
 */
void bear::engine::controller_layout::load( std::istream& f )
{
  unsigned int n;
  unsigned int joy;
  input::keyboard::key_code key_code;
  input::joystick::joy_code joy_code;
  input::mouse::mouse_code mouse_code;
  int action;

  f >> n;

  for (unsigned int i=0; i!=n; ++i)
    if ( f >> key_code >> action )
      m_keyboard[key_code] = (player::action)action;

  f >> n;

  for (unsigned int i=0; i!=n; ++i)
    if ( f >> joy >> joy_code >> action )
      {
        if ( joy >= input::joystick::number_of_joysticks() )
          claw::logger << claw::log_warning << "Invalid joystick index: "
		       << joy << claw::lendl;

        m_joystick[ input::joystick_button(joy, joy_code) ] =
          (player::action)action;
      }

  f >> n;

  for (unsigned int i=0; i!=n; ++i)
    if ( f >> mouse_code >> action )
      m_mouse[mouse_code] = (player::action)action;
} // controller_layout::load()

/*----------------------------------------------------------------------------*/
/**
 * \brief save the layout in a stream.
 * \param f The stream in which we write.
 */
void bear::engine::controller_layout::save( std::ostream& f ) const
{
  keyboard_map::const_iterator it_k;
  joystick_map::const_iterator it_j;
  mouse_map::const_iterator it_m;

  f << m_keyboard.size() << std::endl;

  for (it_k=m_keyboard.begin(); it_k!=m_keyboard.end(); ++it_k)
    f << it_k->first << " " << it_k->second << std::endl;

  f << m_joystick.size() << std::endl;

  for (it_j=m_joystick.begin(); it_j!=m_joystick.end(); ++it_j)
    f << it_j->first.joystick_index << " " << it_j->first.button << " "
      << it_j->second << std::endl;

  for (it_m=m_mouse.begin(); it_m!=m_mouse.end(); ++it_m)
    f << it_m->first << " " << it_m->second << std::endl;
} // controller_layout::save()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove an action from the controller.
 * \param key The keyboard key of the action to remove.
 */
void
 bear::engine::controller_layout::remove_key( input::keyboard::key_code key )
{
  m_keyboard.erase(key);
} // controller_layout::remove_key()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove an action from the controller.
 * \param joy The joystick index of the action to remove.
 * \param key The button of the action to remove.
 */
void bear::engine::controller_layout::remove_joy
( unsigned int joy, input::joystick::joy_code key )
{
  m_joystick.erase( input::joystick_button(joy, key) );
} // controller_layout::remove_joy()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove an action from the controller.
 * \param button The mouse button of the action to remove.
 */
void
bear::engine::controller_layout::remove_mouse( input::mouse::mouse_code button )
{
  m_mouse.erase(button);
} // controller_layout::remove_mouse()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if no keys are configured.
 */
bool bear::engine::controller_layout::empty() const
{
  return m_joystick.empty() && m_keyboard.empty() && m_mouse.empty();
} // controller_layout::empty()

/*----------------------------------------------------------------------------*/
/**
 * \brief Build a human readable name for a joystick button name.
 * \param button The value of the button.
 * \param joy_index The index of the joytick.
 */
std::string bear::engine::controller_layout::build_joystick_button_name
( input::joystick::joy_code button, unsigned int joy_index )
{
  std::ostringstream oss;

  oss << "joy" << (joy_index + 1) << ":"
      << input::joystick::get_name_of(button);

  std::string result( oss.str() );
  string_base::get_instance().get_string( result );

  return result;
} // controller_layout::build_joystick_button_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert the "\\ai" escape sequence.
 * \param result (in/out) The string to which we append the action.
 * \param str The string to convert.
 * \param current The position of the parameters of the 'a' in the \\a
 *        sequence.
 * \return The position of the last character read.
 */
unsigned int bear::engine::controller_layout::append_action_string
( std::string& result, const std::string& str, unsigned int current ) const
{
  std::string::size_type pos = str.find_first_of(';', current);
  bool ok = false;

  if ( pos != std::string::npos )
    {
      std::istringstream iss( str.substr(current+1, pos - current - 1) );
      unsigned int action;

      if ( iss >> action)
        if ( iss.rdbuf()->in_avail() == 0 )
          if ( append_action_string(result, action) )
            {
              current = pos;
              ok = true;
            }
    }

  if (!ok)
    result += "%a";

  return current;
} // controller_layout::append_action_string()

/*----------------------------------------------------------------------------*/
/**
 * \brief Append to a string the string corresponding to the action of a given
 *        player.
 * \param str (in/out) The string to which we append the action.
 * \param action The action from which we want the string.
 * \param player The player doing the action.
 * \return The position of the next character to read.
 */
bool bear::engine::controller_layout::append_action_string
( std::string& str, unsigned int action ) const
{
  bool result = true;

  input::keyboard::key_code key = find_key( action );
  input::joystick_button joy = find_joystick_button( action );
  input::mouse::mouse_code mouse = find_mouse( action );

  if ( key != input::keyboard::kc_not_a_key )
    {
      std::string text( input::keyboard::get_name_of(key) );
      string_base::get_instance().get_string(text);
      str += text;
    }
  else if ( joy.button != input::joystick::jc_invalid )
    str += build_joystick_button_name(joy.button, joy.joystick_index);
  else if ( mouse != input::mouse::mc_invalid )
    {
      std::string text( input::mouse::get_name_of(mouse) );
      string_base::get_instance().get_string(text);
      str += text;
    }
  else
    result = false;

  return result;
} // controller_layout::append_action_string()
