/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file camera.hpp
 * \brief Description of the camera.
 * \author Sebastien Angibaud
 */
#ifndef __ENGINE_CAMERA_HPP__
#define __ENGINE_CAMERA_HPP__

#include <claw/math.hpp>

#include "engine/class_export.hpp"
#include "engine/pointer_to_player.hpp"

namespace bear
{
  namespace engine
  {
    /**
     * \brief Description of the camera.
     *
     * The camera is the focus area in the world.
     * \author Sebastien Angibaud
     */
    class ENGINE_EXPORT camera:
      public communication::messageable
    {
    public:
      /*----------------------------------------------------------------------*/
      /**
       * \brief The different placement modes for the camera.
       */
      enum placement_mode
        {
          /** \brief Centered on the first player. */
          lock_first_player,

          /** \brief Centered on the second player. */
          lock_second_player,

          /** \brief Centered between the two players. */
          shared,

          /** \brief Don't move. */
          do_nothing
        }; // enum mode

      /*----------------------------------------------------------------------*/
      /**
       * \brief A message that set the placement mode of the camera.
       * \author Julien Jorge
       */
      class ENGINE_EXPORT set_placement_message:
        public communication::message
      {
      public:
        set_placement_message( placement_mode mode );

        bool apply_to( communication::messageable& that );

      private:
        /** \brief The mode to set. */
        const placement_mode m_mode;

      }; // class set_placement_message

      /*----------------------------------------------------------------------*/
      /**
       * \brief A message that get the focus rectangle of the camera.
       * \author Julien Jorge
       */
      class ENGINE_EXPORT msg_get_focus:
        public communication::message
      {
      public:
        bool apply_to( communication::messageable& that );

      public:
        /** \brief The focused rectangme. */
        universe::rectangle_type focus;

      }; // class msg_get_focus

      /*----------------------------------------------------------------------*/
    public:
      camera( const std::string& name,
              const claw::math::coordinate_2d<unsigned int>& size,
              unsigned int max_move_length = 50 );

      void auto_position();

      void initialise_position();
      void set_position( const universe::position_type& position );
      void set_valid_area( const universe::rectangle_type& area );

      const universe::rectangle_type& get_focus() const;
      universe::position_type get_center() const;

      void set_mode( placement_mode mode );

      static const std::string& default_name();

    private:
      void set_first_player();
      void set_second_player();
      void set_shared();

      void adjust_position( const universe::position_type& center_position );
      void adjust_position_x( universe::coordinate_type center_position );
      void adjust_position_y( universe::coordinate_type center_position );

      void stay_valid();

    private:
      /** \brief Default name of the camera. */
      static const std::string s_default_name;

      /** \brief The box (position and size) of the camera in the world. */
      universe::rectangle_type m_box;

      /** \brief The area in which the camera can be set. */
      universe::rectangle_type m_valid_area;

      /** \brief Current placement mode. */
      placement_mode m_placement;

      /** \brief Pointer to the first player. */
      pointer_to_player m_first_player;

      /** \brief Pointer to the second player. */
      pointer_to_player m_second_player;

      /** \brief Maximum length of a camera movement. */
      universe::coordinate_type m_max_move_length;

    }; // class camera
  } // namespace engine
} // namespace bear

#endif // __ENGINE_CAMERA_HPP__
