/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file sdl_sound.hpp
 * \brief A class representing a sound. This class uses the SDL_mixer library.
 * \author Julien Jorge
 */
#ifndef __AUDIO_SDL_SOUND_HPP__
#define __AUDIO_SDL_SOUND_HPP__

#include "audio/sound.hpp"

#include <SDL/SDL_mixer.h>
#include <claw/math.hpp>
#include <iostream>
#include <vector>
#include <list>

#include "audio/class_export.hpp"

namespace bear
{
  namespace audio
  {
    class AUDIO_EXPORT sound_manager;

    /**
     * \brief A class representing a sound.
     */
    class AUDIO_EXPORT sdl_sound : public sound
    {
    private:
      /**
       * \brief Informations stored in the s_playing_channels vector.
       */
      class AUDIO_EXPORT channel_attribute
      {
      public:
        channel_attribute();
        ~channel_attribute();

        void set_sound( const sdl_sound& s );
        const sdl_sound& get_sound() const;

        void set_position( const claw::math::coordinate_2d<int>& p );
        const claw::math::coordinate_2d<int>& get_position() const;

        void clear();
        bool is_empty() const;

      private:
        /** \brief The sound played in this channel. */
        const sdl_sound* m_sound;

        /** \brief The position of the sound. */
        claw::math::coordinate_2d<int> m_position;
      }; // class channel_attribute

    public:
      sdl_sound( std::istream& file, sound_manager& owner );
      ~sdl_sound();

      void play();
      void play( const sound_effect& effect );
      void play( const claw::math::coordinate_2d<int>& position );
      void play( const claw::math::coordinate_2d<int>& position,
                 const sound_effect& effect );

      void stop();

      static bool initialize();
      static void release();

    private:
      static void channel_finished(int channel);
      static void distance_tone_down(int channel, void *stream, int length,
                                     void *position);

      int inside_play( int loops );

      void set_channel_position
      ( int channel, const claw::math::coordinate_2d<int>& position );

      void global_add_channel( int channel );
      void finished( int channel );

    private:
      /** \brief The sound allocated by SDL_mixer. */
      Mix_Chunk* m_sound;

      /** \brief Channels attributed by SDL_mixer. */
      std::list<int> m_channels;

      /** \brief Global vector giving, for a channel, the sound currently
          played. */
      static std::vector<channel_attribute*> s_playing_channels;

      /** \brief Output audio rate. 22050 by default. */
      static unsigned int s_audio_rate;

      /** \brief Output audio format. 16 bits, signed by default.  */
      static unsigned int s_audio_format;

      /** \brief Number of channels. 2 by default. */
      static unsigned int s_audio_channels;

      /** \brief Size of the buffer. 4 ko by default. */
      static unsigned int s_audio_buffers;

      /**
       * \brief Distance from which we can't hear a sound, measured as a
       *        Manhattan distance.
       *
       * \remark This value must be strictly greater than
       *         s_full_volume_distance.
       */
      static unsigned int s_silent_distance;

      /**
       * \brief Distance under which sounds are at maximum volume, measured as a
       *        Manhattan distance.
       *
       * \remark This value must be strictly lower than s_silent_distance.
       */
      static unsigned int s_full_volume_distance;

    }; // class sdl_sound
  } // namespace audio
} // namespace bear

#endif // __AUDIO_SDL_SOUND_HPP__
