/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file sound_manager.cpp
 * \brief Implementation of the sound_manager class.
 * \author Julien Jorge
 */
#include "audio/sound_manager.hpp"

#include "audio/sdl_music.hpp"
#include "audio/sdl_sound.hpp"

#include <claw/assert.hpp>
#include <claw/exception.hpp>
#include <fstream>

/*----------------------------------------------------------------------------*/
bool bear::audio::sound_manager::s_initialized = false;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::audio::sound_manager::sound_manager()
  : m_ears_position(0, 0)
{

} // sound_manager::sound_manager()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::audio::sound_manager::~sound_manager()
{
  clear();
} // sound_manager::~sound_manager()

/*----------------------------------------------------------------------------*/
/**
 * \brief Delete all sounds and musics. If a music is playing, she will be
 *        stopped
 * \post m_sounds.empty == true.
 * \post m_musics.empty == true.
 */
void bear::audio::sound_manager::clear()
{
  std::map<std::string, sound*>::iterator it_s;
  std::map<std::string, music*>::iterator it_m;

  for (it_s=m_sounds.begin(); it_s!=m_sounds.end(); ++it_s)
    delete it_s->second;

  for (it_m=m_musics.begin(); it_m!=m_musics.end(); ++it_m)
    delete it_m->second;

  m_sounds.clear();
  m_musics.clear();
} // sound_manager::clear()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a sound to the cache.
 * \param name The name of the sound.
 * \param file A stream containing the sound.
 * \pre name is not used by another sound.
 */
void bear::audio::sound_manager::load_sound
( const std::string& name, std::istream& file )
{
  CLAW_PRECOND( !sound_exists(name) );

  if (s_initialized)
    m_sounds[name] = new sdl_sound(file, *this);
  else
    m_sounds[name] = new sound(*this);
} // sound_manager::load_sound()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a music to the cache.
 * \param name The name of the music.
 * \param file A stream containing the music.
 * \pre name is not used by another music.
 */
void bear::audio::sound_manager::load_music
( const std::string& name, std::istream& file )
{
  CLAW_PRECOND( !music_exists(name) );

  if (s_initialized)
    m_musics[name] = new sdl_music(file);
  else
    m_musics[name] = new music();
} // sound_manager::load_music()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a sound from the cache.
 * \param name The name of the sound to get.
 * \pre There is an sound called "name".
 */
bear::audio::sound&
bear::audio::sound_manager::get_sound(const std::string& name)
{
  CLAW_PRECOND( sound_exists(name) );

  return *m_sounds[name];
} // sound_manager::play_sound()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a music from the cache.
 * \param name The name of the music to get.
 * \pre There is an music called "name".
 */
bear::audio::music&
bear::audio::sound_manager::get_music(const std::string& name)
{
  CLAW_PRECOND( music_exists(name) );

  return *m_musics[name];
} // sound_manager::play_music()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the position of the ears.
 * \param position The new position.
 */
void bear::audio::sound_manager::set_ears_position
( const claw::math::coordinate_2d<int>& position )
{
  m_ears_position = position;
} // sound_manager::set_ears_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the position of the ears.
 */
const claw::math::coordinate_2d<int>&
bear::audio::sound_manager::get_ears_position() const
{
  return m_ears_position;
} // sound_manager::get_ears_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if a sound is in the cache.
 */
bool bear::audio::sound_manager::sound_exists(const std::string& name) const
{
  return m_sounds.find(name) != m_sounds.end();
} // sound_manager::sound_exists()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if a music is in the cache.
 */
bool bear::audio::sound_manager::music_exists(const std::string& name) const
{
  return m_musics.find(name) != m_musics.end();
} // sound_manager::music_exists()

/*----------------------------------------------------------------------------*/
/**
 * \brief Stop all sounds and musics.
 */
void bear::audio::sound_manager::stop_all()
{
  std::map<std::string, sound*>::iterator it_s;

  for (it_s=m_sounds.begin(); it_s!=m_sounds.end(); ++it_s)
    it_s->second->stop();

  std::map<std::string, music*>::iterator it_m;

  for (it_m=m_musics.begin(); it_m!=m_musics.end(); ++it_m)
    it_m->second->stop();
} // sound_manager::stop_all()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the sound system.
 */
void bear::audio::sound_manager::initialize()
{
  s_initialized = sdl_sound::initialize();
} // sound_manager::initialize()

/*----------------------------------------------------------------------------*/
/**
 * \brief Close the sound system.
 */
void bear::audio::sound_manager::release()
{
  sdl_sound::release();
  s_initialized = false;
} // sound_manager::release()
