/*************************************************************************\
*   Copyright (C) 2009 by Ulf Kreissig                                    *
*   udev@gmx.net                                                          *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
\*************************************************************************/

#include "streamlogger.h"

//--- QT4 ---
#include <QFile>
#include <QString>
#include <QApplication>
#include <QMutex>
#include <QMutexLocker>
#include <QTime>
#include <QThread>

#include <iostream>

#define HEADER_MAXLENGTH	75
#define TABS_WIDTH		3

namespace DStreamLogger
{

struct tLogMsgInfo
{
	uint		iSpacer;
	QMutex		mutex;
};

static tLogMsgInfo	logInfo;
static QString          sLogfileName;

#ifdef ENABLE_DSTREAMLOGGER
inline void
printHeader( QDebug & d, QtMsgType type, const char * fileName, int line, const char * funcinfo )
{
	Q_UNUSED(fileName)

	QString sTime = QTime::currentTime().toString( QLatin1String("hh:mm:ss.zzz") );
	
	// strip the function info down to the base function name
	// note that this throws away the template definitions,
	// the parameter types (overloads) and any const/volatile qualifiers
	QString sFunc;
	if (funcinfo)
	{
# ifdef Q_CC_GNU
            // strip the function info down to the base function name
            // note that this throws away the template definitions,
            // the parameter types (overloads) and any const/volatile qualifiers
            QByteArray info = funcinfo;
            int pos = info.indexOf('(');
            Q_ASSERT_X(pos != -1, "dDebug",
                       "Bug in dDebug(): I don't know how to parse this function name");
            while (info.at(pos - 1) == ' ')
                // that '(' we matched was actually the opening of a function-pointer
                pos = info.indexOf('(', pos + 1);

            info.truncate(pos);
            // gcc 4.1.2 don't put a space between the return type and
            // the function name if the function is in an anonymous namespace
            int index = 1;
            forever {
                index = info.indexOf("<unnamed>::", index);
                if ( index == -1 )
                    break;

                if ( info.at(index-1) != ':' )
                    info.insert(index, ' ');

                index += strlen("<unnamed>::");
            }
            pos = info.lastIndexOf(' ');
            if (pos != -1) {
                int startoftemplate = info.lastIndexOf('<');
                if (startoftemplate != -1 && pos > startoftemplate &&
                    pos < info.lastIndexOf(">::"))
                    // we matched a space inside this function's template definition
                    pos = info.lastIndexOf(' ', startoftemplate);
            }

            if (pos + 1 == info.length())
                // something went wrong, so gracefully bail out
                sFunc = funcinfo;
            else
                sFunc = info.constData() + pos + 1;
#else
            sFunc = funcinfo;
#endif
        }

	QString sHeader = sTime + " ";
	int iFancyOffset = 0;

#ifdef EXTENDED_DEBUG_OUTPUT
	if( sLogfileName.isEmpty() )	// do fancy output only on stdout
	{
		switch (type)
		{
		case QtDebugMsg:	break;
		case QtWarningMsg:	sHeader += "\033[36m"; iFancyOffset = -5;  break;  // cyan text output
		case QtFatalMsg:	sHeader += "\033[33m"; iFancyOffset = -5;  break;  // yellow text output
		case QtCriticalMsg:
		default:		sHeader += "\033[31m"; iFancyOffset = -5;  break;  // red text output
		}
	}
#endif

	switch (type)
	{
	case QtDebugMsg:	sHeader += QLatin1String(" Info     ");		break;
	case QtWarningMsg:	sHeader += QLatin1String(" Warning  ");		break;
	case QtFatalMsg:	sHeader += QLatin1String(" Fatal    ");		break;
	case QtCriticalMsg:
	default:		sHeader += QLatin1String(" Error    ");		break;
	}

	QString sLineNumb;
	if( line >= 0 )
		sLineNumb = QString( " (Line %1): ").arg(line);
	else
		sLineNumb = QLatin1String(": ");

	// leave some space for the linenumber
	sHeader += sFunc.left( HEADER_MAXLENGTH - (sHeader.length() + sLineNumb.length() + 5 + iFancyOffset) );

#ifdef	EXTENDED_DEBUG_OUTPUT
	if( sLogfileName.isEmpty() && type != QtDebugMsg )	// do fancy output only on stdout
	{
		sHeader += QLatin1String("\033[0m");	// switch back to normal output
		iFancyOffset -= 4;
	}
#endif

	sHeader += sLineNumb;
	int iOffset = qMax( 0, (HEADER_MAXLENGTH - ((int)sHeader.length() + iFancyOffset)) );
	if( iOffset > 0 )
		sHeader += QString( iOffset, QChar(' ') );
	for( uint i = 0; i < logInfo.iSpacer; ++i )
	{
		if( i % TABS_WIDTH == 1 )
			sHeader += '.';
		else
			sHeader += ' ';
	}
	d << sHeader.toLatin1().data();
}
#endif

void
setLogfileName( const QString & filename )
{
	sLogfileName = filename;
}

void
msgFileDebug( QtMsgType type, const char * msg )
{
	Q_UNUSED(type);
#ifdef ENABLE_DSTREAMLOGGER
	bool bWrite2File = false;
	if( !sLogfileName.isEmpty() )
	{
		QFile file( sLogfileName );
		if( file.open(QIODevice::WriteOnly|QIODevice::Append|QIODevice::Text) )
		{
			file.write(msg, qstrlen(msg));        // write to stderr
			file.putChar('\n');
			file.close();
			bWrite2File = true;
		}
	}
	if( !bWrite2File )
		std::cout << msg << std::endl;
#else
	Q_UNUSED(msg)
#endif
}

QDebug
debugStream( QtMsgType level, const char * fileName, int line, const char * funcinfo )
{
	QDebug d(level);
#ifndef ENABLE_DSTREAMLOGGER
	Q_UNUSED(level)
	Q_UNUSED(fileName)
	Q_UNUSED(line)
	Q_UNUSED(funcinfo)
#else
	QMutexLocker l( &logInfo.mutex );
	printHeader( d, level, fileName, line, funcinfo );
#endif
	return d;
}

QDebug
startFunctionBlock( QtMsgType level, const char * fileName, int line, const char * funcinfo )
{
	QDebug d(level);
#ifndef ENABLE_DSTREAMLOGGER
	Q_UNUSED(level)
	Q_UNUSED(fileName)
	Q_UNUSED(line)
	Q_UNUSED(funcinfo)
#else
	QMutexLocker l( &logInfo.mutex );
	printHeader( d.nospace(), level, fileName, line, funcinfo );
	logInfo.iSpacer += TABS_WIDTH;
	d.space() << "[function starts]";
#endif
	return d;
}

QDebug
endFunctionBlock( QtMsgType level, const char * fileName, int line, const char * funcinfo )
{
	QDebug d(level);
#ifndef ENABLE_DSTREAMLOGGER
	Q_UNUSED(level)
	Q_UNUSED(fileName)
	Q_UNUSED(line)
	Q_UNUSED(funcinfo)
#else
	QMutexLocker l( &logInfo.mutex );
	logInfo.iSpacer = (logInfo.iSpacer >= TABS_WIDTH ? logInfo.iSpacer - TABS_WIDTH : 0 );
	printHeader( d.nospace(), level, fileName, line, funcinfo );
	d.space() << "[function ends]";
#endif
	return d;
}

}
