/***********************************************************************************
* Window List: Plasmoid to show list of opened windows.
* Copyright (C) 2009 Michal Dutkiewicz aka Emdek <emdeck@gmail.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

#include "WindowList.h"

#include <Plasma/Theme>
#include <Plasma/IconWidget>
#include <Plasma/Containment>
#include <Plasma/ToolTipManager>

#include <KMenu>
#include <KLocale>
#include <KIconLoader>
#include <NETRootInfo>
#include <KWindowSystem>
#include <KConfigDialog>
#include <taskmanager/taskitem.h>
#include <taskmanager/taskactions.h>
#include <taskmanager/taskmanager.h>
#include <taskmanager/groupmanager.h>

K_EXPORT_PLASMA_APPLET(windowslist, WindowList)

WindowList::WindowList(QObject *parent, const QVariantList &args) : Plasma::Applet(parent, args)
{
    KGlobal::locale()->insertCatalog("windowlist");

    setAspectRatioMode(Plasma::ConstrainedSquare);

    int iconSize = IconSize(KIconLoader::Desktop);

    resize((iconSize * 2), (iconSize * 2));
}

void WindowList::init()
{
    KConfigGroup configuration = config();
    QString iconName;

    if (configuration.readEntry("useArrowIcon", false))
    {
        switch (location())
        {
            case Plasma::LeftEdge:
                iconName = "arrow-left";
            break;
            case Plasma::RightEdge:
                iconName = "arrow-right";
            break;
            case Plasma::TopEdge:
                iconName = "arrow-down";
            break;
            default:
                iconName = "arrow-up";
        }
    }
    else
    {
        iconName = "preferences-system-windows";
    }

    Plasma::IconWidget *icon = new Plasma::IconWidget(KIcon(iconName), QString(), this);

    m_listMenu = new KWindowListMenu;
    m_listMenu->installEventFilter(this);

    registerAsDragHandle(icon);

    QGraphicsLinearLayout *layout = new QGraphicsLinearLayout(this);
    layout->setContentsMargins(0, 0, 0, 0);
    layout->setSpacing(0);
    layout->addItem(icon);

    Plasma::ToolTipManager::self()->setContent(this, Plasma::ToolTipContent(i18n("Window list"), i18n("Show list of opened windows"), KIcon("preferences-system-windows").pixmap(IconSize(KIconLoader::Desktop))));

    connect(this, SIGNAL(activate()), this, SLOT(showMenu()));
    connect(this, SIGNAL(destroyed()), m_listMenu, SLOT(deleteLater()));
    connect(icon, SIGNAL(clicked()), this, SLOT(showMenu()));
    connect(m_listMenu, SIGNAL(triggered(QAction*)), this, SLOT(triggered(QAction*)));
}

void WindowList::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    Applet::mousePressEvent(event);

    if (event->buttons() == Qt::MidButton)
    {
        showMenu(true);
    }
}

bool WindowList::eventFilter(QObject *object, QEvent *event)
{
    if (event->type() == QEvent::ContextMenu)
    {
        if (m_listMenu->activeAction() && m_listMenu->activeAction()->data().type() == QVariant::ULongLong)
        {
            QList<QAction*> actionList;
            KWindowInfo *window = new KWindowInfo(m_listMenu->activeAction()->data().toULongLong(), NET::WMGeometry | NET::WMFrameExtents | NET::WMWindowType | NET::WMDesktop | NET::WMState | NET::XAWMState | NET::WMVisibleName);
            TaskManager::BasicMenu *taskMenu = new TaskManager::BasicMenu(NULL, new TaskManager::TaskItem(NULL, TaskManager::TaskManager::self()->findTask(window->win())), new TaskManager::GroupManager(NULL), actionList);
            taskMenu->exec(m_listMenu->mapToGlobal(m_listMenu->actionGeometry(m_listMenu->activeAction()).center()));

            m_listMenu->hide();

            delete taskMenu;
            delete window;

            return true;
        }
    }

    return QObject::eventFilter(object, event);
}

void WindowList::showMenu(bool onlyCurrentDesktop)
{
    QList<WId> windows = KWindowSystem::windows();
    QList<QAction*> actionList;
    QList< QList<QAction*> > windowList;
    int number;

    m_listMenu->clear();

    if (!onlyCurrentDesktop)
    {
        m_listMenu->addTitle(i18n("Actions"));

        QAction *unclutterAction = m_listMenu->addAction(i18n("Unclutter windows"));

        connect(unclutterAction, SIGNAL(triggered()), m_listMenu, SLOT(slotUnclutterWindows()));

        QAction *cascadeAction = m_listMenu->addAction(i18n("Cascade windows"));

        connect(cascadeAction, SIGNAL(triggered()), m_listMenu, SLOT(slotCascadeWindows()));
    }

    for (int i = 0; i <= KWindowSystem::numberOfDesktops(); ++i)
    {
        windowList.append(QList<QAction*>());
    }

    for (int i = 0; i < windows.count(); ++i)
    {
        KWindowInfo *window = new KWindowInfo(windows.at(i), NET::WMGeometry | NET::WMFrameExtents | NET::WMWindowType | NET::WMDesktop | NET::WMState | NET::XAWMState | NET::WMVisibleName);
        NET::WindowType type = window->windowType(NET::NormalMask | NET::DialogMask | NET::OverrideMask | NET::UtilityMask | NET::DesktopMask | NET::DockMask | NET::TopMenuMask | NET::SplashMask | NET::ToolbarMask | NET::MenuMask);

        if ((onlyCurrentDesktop && !window->isOnDesktop(KWindowSystem::currentDesktop())) || type == NET::Desktop || type == NET::Dock || type == NET::TopMenu || type == NET::Splash || type == NET::Menu || type == NET::Toolbar || window->hasState(NET::SkipPager))
        {
            windows.removeAt(i);

            --i;

            delete window;

            continue;
        }

        QAction *action = new QAction(QIcon(KWindowSystem::icon(window->win())), window->visibleName(), this);
        action->setData((unsigned long long) window->win());

        QFont font = QFont(action->font());

        if (window->isMinimized())
        {
            font.setItalic(true);
        }
        else if (KWindowSystem::activeWindow() == window->win())
        {
            font.setUnderline(true);
            font.setBold(true);
        }

        action->setFont(font);

        number = ((onlyCurrentDesktop || window->onAllDesktops())?0:window->desktop());

        QList<QAction*> subList = windowList.value(number);
        subList.append(action);

        windowList.replace(number, subList);

        delete window;
    }

    number = 0;

    for (int i = 0; i <= KWindowSystem::numberOfDesktops(); ++i)
    {
        if (windowList.value(i).isEmpty())
        {
            continue;
        }

        m_listMenu->addTitle(i?KWindowSystem::desktopName(i):(onlyCurrentDesktop?i18n("Current desktop"):i18n("On all desktops")));

        for (int j = 0; j < windowList.value(i).count(); ++j)
        {
            ++number;

            m_listMenu->addAction(windowList.value(i).value(j));
        }
    }

    if (!number)
    {
        m_listMenu->clear();

        QAction *noWindows = m_listMenu->addAction(i18n("No windows"));
        noWindows->setEnabled(false);
    }


    m_listMenu->popup(QCursor::pos());
}

void WindowList::triggered(QAction *action)
{
    if (action->data().type() == QVariant::ULongLong)
    {
        if (KWindowSystem::activeWindow() == action->data().toULongLong())
        {
            KWindowSystem::minimizeWindow(action->data().toULongLong());
        }
        else
        {
            KWindowSystem::activateWindow(action->data().toULongLong());
        }
    }
}

#include "WindowList.moc"
