/******************************************************************************
*                                PlayWolf                                     *
*******************************************************************************
*                                                                             *
*                   Copyright (C) 2008-2009 Giulio Camuffo		      *
*                                                                             *
*   This program is free software; you can redistribute it and/or modify      *
*   it under the terms of the GNU General Public License as published by      *
*   the Free Software Foundation; either version 2 of the License, or         *
*   (at your option) any later version.                                       *
*                                                                             *
*   This program is distributed in the hope that it will be useful,           *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of            *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
*   GNU General Public License for more details.                              *
*                                                                             *
*   You should have received a copy of the GNU General Public License along   *
*   with this program; if not, write to the Free Software Foundation, Inc.,   *
*   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA                *
*                                                                             *
*                                                                             *
*                                                                             *
*   For further information contact me at giuliocamuffo@gmail.com	      *
******************************************************************************/

#include "playwolf.h"

///Qt includes
#include <QVariantMap>
#include <QtGui/QGraphicsGridLayout>
#include <QtGui/QGraphicsSceneHoverEvent>
#include <QtGui/QGraphicsSceneWheelEvent>
///KDE includes
#include <KConfigDialog>
#include <KIcon>
#include <Plasma/ExtenderItem>
#include <Plasma/Extender>
///PlayWolf includes
#include "infomanager.h"
#include "wolfbuttons.h"
#include "playercontroller.h"
#include "infowidget.h"
#include "wolfmeter.h"
#include "wolflabel.h"
#include "dialogwidget.h"

PlayWolf::PlayWolf(QObject *parent, const QVariantList &args)
    : Plasma::PopupApplet(parent, args)
{
    setHasConfigurationInterface(true);
    setBackgroundHints(StandardBackground);

    resize(200,200);
}

PlayWolf::~PlayWolf() {
    delete infoManager;
    PlayerController::destroy();
}

void PlayWolf::init() {
    KConfigGroup cg = config();
    widthOnPanel = cg.readEntry("widthOnPanel", 250); //defines the width of the applet when in a horizontal panel
    heightOnPanel = cg.readEntry("heightOnPanel", 150); //defines the height of the applet when in a vertical panel
    bool amarokButtonShown = cg.readEntry("amarokButtonShown", true);
    showCover = cg.readEntry("showCover", true);
    wheelSeek = cg.readEntry("wheelSeek", true);
    bool resizeFont = cg.readEntry("resizeFont", true);
    int minFont = cg.readEntry("minFont", 6.0);
    bool useCustomFont = cg.readEntry("useCustomFont", false);
    customFont = cg.readEntry("customFont", QFont());
    showInformationsOnHPanel = cg.readEntry("showInformationsOnHPanel", true);
    bool useCustomAmarokBinPath = cg.readEntry("useCustomAmarokBinPath", false);
    QString amarokBinPath = cg.readEntry("customAmarokBinPath", "/usr/bin/amarok");

    layout = new QGraphicsGridLayout(this);
    layout->setContentsMargins(0,0,0,0);
    layout->setSpacing(0);
    setLayout(layout);

    infoManager = new InfoManager();
    buttons = new WolfButtons(this);
    infos = new InfoWidget(this);
    meter = new WolfMeter(this);
    cover = new WolfLabel(this);

    infoManager->registerInfoWidget(infos);
    infoManager->registerButtonsWidget(buttons);
    infoManager->registerCoverWidget(cover);
    infoManager->registerMeterWidget(meter);

    Plasma::ExtenderItem *infosItem = new Plasma::ExtenderItem(extender());
    infosItem->setName("info");
    initExtenderItem(infosItem);
    Plasma::ExtenderItem *miscItem = new Plasma::ExtenderItem(extender());
    miscItem->setName("misc");
    initExtenderItem(miscItem);

    meter->setMinimum(0);
    meter->setMinimumHeight(5);
    meter->setMinimumWidth(5);
    buttons->setZValue(5);

    buttons->setAmarokButtonShown(amarokButtonShown);
    infos->setResizeFont(resizeFont, minFont);

    if (useCustomFont)
	infos->setFont(customFont);

    PlayerController::registerPlayWolfInstance();
    PlayerController::self()->setAmarokPath(useCustomAmarokBinPath, amarokBinPath);
    PlayerController::self()->updateInfos();
}

void PlayWolf::constraintsEvent(Plasma::Constraints constraints) {

    if (formFactor() == Plasma::Horizontal) {
	if (constraints >= (Plasma::SizeConstraint | Plasma::LocationConstraint)) {
	    resetWidgets();

	    meter->setMeterType(Plasma::Meter::BarMeterHorizontal);
	    meter->setMaximumHeight(6);

	    layout->addItem(meter,1,1,1,1);
	    if (showInformationsOnHPanel) {
		infos->show();
		layout->addItem(infos,0,1,1,1);
	    }
	    else {
		buttons->show();
		layout->addItem(buttons,0,1,1,1);
	    }
	    if (showCover) {
		cover->show();
		layout->addItem(cover,0,0,2,1);
	    }

	}
	if (showInformationsOnHPanel) {
	    buttons->setMaximumHeight(20);
	    buttons->resize(infos->contentsRect().size());
	    buttons->setPos(infos->x(),infos->y() + (contentsRect().height() - buttons->contentsRect().height())/2.0);
	}
    }
    else if (formFactor() == Plasma::Vertical) {
	if (constraints > (Plasma::SizeConstraint | Plasma::LocationConstraint)) {
	    resetWidgets();

	    meter->setMeterType(Plasma::Meter::BarMeterVertical);

	    layout->addItem(buttons,1,0,1,1);
	    layout->addItem(meter,1,1,1,1);
	    if (showCover) {
		cover->show();
		layout->addItem(cover,0,0,1,2);
	    }

	    buttons->show();
	}
	if (showCover)
	    cover->setMinimumHeight(size().width());
	else
	    cover->setMinimumHeight(-1);
    }
    else if (formFactor() == Plasma::Planar) {
	if (constraints > (Plasma::SizeConstraint | Plasma::LocationConstraint)) {
	    resetWidgets();

	    meter->setMeterType(Plasma::Meter::BarMeterHorizontal);

	    layout->addItem(infos,0,1,1,1);
	    layout->addItem(meter,1,0,1,2);
	    layout->addItem(buttons,2,0,1,2);
	    if (showCover) {
		cover->show();
		layout->addItem(cover,0,0,1,1);
	    }
	    layout->setRowSpacing(1,5);

	    infos->show();
	    buttons->show();
	}
	meter->setMaximumHeight(contentsRect().height()/6);
	buttons->setMaximumHeight(contentsRect().height()/6);
    }
}

void PlayWolf::resetWidgets() {
    meter->setMaximumHeight(-1);
    meter->setMinimumHeight(-1);
    cover->setMinimumHeight(-1);
    buttons->setMaximumHeight(-1);
    buttons->setGeometry(0,0,0,0);
    while (layout->count() > 0)
	layout->removeAt(0);
    infos->hide();
    buttons->hide();
    cover->hide();
}

void PlayWolf::initExtenderItem(Plasma::ExtenderItem *item) {
    if (item->name() == QString("misc")) {
	miscDialog = new DialogWidget(item);
	miscDialog->setPreferredSize(250,150);
	item->setWidget(miscDialog);
	item->setTitle("Miscellaneous");
	infoManager->registerRatingWidget(miscDialog);
	infoManager->registerVolumeWidget(miscDialog);
	infoManager->registerPlaylistWidget(miscDialog);
    }
    else if (item->name() == QString("info")) {
	infosDialog = new InfoWidget(item);
	infosDialog->setPreferredSize(250,100);
	item->setWidget(infosDialog);
	item->setTitle("Information");
	infoManager->registerInfoWidget(infosDialog);
    }
}

void PlayWolf::popupEvent(bool show) {
    if (show)
	miscDialog->updateContents();
}

void PlayWolf::createConfigurationInterface(KConfigDialog *parent) {
    QWidget *generalWidget = new QWidget();
    QWidget *horizontalWidget = new QWidget();
    QWidget *verticalWidget = new QWidget();
    generalUi.setupUi(generalWidget);
    horizontalUi.setupUi(horizontalWidget);
    verticalUi.setupUi(verticalWidget);
    parent->addPage(generalWidget, i18n("General"), icon());
    KPageWidgetItem *horizontalPage = parent->addPage(horizontalWidget, i18n("Horizontal FormFactor"), icon());
    KPageWidgetItem *verticalPage = parent->addPage(verticalWidget, i18n("Vertical FormFactor"), icon());
    parent->setButtons( KDialog::Ok | KDialog::Cancel);
    connect(parent, SIGNAL(okClicked()), this, SLOT(configAccepted()));
    parent->resize(600,380);

    generalUi.amarokButtonShown->setChecked(buttons->isAmarokButtonShown());
    generalUi.showCover->setChecked(showCover);
    generalUi.wheelSeek->setChecked(wheelSeek);
    generalUi.resizeFont->setChecked(infos->resizeFont());
    generalUi.minFont->setEnabled(infos->resizeFont());
    generalUi.minFont->setValue(infos->minimumFont());
    generalUi.scrollLabel->setChecked(! infos->resizeFont());
    generalUi.useCustomFont->setChecked(infos->font() != QFont());
    generalUi.fontChooser->setFont(customFont);
    generalUi.customAmarokBinPath->setText(PlayerController::self()->amarokPath());
    generalUi.useCustomAmarokBinPath->setChecked(PlayerController::self()->useCustomAmarokPath());

    if (formFactor() == Plasma::Horizontal) {
	verticalPage->setEnabled(false);
	horizontalUi.widthOnPanel->setValue(widthOnPanel);
	horizontalUi.showInformationsOnHPanel->setChecked(showInformationsOnHPanel);
    }
    else if (formFactor() == Plasma::Vertical) {
	horizontalPage->setEnabled(false);
	verticalUi.heightOnPanel->setValue(heightOnPanel);
    }
    else {
	horizontalPage->setEnabled(false);
	verticalPage->setEnabled(false);
    }
}

void PlayWolf::configAccepted() {
    KConfigGroup cg = config();

    bool amarokButtonShown = generalUi.amarokButtonShown->isChecked();
    showCover = generalUi.showCover->isChecked();
    wheelSeek = generalUi.wheelSeek->isChecked();
    bool resizeFont = generalUi.resizeFont->isChecked();
    int minFont = generalUi.minFont->value();
    bool useCustomFont = generalUi.useCustomFont->isChecked();
    QFont font = generalUi.fontChooser->font();
    showInformationsOnHPanel = horizontalUi.showInformationsOnHPanel->isChecked();
    QString amarokBinPath = generalUi.customAmarokBinPath->text();
    bool useCustom = generalUi.useCustomAmarokBinPath->isChecked();

    buttons->setAmarokButtonShown(amarokButtonShown);
    infos->setResizeFont(resizeFont,minFont);
    if (!useCustomFont)
	font = QFont();
    else
	customFont = font;
    infos->setFont(font);
    PlayerController::self()->setAmarokPath(useCustom, amarokBinPath);

    cg.writeEntry("useCustomAmarokBinPath", useCustom);
    cg.writeEntry("customAmarokBinPath", amarokBinPath);
    cg.writeEntry("amarokButtonShown", amarokButtonShown);
    cg.writeEntry("showCover", showCover);
    cg.writeEntry("wheelSeek", wheelSeek);
    cg.writeEntry("resizeFont", resizeFont);
    cg.writeEntry("minFont", minFont);
    cg.writeEntry("useCustomFont", useCustomFont);
    cg.writeEntry("customFont", customFont);
    cg.writeEntry("showInformationsOnHPanel", showInformationsOnHPanel);
    if (formFactor() == Plasma::Horizontal) {
	int newWidth;
	if (widthOnPanel != (newWidth = horizontalUi.widthOnPanel->value())) {
	    cg.writeEntry("widthOnPanel" ,newWidth);
	    widthOnPanel = newWidth;
	    resize(newWidth,30);
	}
    }
    else if (formFactor() == Plasma::Vertical) {
	int newHeight;
	if (heightOnPanel != (newHeight = verticalUi.heightOnPanel->value())) {
	    cg.writeEntry("heightOnPanel" ,newHeight);
	    heightOnPanel = newHeight;
	    resize(30,newHeight);
	}
    }
    updateConstraints();
}

QSizeF PlayWolf::sizeHint(Qt::SizeHint which, const QSizeF &constraint) const {
    QSizeF hint = Plasma::Applet::sizeHint(which,constraint);

    if(formFactor() == Plasma::Horizontal) {
	return QSizeF(widthOnPanel, hint.height());
    }
    if(formFactor() == Plasma::Vertical) {
	return QSizeF(hint.width(), heightOnPanel);
    }

    return hint;
}

void PlayWolf::hoverEnterEvent(QGraphicsSceneHoverEvent *event) {
    if (event->pos().x() <= (contentsRect().width() - contentsRect().x()))
	if ((formFactor() == Plasma::Horizontal) and showInformationsOnHPanel)
	    buttons->show();
}

void PlayWolf::hoverLeaveEvent(QGraphicsSceneHoverEvent *) {
    if ((formFactor() == Plasma::Horizontal) and showInformationsOnHPanel)
	buttons->hide();
}

void PlayWolf::wheelEvent(QGraphicsSceneWheelEvent *event) {
    if (wheelSeek) {
	if (meter->contains(event->pos())) {
	    if (event->delta() > 0)
		PlayerController::self()->seekForward(5);
	    else
		PlayerController::self()->seekBack(5);
	}
	else if (cover->contains(event->pos())) {
	    if (event->delta() > 0)
		PlayerController::self()->volumeUp();
	    else
		PlayerController::self()->volumeDown();
	}
	else if (infos->contains(event->pos())) {
	    if (event->delta() > 0)
		PlayerController::self()->next();
	    else
		PlayerController::self()->prev();
	}
    }
}

#include "playwolf.moc"