;;; planner-diary.el --- Diary integration for the Emacs Planner (planner.el)

;; Copyright (C) 2003,2004 Thomas Gehrlein <Thomas.Gehrlein@t-online.de>

;; Emacs Lisp Archive Entry
;; Filename: planner-diary.el
;; Time-stamp: "2004-03-27 17:04:58 Thomas Gehrlein"
;; Version: 1.0
;; Keywords: hypermedia
;; Author: Thomas Gehrlein <Thomas.Gehrlein@t-online.de>
;; Maintainer: Thomas Gehrlein <Thomas.Gehrlein@t-online.de>
;; Description: Integrate the Emacs Planner with Calendar and Diary
;; URL: http://sacha.free.net.ph/notebook/emacs/stable/planner/planner-diary.el
;; ChangeLog: Can be requested from the maintainer
;; Compatibility: Emacs20, Emacs21

;; This file is not part of GNU Emacs.

;; This is free software; you can redistribute it and/or modify it under
;; the terms of the GNU General Public License as published by the Free
;; Software Foundation; either version 2, or (at your option) any later
;; version.
;;
;; This is distributed in the hope that it will be useful, but WITHOUT
;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
;; for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
;; MA 02111-1307, USA.

;;; Commentary:
;;

;;; PLANNER-DIARY
;;
;; (The following documentation describes the stable version of planner-diary
;; (v1.0).  If planner-diary doesn't do what I claim here, then it's a bug and you
;; will fix it.  This file contains additional code that is part of the
;; development version of planner.  This code may or may not work.  Use it at your
;; own risk.)
;;
;; If you use Emacs' diary feature, planner-diary could be helpful for you.  It
;; puts all diary entries for the current day in the "* Diary" section of your day
;; plan page.  This section is updated every time you display the file in Emacs.
;; By default the diary section of past pages is not updated (it's pretty unlikely
;; that you want to add new diary entries for the past).
;;
;; If you want to use planner-diary.el, put the file in your load path and add
;; this to your .emacs:
;;
;; (require 'planner-diary)
;;
;; (This step should not be necessary once a stable planner package will be put
;; together.)
;;
;; planner-diary.el needs fancy-diary-display.  To use fancy-diary-display add
;; this to your .emacs:
;;
;; (add-hook 'diary-display-hook 'fancy-diary-display)
;;
;; You can use planner diary in two different ways:
;;
;; 1) You can put the following line of lisp code in your day plan pages to
;;    display your diary entries:
;;
;;    <lisp>(planner-diary-entries-here)</lisp>
;;
;;    You can do this automatically for all day plan pages:
;;
;;    (setq planner-day-page-template
;;          "* Tasks\n\n\n* Diary\n\n<lisp>(planner-diary-entries-here)</lisp>\n\n* Notes")
;;
;;    When you open a day plan page outside emacs, you will see the line of lisp
;;    code and not your diary entries.
;;
;; 2) If you want the saved files to contain your entries and not just a line
;;    of lisp, add the following lines to your .emacs:
;;
;;    (setq planner-diary-use-diary t)
;;    (planner-diary-insinuate)
;;
;;    You should also customize or set planner-day-page-template to include a
;;    "* Diary":
;;
;;    (setq planner-day-page-template
;;          "* Tasks\n\n\n* Schedule\n\n\n* Diary\n\n\n* Notes")
;;
;;    C-c C-e updates the diary sections.  C-u C-c C-e forces an update, it
;;    inserts the diary section for the day, even if the day is in the past or
;;    if there is no Diary section in the buffer.
;;
;; If you want to see your diary entries for more than just 1 day, set
;; `planner-diary-number-of-diary-entries' accordingly.  This works for
;; either of the 2 approaches.
;;
;; If you want to use the cal-desk package, simply follow the instructions in
;; cal-desk.el.  If you get the cal-desk layout from the Calendar buffer, you get
;; it in the day plan buffer, too.
;;
;; If you use planner-diary, you might consider using the Calendar support of planner.
;; To get Calendar integration add this to your .emacs:
;;
;;    (planner-insinuate-calendar)
;;
;; For further information refer to the documentation for this function.
;;
;; If you have questions about planner-diary, feature requests, bug reports or
;; anything else you want to tell me: thomas.gehrlein@t-online.de.

;;; BUGS
;;
;; Older version of planner.el use the variable `planner-diary-file'.  If you
;; don't see any entries in your Diary section, please check the value of
;; `planner-diary-file'.  If it's not what you expect, please set it explicitly
;; or update to a newer version of planner.el.

;;; THANKS TO:
;;
;; - Jody Klymak (jklymak AT coas DOT oregonstate DOT edu)
;;   Added some documentation
;;
;; - Sacha Chua (sacha AT free DOT net DOT ph)
;;   Thoughts and ideas
;;
;; - Bastien Guerry
;;   Bug report and appointment integration


;;; HISTORY:
;; version 1.0 First stable version.  Expected to work as documented.  Same
;; feature of previous versions are not supported.
;;
;; version 0.7 Major rewrite.  New diary types (cal-desk, appointments, privat,
;; public diaries).  Add diary entries from planner pages.
;;
;; version 0.6 minor changes
;;
;; version 0.5 appointment integration (thanks to Bastien Guerry)

(require 'planner)
(require 'diary-lib)
(require 'calendar)

;;; USER VARIABLES
;; STANDARD DIARY SUPPORT (no fancy extras - this is what I, the author of
;: this file, use)
;;; Code:
(defcustom planner-diary-string "* Diary"
  "*Header for the diary section in a day plan page."
  :type 'string
  :group 'planner)

(defcustom planner-diary-file diary-file
  ;; FIXME: Test if this works as buffer-local var
  "*The diary file you want to use with planner-diary."
  :type 'file
  :group 'planner)

(defcustom planner-diary-use-diary nil
;;   (if (string-match planner-diary-string planner-day-page-template)
;;       t
;;     nil)
  "*Non-nil means: Automatically insert a diary section into day plan pages.
Uses the standard fancy-diary display."
  :type 'boolean
  :group 'planner)

(defcustom planner-diary-number-of-days 1
  "*Number of days of diary entries to display in the standard diary section."
  :type 'number
  :group 'planner)

;; for backward compatability
;; (defcustom planner-diary-number-of-diary-entries 1
;;   "*Obsolete, use `planner-diary-number-of-days' instead."
;;   :type 'number
;;   :group 'planner)


;;; INTERNAL VARS
;; FIXME: Is this used for anything?
(defcustom planner-diary-exclude-regexp ""
  "*Regexp for diary entries that should not be displayed.
Used by the schedule code."
  :type 'regexp
  :group 'planner)

(defcustom planner-diary-time-regexp "[0-2][0-9]:[0-5][0-9]"
  ;; FIXME: Internationalize?  (AM and PM)
  "A regexp for time in a diary appt entry."
  :type 'regexp
  :group 'planner)


;;; FUNCTIONS
;; GETTING THE RELEVANT ENTRIES
;; planner-diary-get-diary-entries is the main function,
;; planner-diary-get-appts-entries needs to be rewritten.
;; planner-diary-get-[public/private/cal-desk]-entries call
;; planner-diary-get-diary-entries.
(defun planner-diary-get-diary-entries (date &optional no-of-days file)
  "Get the diary entries for DATE and the following NO-OF-DAYS days from FILE.
DATE is a list (month day year).  NO-OF-DAYS defaults to
`planner-diary-number-of-days'.  FILE defaults to
`planner-diary-file'."
  (save-window-excursion
    (let* ((fancy-diary-buffer "temporary-fancy-diary-buffer")
           (entries)
           (no-of-days (or no-of-days planner-diary-number-of-days))
           (diary-display-hook '(sort-diary-entries fancy-diary-display))
           (diary-file (or  file planner-diary-file)))
      ;; no messages from list-diary-entries
      (list-diary-entries date no-of-days)
      (switch-to-buffer fancy-diary-buffer)
      ;; return empty string if buffer is empty
      (if (= (point-max) 1)
          (setq entries "No entries.")
        (setq entries
              (buffer-substring
               (if (> no-of-days 1)   ; if more than 1 day, show everything
                   (point-min)
                 ;; remove date and lots of =s if just for 1 day
                 (progn (goto-char (point-min))
                        (search-forward-regexp "^=+$") ; one or more =
                        (1+ (point))))
               ;; remove final newline
               (progn (goto-char (point-max))
                      (when (bolp) (backward-char 1))
                      (point)))))
      (kill-buffer fancy-diary-buffer)
      entries)))

(defun planner-diary-get-cal-desk-entries (date &optional no-of-days file)
  "Get the cal-desk style diary entries for DATE.
DATE is a list (month day year). If NO-OF-DAYS is non-nil,
entries for NO-OF-DAYS are included. NO-OF-DAYS defaults to
`planner-diary-cal-desk-number-of-days'. FILE defaults to
`planner-diary-cal-desk-file'."
  (planner-diary-get-diary-entries date
                                   (or no-of-days
                                       planner-diary-cal-desk-number-of-days)
                                   (or file
                                       planner-diary-cal-desk-file)))



;;; LISP FUNCTIONS FOR USE IN PLANNER DAY PAGES
;;; arg FILE to specify a diary-file (suggested by David O'Toole)
(defun planner-diary-entries-here (&optional file no-of-days)
  "Display the diary entries from FILE for the next NO-OF-DAYS days.
FILE defaults to `planner-diary-file', NO-OF-DAYS defaults to
`planner-diary-number-of-days'.

Put this is your day pages:
\"<lisp>(planner-diary-entries-here)</lisp>\"
or this, if you want to do fancy things:
\"<lisp>(planner-diary-entries-here \"/path/to/diary/file\" 1)</lisp>\"

You might want to use `planner-day-page-template' to do so."
  (planner-diary-get-diary-entries
   (planner-filename-to-calendar-date (emacs-wiki-page-name))
   (or no-of-days planner-diary-number-of-days)
   (or file planner-diary-file)))


;;; CODE FOR DEALING WITH SECTIONS
;; There's a lot of code duplication in the following 3 functions.
(defun planner-diary-update-section (file title text &optional force)
  ;;FIXME: Find a good place to insert a new section
  "In FILE, update existing section TITLE by replacing existing text with TEXT.
If optional arg FORCE is non-nil, update the section even if it doesn't exist,
i.e. insert TITLE followed by TEXT at the top of the buffer."
  ;; search for something like "^* Diary$", delete buffer content to the next
  ;; "^* "
  ;; sanity checks
  (unless (equal major-mode 'planner-mode)
    (error "This is not a planner buffer"))
  (save-excursion
    (goto-char (point-min))
    (or (re-search-forward (concat "^" title "$") (point-max) t)
        (when force
          (insert title "\n\n\n")
          (backward-char 3)
          t)
        (error "No \"%s\" section in this buffer" title))
    ;; point is at the end of something like "* Diary"
    ;; delete the old text
    (let ((beg (point))
          (end (if (re-search-forward "^* " (point-max) t)
                   (progn (beginning-of-line)
                          (backward-char 1)
                          (point))
                 (point-max))))
      (delete-region beg end)
      ;; point is at the end of "* Diary"
      (insert "\n\n")
      (unless (string= text "")
          (insert text "\n")))))


(defun planner-diary-delete-section-text (file title)
  "In FILE, delete the text of section TITLE."
  ;; search for something like "^* Diary$", delete buffer content to the next
  ;; "^* "
  ;; sanity checks
  (unless (equal major-mode 'planner-mode)
    (error "This is not a planner buffer"))
  (save-excursion
    (goto-char (point-min))
    (unless (re-search-forward (concat "^" title "$") (point-max) t)
      (error "No \"%s\" section in this buffer" title))
    ;; point is at the end of something like "* Diary"
    ;; delete the old text
    (let ((beg (point))
          (end (if (re-search-forward "^* " (point-max) t)
                   (progn (beginning-of-line)
                          (backward-char 1)
                          (point))
                 (point-max))))
      (delete-region beg end)
      ;; point is at the end of "* Diary"
      (insert "\n\n"))))


(defun planner-diary-delete-section (file title)
  "In FILE, delete the whole section TITLE, including the text."
  ;; search for something like "^* Diary$", delete buffer content to the next
  ;; "^* "
  ;; sanity checks
  (unless (equal major-mode 'planner-mode)
    (error "This is not a planner buffer"))
  (save-excursion
    (goto-char (point-min))
    (unless (re-search-forward (concat "^" title "$") (point-max) t)
      (error "No \"%s\" section in this buffer" title))
    ;; point is at the end of something like "* Diary"
    ;; delete the old text from the beginning of the line
    (let ((beg (line-beginning-position))
          (end (if (re-search-forward "^* " (point-max) t)
                   (line-beginning-position)
                   ;;                    (progn (beginning-of-line)
                   ;;                           (backward-char 1)
                   ;;                           (point))
                  (point-max))))
      (delete-region beg end))))


(defun planner-diary-insert-diary (&optional force)
  "Insert the fancy diary for the day into the day plan file.
If FORCE is non-nil, insert a diary section even if there is no
`planner-diary-string' in the buffer."
  (interactive "P")
  ;; sanity check
  (unless (string-match planner-date-regexp (buffer-name))
    (error "Cannot insert diary in this buffer"))
  (planner-diary-update-section (buffer-file-name) ; file
                                planner-diary-string ; title
                                (planner-diary-get-diary-entries     ; text
                                 (planner-filename-to-calendar-date  ; date
                                  (file-name-nondirectory (buffer-file-name)))
                                 planner-diary-number-of-days
                                 planner-diary-file))
                                force)

(defun planner-diary-insert-diary-maybe (&optional force)
  "Maybe insert the fancy diary for the day into the day plan file.
If the current day is in the past and FORCE is nil, don't do anything.  If
FORCE is non-nil, insert a diary section even if there is no
`planner-diary-string' in the buffer."
  (interactive "P")
  ;; check if we're in a day plan buffer
  (if (and (string-match planner-date-regexp (buffer-name))
           (or force                    ; don't care about the date
               (not (string< (buffer-name) (planner-today))))) ; not past
      ;; today, future, or force
      (planner-diary-insert-diary force)
    ;; we are in the past -> do nothing, message when interactive
    (when (interactive-p)
      (message "No day plan buffer or date is in the past.  No diary entries inserted."))))



;; UPDATE ALL DIARIES
(defun planner-diary-insert-all-diaries (&optional force)
  "Update all diary sections in a day plan file.
If FORCE is non-nil, insert a diary section even if there is no section header.
Inserts only diaries if the corresponding `planner-diary-use-*' variable is t."
  (interactive)
  (when planner-diary-use-diary
    (planner-diary-insert-diary force)))

(defun planner-diary-insert-all-diaries-maybe (&optional force)
  "Update all diary sections in a day plan file.
If the current day is in the past and FORCE is nil, don't do anything.
If FORCE is non-nil, insert a diary section even if there is no section header.
Inserts only diaries if the corresponding `planner-diary-use-*' variable is t."
  (interactive)
  ;; I intentionally call these individual functions rather than
  ;; planner-diary-insert-all-diaries.  It might make future code changes
  ;; simpler.
  (when planner-diary-use-diary
    (planner-diary-insert-diary-maybe force)))


(defun planner-diary-show-day-plan-or-diary ()
  "Show the day plan or diary entries for the date under point in calendar.
Add this to `calendar-move-hook' if you want to use it.  In that case you
should also `remove-hook' `planner-calendar-show' from `calendar-move-hook'."
  (interactive)
  (or (planner-calendar-show)
      (view-diary-entries 1)))


(defun planner-diary-insinuate ()
  "Hook Diary into Planner.
Automatically insert and update a Diary section in day plan files.
This adds a new key binding to `planner-mode-map':
C-cC-e updates the diary sections."
  ;; FIXME: update all diary sections: planner-diary-insert-all-diaries-maybe
  (define-key planner-mode-map "\C-c\C-e" 'planner-diary-insert-all-diaries-maybe)
  (add-hook 'planner-goto-hook 'planner-diary-insert-all-diaries-maybe))

(defalias 'planner-insinuate-diary 'planner-diary-insinuate)

(provide 'planner-diary)

;;; planner-diary.el ends here
