/***************************************************************************
 *            qof-expenses.c
 *
 *  Thu Oct 21 07:59:13 2004-2005
 *  Copyright  2004-2005  Neil Williams  <linux@codehelp.co.uk>
 ****************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
/** @addtogroup QOF
	@{ */
/** @addtogroup Exp Expenses
	@{ */
/** @file  qof-expenses.c
	@brief QOF expense implementation for pilot-link
	@author Copyright (c) 2004-2005 Neil Williams <linux@codehelp.co.uk>
*/

#include "qof-expenses.h"
#include "pilot-todo.h"
#include "pilot-qof.h"

#define QOF_EXPENSES_DESC "Pilot-link QOF expenses"

static QofLogModule log_module = PQ_MOD_PILOT;

qof_exp *
expense_create (QofBook *book)
{
	Expense_t *qe;
	qof_exp *obj;
	time_t current;
	QofCollection *coll;
	GList *all;

	obj = g_new0(qof_exp, 1);
	qof_instance_init (&obj->inst, PILOT_LINK_QOF_EXPENSES, book);
	coll = qof_book_get_collection (book, PILOT_LINK_QOF_EXPENSES);
	all = qof_collection_get_data (coll);
	all = g_list_prepend (all, obj);
	qof_collection_set_data (coll, all);
	qe = &obj->wrap;
	current = time (NULL);
	localtime_r (&current, &qe->date);
	qe->amount = "0";
	return obj;
}

static const char*
qof_exp_paymentAsString(enum ExpensePayment payment)
{
	char *string;
	int i;

	string = "Cash"; // following install-expenses convention.
	for(i = 0; ExpensePaymentNames[i] != NULL; i++)
	{
		if (i == (int)payment)
		{
			string = g_strdup(ExpensePaymentNames[(int)payment]);
		}
	}
	return string;
}

static const char*
qof_exp_typeAsString(enum ExpenseType type)
{
	char* string;
	int i;

	string = "Bus"; // following install-expenses convention.
	for (i = 0; ExpenseTypeNames[i] != NULL; i++)
	{
		if (i == (int)type)
		{
			string = g_strdup(ExpenseTypeNames[(int)type]);
		}
	}
	return string;
}

static enum ExpensePayment
qof_exp_paymentFromString(const char* payment_string)
{
	int i;

	for(i = 0; ExpensePaymentNames[i] != NULL; i++)
	{
		if(0 == safe_strcmp(ExpensePaymentNames[i], payment_string))
		{
			return i;
		}
	}
	return epCash;
}

static enum ExpenseType
qof_exp_typeFromString(const char* type_string)
{
	int i;

	for (i = 0; ExpenseTypeNames[i] != NULL; i++)
	{
		if(0 == safe_strcmp(ExpenseTypeNames[i], type_string))
		{ 
			return i;
		}
	}
	return etBus;
}

static Timespec
exp_getDate (qof_exp * e)
{
	Expense_t *qe;
	Timespec t;

	t.tv_sec = 0;
	t.tv_nsec = 0;
	g_return_val_if_fail (e != NULL, t);
	qe = &e->wrap;
	timespecFromTime_t (&t, mktime (&qe->date));
	return t;
}

static char*
exp_getType (qof_exp * e)
{
	Expense_t *qe;
	char* string;

	g_return_val_if_fail (e != NULL, NULL);
	qe = &e->wrap;
	string = g_strdup(qof_exp_typeAsString(qe->type));
	return string;
}

static char*
exp_getPayment (qof_exp * e)
{
	Expense_t *qe;
	char* string;

	g_return_val_if_fail (e != NULL, NULL);
	qe = &e->wrap;
	string = g_strdup(qof_exp_paymentAsString(qe->payment));
	return string;
}

static gint
exp_getCurrency (qof_exp * e)
{
	Expense_t *qe;

	g_return_val_if_fail (e != NULL, -1);
	qe = &e->wrap;
	return qe->currency;
}

static gnc_numeric
exp_getAmount (qof_exp * e)
{
	Expense_t *qe;
	gnc_numeric amount;
	double pi_amount;
	gchar *numeric_char;

	amount = gnc_numeric_zero ();
	g_return_val_if_fail (e != NULL, amount);
	qe = &e->wrap;
	/* floating point as a string converts to gnc_numeric */
	PINFO (" get_char=%s", qe->amount);
	pi_amount = strtod (qe->amount, NULL);
	PINFO (" get_double=%f", pi_amount);
	/** \todo Don't assume 100 */
	amount = double_to_gnc_numeric (pi_amount, 100, 
	 GNC_HOW_DENOM_EXACT | GNC_HOW_RND_ROUND);
	numeric_char = gnc_numeric_to_string(amount);
	PINFO (" get_numeric=%s", numeric_char);
	g_free(numeric_char);
	if (gnc_numeric_check (amount) == GNC_ERROR_OK)
	{
		PINFO (" valid");
		return amount;
	}
	return gnc_numeric_zero ();
}

static char *
exp_getVendor (qof_exp * e)
{
	Expense_t *qe;

	g_return_val_if_fail (e != NULL, NULL);
	qe = &e->wrap;
	return qe->vendor;
}

static char *
exp_getCity (qof_exp * e)
{
	Expense_t *qe;

	g_return_val_if_fail (e != NULL, NULL);
	qe = &e->wrap;
	return qe->city;
}

static char *
exp_getAttendees (qof_exp * e)
{
	Expense_t *qe;

	g_return_val_if_fail (e != NULL, NULL);
	qe = &e->wrap;
	return qe->attendees;
}

static char *
exp_getNote (qof_exp * e)
{
	Expense_t *qe;

	g_return_val_if_fail (e != NULL, NULL);
	qe = &e->wrap;
	return qe->note;
}

static enum ExpenseDistance
exp_getDistance(qof_exp *e)
{
	guint i;

	for(i = 0; ExpenseDistanceNames[i] != NULL; i++)
	{
		if(i == e->distance_unit)
		{
			return (gint)i;
		}
	}
	return edMiles;
}

static char* exp_getCategory(qof_exp *e)
{
	g_return_val_if_fail(e != NULL, NULL);
	return e->category;
}

static void
exp_setDate (qof_exp * e, Timespec h)
{
	Expense_t *qe;
	time_t s;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	s = timespecToTime_t (h);
	qe->date = *localtime (&s);	/* weird? */
}

static void
exp_setType (qof_exp * e, const char *type_string)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	qe->type = qof_exp_typeFromString(type_string);
}

static void
exp_setPayment (qof_exp * e, const char *payment_string)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	qe->payment = qof_exp_paymentFromString(payment_string);
}

static void
exp_setCurrency (qof_exp * e, gint data)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	qe->currency = data;
}

/* stored in the Palm as a string version of a floating point number. */
static void
exp_setAmount (qof_exp * e, gnc_numeric h)
{
	Expense_t *qe;
	double exp_d;
	gchar *numeric_char;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	numeric_char = gnc_numeric_to_string(h);
	PINFO (" set_numeric=%s", numeric_char);
	g_free(numeric_char);
	exp_d = gnc_numeric_to_double (h);
	PINFO (" set_double=%f", exp_d);
	/** \todo Don't assume two decimal places */
	qe->amount = g_strdup_printf ("%.2f", exp_d);
	PINFO (" set_char=%s", qe->amount);
}

static void
exp_setVendor (qof_exp * e, const char *h)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	qe->vendor = g_strdup (h);
}

static void
exp_setCity (qof_exp * e, const char *h)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	qe->city = g_strdup (h);
}

static void
exp_setAttendees (qof_exp * e, const char *h)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	qe->attendees = g_strdup (h);
}

static void
exp_setNote (qof_exp * e, const char *h)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	qe->note = g_strdup (h);
}

static void
exp_setDistance(qof_exp *e, const char *distance_name)
{
	int i;
/** \todo Convert when setting into ExpensePref */
	for(i = 0; ExpenseDistanceNames[i] != NULL; i++)
	{
		if(0 == safe_strcmp(ExpenseDistanceNames[i], distance_name))
		{
			e->distance_unit = i;
		}
	}
}

static void
exp_setCategory(qof_exp *e, const char *n)
{
	g_return_if_fail(e != NULL);
	e->category = g_strdup(n);
}

int
exp_unpack(QofEntity *ent, gpointer user_data)
{
	Expense_t *qe;
	qof_exp *obj;
	pi_buffer_t *pi_buf;
	pilot_qof_data *context;
	size_t len;
	int size;

	context = (pilot_qof_data*)user_data;
	g_return_val_if_fail(context != NULL, -1);
	g_return_val_if_fail(ent != NULL, -1);
	ENTER (" ");
	obj = (qof_exp*)ent;
	qe = &obj->wrap;
	pi_buf = (pi_buffer_t*)context->pi_buf;
	len = sizeof(pi_buf->data);
	size = unpack_Expense(qe, pi_buf->data, pi_buf->used);
	exp_setCategory(obj, context->pi_cat.name[context->ent_category]);
	LEAVE (" ");
	return size;
}

int 
exp_pack(QofEntity *ent, gpointer user_data)
{
	pilot_qof_data *context;
	Expense_t *qe;
	qof_exp *obj;
	int size, len;

	size = 0;
	len = PQ_DEF_BUFSZ;
	context = (pilot_qof_data*)user_data;
	g_return_val_if_fail((context), -1);
	ENTER (" ");
	obj = (qof_exp*)ent;
	qe = &obj->wrap;
	/* pack_Expense still uses the old prototype
	using len instead of pi_buf->used. */
	size = pack_Expense(qe, context->pi_buf->data, len);
	/* work around the old prototype */
	if( size > 0 ) { context->pi_buf->used = size; }
	LEAVE (" ");
	return size;
}

int
exp_appinfo_unpack(QofEntity *ent, gpointer user_data)
{
	ExpenseAppInfo_t app_e;
	pilot_qof_data *context;

	/* There is never an entity at this stage */
	context = (pilot_qof_data*)user_data;
	g_return_val_if_fail(context != NULL, -1);
	ENTER (" ");
	unpack_ExpenseAppInfo(&app_e, context->app_buf->data, context->app_buf->used);
	context->pi_cat = app_e.category;
	LEAVE (" ");
	return 0;
}

int
qof_exp_free(QofEntity *ent, gpointer user_data)
{
	Expense_t *qe;
	qof_exp *obj;

	g_return_val_if_fail(ent != NULL, -1);
	ENTER (" ");
	obj = (qof_exp*)ent;
	qe = &obj->wrap;
	free_Expense(qe);
	LEAVE (" ");
	return 0;
}

static QofObject expenses_object_def = {
      interface_version: QOF_OBJECT_VERSION,
      e_type:            PILOT_LINK_QOF_EXPENSES,
      type_label:        QOF_EXPENSES_DESC,
      create:            (gpointer) expense_create,
      book_begin:        NULL,
      book_end:          NULL,
      is_dirty:          qof_collection_is_dirty,
      mark_clean:        qof_collection_mark_clean,
      foreach:           qof_collection_foreach,
      printable:         NULL,
      version_cmp:       (int (*)(gpointer, gpointer)) qof_instance_version_cmp,
};

static pilot_pack expenses_pack_def = {
        e_type:          PILOT_LINK_QOF_EXPENSES,
        pack_func:       exp_pack,
        unpack_func:     exp_unpack,
        free_pack_func:  qof_exp_free,
        palm_db_name:    "ExpenseDB",
        app_info_unpack: exp_appinfo_unpack,
};

gboolean
ExpensesRegister (void)
{
	static QofParam params[] = {
	 { EXP_DATE,       QOF_TYPE_DATE,    (QofAccessFunc) exp_getDate,      (QofSetterFunc) exp_setDate },
	 { EXP_TYPE,       QOF_TYPE_STRING,  (QofAccessFunc) exp_getType,      (QofSetterFunc) exp_setType },
	 { EXP_PAYMENT,    QOF_TYPE_STRING,  (QofAccessFunc) exp_getPayment,   (QofSetterFunc) exp_setPayment },
	 { EXP_CURRENCY,   QOF_TYPE_INT32,   (QofAccessFunc) exp_getCurrency,  (QofSetterFunc) exp_setCurrency },
	 { EXP_AMOUNT,     QOF_TYPE_NUMERIC, (QofAccessFunc) exp_getAmount,    (QofSetterFunc) exp_setAmount },
	 { EXP_VENDOR,     QOF_TYPE_STRING,  (QofAccessFunc) exp_getVendor,    (QofSetterFunc) exp_setVendor },
	 { EXP_CITY,       QOF_TYPE_STRING,  (QofAccessFunc) exp_getCity,      (QofSetterFunc) exp_setCity },
	 { EXP_ATTENDEES,  QOF_TYPE_STRING,  (QofAccessFunc) exp_getAttendees, (QofSetterFunc) exp_setAttendees },
	 { EXP_NOTE,       QOF_TYPE_STRING,  (QofAccessFunc) exp_getNote,      (QofSetterFunc) exp_setNote },
	 { EXP_DISTANCE,   QOF_TYPE_STRING,  (QofAccessFunc) exp_getDistance,  (QofSetterFunc) exp_setDistance },
	 { EXP_CATEGORY,   QOF_TYPE_STRING,  (QofAccessFunc) exp_getCategory,  (QofSetterFunc) exp_setCategory },
	 { QOF_PARAM_BOOK, QOF_ID_BOOK,      (QofAccessFunc) qof_instance_get_book, NULL },
	 { QOF_PARAM_GUID, QOF_TYPE_GUID,    (QofAccessFunc) qof_instance_get_guid, NULL },
	 { NULL },
	};

	qof_class_register (PILOT_LINK_QOF_EXPENSES, NULL, params);

	pilot_qof_pack_register (&expenses_pack_def);

	return qof_object_register (&expenses_object_def);
}
/** @} */
/** @} */
