/***************************************************************************
 *            qof-datebook.c
 *
 *  Thu Oct 21 15:38:39 2004
 *  Copyright  2004-2005  Neil Williams
 *  linux@codehelp.co.uk
 ****************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
//#define _GNU_SOURCE

#include "qof-datebook.h"
#include "pilot-todo.h"
#include "pilot-qof.h"

static QofLogModule log_module = PQ_MOD_PILOT;

#define DATEBOOK_VERSION datebook_v1
#define DATEBOOK_EVENT        "untimed_event"
#define DATEBOOK_BEGIN        "start_time"
#define DATEBOOK_END          "end_time"
#define DATEBOOK_ALARM        "use_alarm"
#define DATEBOOK_ADVANCE      "alarm_advance"
#define DATEBOOK_ADV_UNIT     "advance_unit"
#define DATEBOOK_REPEAT_TYPE  "repeat_type"
/** \brief Does the repeat have an end?

If set, any value in repeat-end is ignored.
*/
#define DATEBOOK_REPEAT_FOREVER    "repeat_forever"
#define DATEBOOK_REPEAT_END        "repeat_end"
#define DATEBOOK_REPEAT_FREQUENCY  "repeat_frequency"
#define DATEBOOK_REPEAT_DAY        "repeat_day"
#define DATEBOOK_REPEAT_WEEK_START "repeat_week_start"
#define DATEBOOK_EXCEPTIONS        "exception_count"
#define DATEBOOK_EXCEPTION         "exception_list"
#define DATEBOOK_NOTE              "note"
#define DATEBOOK_CATEGORY          "category"
#define DATEBOOK_KVP_PATH          "datebook/exceptions"

AS_STRING_FUNC_NON_TYPEDEF(repeatTypes, ENUM_REPEAT_LIST)
FROM_STRING_FUNC_NON_TYPEDEF(repeatTypes, ENUM_REPEAT_LIST)

AS_STRING_FUNC_NON_TYPEDEF(DayOfMonthType, ENUM_MONTH_DAYQ)
FROM_STRING_FUNC_NON_TYPEDEF(DayOfMonthType, ENUM_MONTH_DAYQ)

AS_STRING_FUNC_NON_TYPEDEF (alarmTypes, ENUM_ALARM_UNIT)
FROM_STRING_FUNC_NON_TYPEDEF (alarmTypes, ENUM_ALARM_UNIT)

qof_datebook *
datebook_create (QofBook *book)
{
	Appointment_t *qd;
	qof_datebook *obj;
	time_t current;
	QofCollection *coll;
	GList *all;

	obj = g_new0(qof_datebook, 1);
	qof_instance_init (&obj->inst, PILOT_LINK_QOF_DATEBOOK, book);
	coll = qof_book_get_collection (book, PILOT_LINK_QOF_DATEBOOK);
	all = qof_collection_get_data (coll);
	all = g_list_prepend (all, obj);
	qof_collection_set_data (coll, all);
	qd = &obj->wrap;
	current = time (NULL);
	obj->repeater = FALSE;
	localtime_r (&current, &qd->begin);
	qd->end = qd->begin;
	qd->repeatEnd = qd->begin;
	qd->exception = NULL;
	return obj;
}

static gboolean
datebook_getEvent (qof_datebook * d)
{
	Appointment_t *qd;

	g_return_val_if_fail (d != NULL, -1);
	qd = &d->wrap;
	if (qd->event) { return TRUE; }
	return FALSE;
}

static Timespec
datebook_getBegin (qof_datebook * d)
{
	Appointment_t *qd;
	Timespec t;

	t.tv_sec = 0;
	t.tv_nsec = 0;
	g_return_val_if_fail (d != NULL, t);
	qd = &d->wrap;
	timespecFromTime_t (&t, mktime (&qd->begin));
	return t;
}

static Timespec
datebook_getEnd (qof_datebook * d)
{
	Appointment_t *qd;
	Timespec t;

	t.tv_sec = 0;
	t.tv_nsec = 0;
	g_return_val_if_fail (d != NULL, t);
	qd = &d->wrap;
	timespecFromTime_t (&t, mktime (&qd->end));
	return t;
}

static gboolean
datebook_getAlarm (qof_datebook * d)
{
	Appointment_t *qd;

	g_return_val_if_fail (d != NULL, FALSE);
	qd = &d->wrap;
	if(qd->alarm == 1) { return TRUE; }
	return FALSE;
}

static int
datebook_getAdvance (qof_datebook * d)
{
	Appointment_t *qd;

	g_return_val_if_fail (d != NULL, -1);
	qd = &d->wrap;
	if(datebook_getAlarm(d)) {
		return qd->advance;
	}
	return 0;
}

/** \brief Uses ENUM_ALARM_UNIT to convert to a string

The enumerator values are converted to text for XML.
*/
static const char*
datebook_getAdvanceUnit (qof_datebook * d)
{
	Appointment_t *qd;
	char *unit;

	g_return_val_if_fail (d != NULL, NULL);
	qd = &d->wrap;
	if(datebook_getAlarm(d) == FALSE) { return NULL; }
	unit = g_strdup(alarmTypesasString(qd->advanceUnits));
	return unit;
}

static const char*
datebook_getRepeatType (qof_datebook * d)
{
	Appointment_t *qd;
	char *type_label;

	g_return_val_if_fail (d != NULL, NULL);
	qd = &d->wrap;
	type_label = g_strdup(repeatTypesasString(qd->repeatType));
	return type_label;
}

static gboolean
datebook_getRepeatForever (qof_datebook * d)
{
	Appointment_t *qd;

	g_return_val_if_fail (d != NULL, -1);
	qd = &d->wrap;
	if(qd->repeatForever == 1) { return TRUE; }
	return FALSE;
}

static Timespec
datebook_getRepeatEnd (qof_datebook * d)
{
	Appointment_t *qd;
	Timespec t;

	t.tv_sec = 0;
	t.tv_nsec = 0;
	g_return_val_if_fail (d != NULL, t);
	qd = &d->wrap;
	if(qd->repeatType == repeatNone) {
		qd->repeatEnd = qd->end;
	}
	timespecFromTime_t (&t, mktime (&qd->repeatEnd));
	return t;
}

/** \brief How often to repeat

Actual frequency depends on the value of repeatType.
1 == every time that repeat type comes around.
2 == every other repeat.
*/
static int
datebook_getRepeatFrequency (qof_datebook * d)
{
	Appointment_t *qd;

	g_return_val_if_fail (d != NULL, -1);
	qd = &d->wrap;
	return qd->repeatFrequency;
}

static const char*
datebook_getRepeatDay (qof_datebook * d)
{
	Appointment_t *qd;
	char* g;

	g_return_val_if_fail (d != NULL, NULL);
	qd = &d->wrap;
	if(qd->repeatType == repeatNone) { return NULL; }
	g = g_strdup(DayOfMonthTypeasString((enum DayOfMonthType)qd->repeatDay));
	return g;
}

/** \brief Get the locale name of the day to start the week.

Calculate the current day of the week, compare with the integer
set in the preferences and deduce the full weekday name
according to the current locale. 

\note \b Current. QOF deals with the timezone issues but if
	you take your Palm to another country, be prepared for this
	value to be translated!

This value is not used to set the preference in the Palm, so
it does not need to be translated back. Setting the value in
\ref DB only affects any exported XML.

\verbatim
repeatWeekstart contains 0=sunday, 1=monday etc.
Say it's Tues 26th and weekstart is sunday
using %u monday = 1; 
so today = 2, weekstart = 0; 
local_tm.tm_mday; starts at 1.
weekstart is two days ago: (0 - 2) = sunday.
diff == 2, sunday was the 24th.
if(local_tm->tm_mday > diff) { local_tm->tm_mday -= diff;}

in case it's near the start of the month, go to the next week. 
Say Fri 1st. today = 4, weekstart still sunday, 0 
diff now is -4 ( weekstart - today)
move on to the next sunday: 7 + (0 - 4) = 3 
so we use Sunday 3rd. 
else { local_tm->tm_mday += (7 + diff); }
\endverbatim
*/
static char*
datebook_getRepeatWeekStart (qof_datebook * d)
{
	Appointment_t *qd;
	char day[256];
	char *pday;
	time_t local;
	struct tm *local_tm;
	int today, diff;

	g_return_val_if_fail (d != NULL, NULL);
	qd = &d->wrap;
	if(qd->repeatType == repeatNone) { return NULL; }
	diff = 0;
	local = time(NULL);
	local_tm = localtime(&local);
	strftime(day, 255, "%u", local_tm);
	today = atoi(day);
	if(today != qd->repeatWeekstart)
	{
		diff = qd->repeatWeekstart - today - 1;
		if(local_tm->tm_mday > diff) { local_tm->tm_mday -= diff;}
		else { local_tm->tm_mday += (7 + diff); }
	}
	strftime(day, 255, "%A", local_tm);
	pday = g_strdup(day);
	return pday;
}

static int
datebook_getExceptions (qof_datebook * d)
{
	Appointment_t *qd;

	g_return_val_if_fail (d != NULL, -1);
	qd = &d->wrap;
	return qd->exceptions;
}

static char *
datebook_getDescription (qof_datebook * d)
{
	Appointment_t *qd;

	g_return_val_if_fail (d != NULL, NULL);
	qd = &d->wrap;
	return qd->description;
}

static char *
datebook_getNote (qof_datebook * d)
{
	Appointment_t *qd;

	g_return_val_if_fail (d != NULL, NULL);
	qd = &d->wrap;
	return qd->note;
}

static char* 
datebook_getCategory(qof_datebook *d)
{
	g_return_val_if_fail(d != NULL, NULL);
	return d->category;
}

static double
datebook_getDuration (qof_datebook *d)
{
	Timespec ts_diff, ts_start, ts_end;

	g_return_val_if_fail(d, 0);
	ts_start = datebook_getBegin(d);
	ts_end = datebook_getEnd(d);
	ts_diff = timespec_diff(&ts_end, &ts_start);
	return ts_diff.tv_sec ? ts_diff.tv_sec / 3600 : 0;
}

static gboolean
datebook_check_repeater(qof_datebook *d)
{
	g_return_val_if_fail(d, FALSE);
	return d->repeater;
}

static void
datebook_set_repeater(qof_datebook *d, gboolean e)
{
	g_return_if_fail (d != NULL);
	d->repeater = e;	
}

static void
datebook_setEvent (qof_datebook * d, gboolean e)
{
	Appointment_t *qd;

	g_return_if_fail (d != NULL);
	qd = &d->wrap;
	qd->event = 0;
	if(e == TRUE) {
		qd->event = 1;
	}
}

static void
datebook_setBegin (qof_datebook * d, Timespec t)
{
	Appointment_t *qd;
	time_t s;

	g_return_if_fail (d != NULL);
	qd = &d->wrap;
	s = timespecToTime_t (t);
	qd->begin = *localtime (&s);
}

static void
datebook_setEnd (qof_datebook * d, Timespec t)
{
	Appointment_t *qd;
	time_t s;

	g_return_if_fail (d != NULL);
	qd = &d->wrap;
	s = timespecToTime_t (t);
	qd->end = *localtime (&s);
}

static void
datebook_setAlarm (qof_datebook * d, gboolean e)
{
	Appointment_t *qd;

	g_return_if_fail (d != NULL);
	qd = &d->wrap;
	if(e) { qd->alarm = 1; }
}

static void
datebook_setAdvance (qof_datebook * d, int e)
{
	Appointment_t *qd;

	g_return_if_fail (d != NULL);
	qd = &d->wrap;
	qd->advance = e;
}

static void
datebook_setAdvanceUnit (qof_datebook * d, const char *e)
{
	Appointment_t *qd;

	g_return_if_fail (d != NULL);
	if(e == NULL) { return; }
	qd = &d->wrap;
	alarmTypesfromString(e, (enum alarmTypes*)&qd->advanceUnits);
}

/** \brief Uses ENUM_REPEAT_LIST to convert to an enum value.

The XML string is converted back to an enumerator value.
*/
static void
datebook_setRepeatType (qof_datebook * d, const char* type_label)
{
	Appointment_t *qd;

	g_return_if_fail (d != NULL);
	qd = &d->wrap;
	repeatTypesfromString(type_label, (enum repeatTypes*)&qd->repeatType);
}

static void
datebook_setRepeatForever (qof_datebook * d, gboolean e)
{
	Appointment_t *qd;

	g_return_if_fail (d != NULL);
	qd = &d->wrap;
	if(e) { qd->repeatForever = 1; }
}

static void
datebook_setRepeatEnd (qof_datebook * d, Timespec t)
{
	Appointment_t *qd;
	time_t s;

	g_return_if_fail (d != NULL);
	qd = &d->wrap;
	s = timespecToTime_t (t);
	qd->repeatEnd = *localtime (&s);
}

static void
datebook_setRepeatFrequency (qof_datebook * d, int e)
{
	Appointment_t *qd;

	g_return_if_fail (d != NULL);
	qd = &d->wrap;
	qd->repeatFrequency = e;
}

static void
datebook_setRepeatDay (qof_datebook * d, const char *e)
{
	Appointment_t *qd;

	g_return_if_fail (d != NULL);
	if(e == NULL) { return; }
	qd = &d->wrap;
	DayOfMonthTypefromString(e, (enum DayOfMonthType*)&qd->repeatDay);
}

/** \brief Set the day to start the week from the locale name.

If the value is not recognised, QOF sets a default of zero -
which would start the week on Sunday.

However, this value is not synced to the Palm - it is XML only.
*/
static void
datebook_setRepeatWeekStart (qof_datebook * d, char *e)
{
	char day[MAX_DATE_LENGTH];
	Appointment_t *qd;
	time_t local;
	struct tm *local_tm;
	int i, diff;
	gboolean found;

	g_return_if_fail (d != NULL);
	qd = &d->wrap;
	diff = 0;
	found = FALSE;
	local = time(NULL);
	local_tm = localtime(&local);
	if(local_tm->tm_wday <= 7) { diff = 1; }
	else { diff = -1; }
	for(i = 0; i < 7; i++) {
		strftime(day, MAX_DATE_LENGTH, "%A", local_tm);
		if(0 == safe_strcmp(e, day)) { found = TRUE; break; }
		local_tm->tm_mday += diff;
	}
	if(!found) { i = 0; }
	qd->repeatWeekstart = i;
}

static void
datebook_setExceptions (qof_datebook * d, int e)
{
	Appointment_t *qd;

	g_return_if_fail (d != NULL);
	qd = &d->wrap;
	qd->exceptions = e;
}

static void
datebook_setDescription (qof_datebook * d, char *h)
{
	Appointment_t *qd;

	g_return_if_fail (d);
	qd = &d->wrap;
	qd->description = g_strdup (h);
}

static void
datebook_setNote (qof_datebook * d, char *h)
{
	Appointment_t *qd;

	g_return_if_fail (d);
	qd = &d->wrap;
	if(!h) { return; }
	qd->note = g_locale_to_utf8 (h, -1, NULL, NULL, NULL);
}

static void
datebook_setCategory(qof_datebook *d, char *n)
{
	g_return_if_fail(d != NULL);
	d->category = g_strdup(n);
}

int
datebook_unpack(QofEntity *ent, gpointer user_data)
{
	pi_buffer_t   *pi_buf;
	Appointment_t *qa;
	qof_datebook  *obj, *clone;
	QofInstance   *inst;
	QofBook       *book;
	KvpFrame      *inst_frame;
	Timespec      ts, ts_increment, ts_end, repeat_end;
	time_t        db_t;
	pilot_qof_data *context;
	gint size, i, day_interval, month_interval;

	context = (pilot_qof_data*)user_data;
	ts.tv_sec = 0;
	ts.tv_nsec = 0;
	day_interval = 0;
	month_interval = 0;
	g_return_val_if_fail(context != NULL, -1);
	g_return_val_if_fail(ent != NULL, -1);
	obj = (qof_datebook*)ent;
	inst = (QofInstance*)ent;
	inst_frame = qof_instance_get_slots(inst);
	qa = &obj->wrap;
	pi_buf = (pi_buffer_t*)context->pi_buf;
	size = 0;
	size = unpack_Appointment(qa, pi_buf, DATEBOOK_VERSION); // 0.12
	datebook_setCategory(obj, context->pi_cat.name[context->ent_category]);
	/* Use <= and handle zero if < omits last entry (&623)*/
	for (i = 0; i < qa->exceptions; i++)
	{
		gchar *extend;

		extend = NULL;
		db_t = mktime(&qa->exception[i]);
		timespecFromTime_t(&ts, db_t);
		extend = g_strdup_printf("%s/%d", DATEBOOK_KVP_PATH, i + 1);
		kvp_frame_set_timespec(inst_frame, extend, ts);
		inst->kvp_data = inst_frame;
		g_free(extend);
	}
	if(qa->repeatType == repeatNone) {
		qa->repeatEnd = qa->end;
	}
	/* set a marker for the interval. Do the iteration once, outside the switch.*/
	switch (qa->repeatType) {
		case repeatNone   : { day_interval = 0; month_interval = 0; break; }
		case repeatDaily  : { day_interval = 1; month_interval = 0; break; }
		case repeatWeekly : { day_interval = 7; month_interval = 0; break; }
		case repeatMonthlyByDay :
		case repeatMonthlyByDate :
							{ day_interval = 0; month_interval = 1; break; }
		case repeatYearly : { day_interval = 0; month_interval = 12; break; }
		default : { PERR (" unsupported repeatType=%d", qa->repeatType); }
	}
	if(day_interval == 0 && month_interval == 0) { return size; }
	/* Now create a repeater in the SAME book. */
	ts = datebook_getBegin(obj);
	ts_end = datebook_getEnd(obj);
	if(qa->repeatForever == 0) { repeat_end = datebook_getRepeatEnd(obj); }
	else {
		/* 	if qa->repeatForever == 1 (true), calculate year and a day from
		start_time. qof_date_add_months(12)
		qof_date_add_days(1). Store for use as repeatEnd
		*/
		repeat_end = ts;
		qof_date_add_months(&repeat_end, 12, FALSE);
		qof_date_add_days(&repeat_end, 1);
	}
	ts_increment = ts;
	/*  qa->exception is an array of struct tm* qa->exceptions long. */
	/* while datebook_getBegin is less (earlier) than repeat_end */
	while(timespec_cmp(&ts_increment, &repeat_end) < 0) {
		gboolean skip;

		skip = FALSE;
		if(day_interval)   { 
			qof_date_add_days(&ts_increment, day_interval);
			qof_date_add_days(&ts_end, day_interval);
		}
		if(month_interval) { 
			qof_date_add_months(&ts_increment, month_interval, FALSE); 
			qof_date_add_months(&ts_end, month_interval, FALSE);
		}
		for(i=0; i < qa->exceptions;i++)
		{
			time_t t;
			struct tm check;

			t = timespecToTime_t(ts_increment);
			check = *gmtime_r(&t, &check);
			if((check.tm_year == qa->exception[i].tm_year) &&
				(check.tm_mday == qa->exception[i].tm_mday) &&
				(check.tm_mon == qa->exception[i].tm_mon))
			{
				/* exclude */
				skip = TRUE;
			}
		}
		/* create the repeater */
		if(!skip)
		{
			book = inst->book;
			clone = datebook_create(book);
			clone->repeater = TRUE;
			/* clone the main parts of the event */
			datebook_setEvent(clone, datebook_getEvent(obj));
			datebook_setAlarm(clone, datebook_getAlarm(obj));
			datebook_setAdvance(clone, datebook_getAdvance(obj));
			datebook_setAdvanceUnit(clone, datebook_getAdvanceUnit(obj));
			datebook_setDescription(clone, datebook_getDescription(obj));
			datebook_setNote(clone, datebook_getNote(obj));
			datebook_setCategory(clone, datebook_getCategory(obj));
			/* the clone does not repeat */
			datebook_setExceptions(clone, 0);
			datebook_setRepeatEnd(clone, ts_end);
			datebook_setRepeatForever(clone, FALSE);
			datebook_setRepeatType(clone, repeatTypesasString(repeatNone));
			datebook_setBegin(clone, ts_increment);
			datebook_setEnd(clone, ts_end);
		}
	}
	return size;
}

int 
datebook_pack(QofEntity *ent, gpointer user_data)
{
	pilot_qof_data *context;
	Appointment_t  *qa;
	qof_datebook   *obj;
	Timespec       ts, tss;
	KvpFrame       *frame;
	gint           size, i;
	time_t         t;
	gchar          *path;

	size = 0;
	i = 1;
	ts.tv_sec = 0;
	ts.tv_nsec = 0;
	tss.tv_sec = 0;
	tss.tv_nsec = 0;
	context = (pilot_qof_data*)user_data;
	g_return_val_if_fail((context || ent), -1);
	ENTER (" ");
	obj = (qof_datebook*)ent;
	if(obj->repeater == TRUE) { return 0; }
	qa = &obj->wrap;
	size = pack_Appointment(qa, context->pi_buf, DATEBOOK_VERSION); // 0.12
	/* pack slots into exceptions */
	frame = qof_instance_get_slots((QofInstance*)ent);
	if(frame)
	{
		path = g_strdup_printf("%s/%d", DATEBOOK_KVP_PATH, i);
		while(kvp_frame_get_value(frame, path))
		{
			/* get Timespec */
			ts = kvp_frame_get_timespec(frame, path);
			if(!timespec_equal(&ts, &tss)) {
				t = timespecToTime_t(ts);
				qa->exception[i] = *gmtime_r(&t, &qa->exception[i]);
			}
			g_free(path);
			i++;
			path = g_strdup_printf("%s/%d", DATEBOOK_KVP_PATH, i);
		}
		g_free(path);
	}
	LEAVE (" ");
	return size;
}

int
datebook_appinfo_unpack(QofEntity *ent, gpointer user_data)
{
	AppointmentAppInfo_t app_db;
	Appointment_t     *qa;
	pilot_qof_data *context;

	context = (pilot_qof_data*)user_data;
	g_return_val_if_fail(context != NULL, -1);
	ENTER (" ");
	qa = &((qof_datebook*)ent)->wrap;
	unpack_AppointmentAppInfo(&app_db, context->app_buf->data, context->app_buf->used);
	context->pi_cat = app_db.category;
	if(app_db.startOfWeek) { qa->repeatWeekstart = app_db.startOfWeek; }
	LEAVE (" ");
	return 0;
}

int
qof_datebook_free(QofEntity *ent, gpointer user_data)
{
	Appointment_t *qd;
	qof_datebook *obj;

	g_return_val_if_fail(ent != NULL, -1);
	ENTER (" ");
	obj = (qof_datebook*)ent;
	qd = &obj->wrap;
	free_Appointment(qd);
	LEAVE (" ");
	return 0;
}

static const char*
datebookPrintable (gpointer instance)
{
	return datebook_getDescription((qof_datebook*)instance);
}

static QofObject datebook_object_def = {
      interface_version: QOF_OBJECT_VERSION,
      e_type:            PILOT_LINK_QOF_DATEBOOK,
      type_label:        QOF_DATEBOOK_DESC,
      create:            (gpointer) datebook_create,
      book_begin:        NULL,
      book_end:          NULL,
      is_dirty:          qof_collection_is_dirty,
      mark_clean:        qof_collection_mark_clean,
      foreach:           qof_collection_foreach,
      printable:         datebookPrintable,
      version_cmp:       (int (*)(gpointer, gpointer))qof_instance_version_cmp,
};

static pilot_pack datebook_pack_def = {
        e_type:          PILOT_LINK_QOF_DATEBOOK,
        pack_func:       datebook_pack,
        unpack_func:     datebook_unpack,
        free_pack_func:   qof_datebook_free,
        palm_db_name:    "DatebookDB",
        app_info_unpack: datebook_appinfo_unpack,
};

gboolean
DateBookRegister (void)
{
	static QofParam params[] = {
		{DATEBOOK_EVENT, QOF_TYPE_BOOLEAN, (QofAccessFunc) datebook_getEvent, 
			(QofSetterFunc) datebook_setEvent},
		{DATEBOOK_BEGIN, QOF_TYPE_DATE,    (QofAccessFunc) datebook_getBegin, 
			(QofSetterFunc) datebook_setBegin},
		{DATEBOOK_END,   QOF_TYPE_DATE,    (QofAccessFunc) datebook_getEnd, 
			(QofSetterFunc) datebook_setEnd},
		{DATEBOOK_ALARM, QOF_TYPE_BOOLEAN, (QofAccessFunc) datebook_getAlarm, 
			(QofSetterFunc) datebook_setAlarm},
		{DATEBOOK_ADVANCE, QOF_TYPE_INT32, (QofAccessFunc) datebook_getAdvance, 
			(QofSetterFunc) datebook_setAdvance},
		{DATEBOOK_ADV_UNIT, QOF_TYPE_STRING, (QofAccessFunc) datebook_getAdvanceUnit,
		 (QofSetterFunc) datebook_setAdvanceUnit},
		{DATEBOOK_REPEAT_TYPE, QOF_TYPE_STRING, (QofAccessFunc) datebook_getRepeatType,
		 (QofSetterFunc) datebook_setRepeatType},
		{DATEBOOK_REPEAT_FOREVER, QOF_TYPE_BOOLEAN, 
			(QofAccessFunc) datebook_getRepeatForever,
		 (QofSetterFunc) datebook_setRepeatForever},
		{DATEBOOK_REPEAT_END, QOF_TYPE_DATE, (QofAccessFunc) datebook_getRepeatEnd,
			(QofSetterFunc) datebook_setRepeatEnd},
		{DATEBOOK_REPEAT_FREQUENCY, QOF_TYPE_INT32, 
			(QofAccessFunc) datebook_getRepeatFrequency,
			(QofSetterFunc) datebook_setRepeatFrequency},
		{DATEBOOK_REPEAT_DAY, QOF_TYPE_STRING, (QofAccessFunc) datebook_getRepeatDay,
		 (QofSetterFunc) datebook_setRepeatDay},
		{DATEBOOK_REPEAT_WEEK_START, QOF_TYPE_STRING, 
			(QofAccessFunc) datebook_getRepeatWeekStart, 
			(QofSetterFunc) datebook_setRepeatWeekStart},
		{DATEBOOK_EXCEPTIONS, QOF_TYPE_INT32, (QofAccessFunc) datebook_getExceptions,
		 (QofSetterFunc) datebook_setExceptions},
		{DATEBOOK_DESCRIPTION, QOF_TYPE_STRING, (QofAccessFunc) datebook_getDescription,
		 (QofSetterFunc) datebook_setDescription},
		{DATEBOOK_CATEGORY, QOF_TYPE_STRING, (QofAccessFunc) datebook_getCategory,
		 (QofSetterFunc) datebook_setCategory},
		{DATEBOOK_REPEATER, QOF_TYPE_BOOLEAN, 
			(QofAccessFunc) datebook_check_repeater, 
			(QofSetterFunc) datebook_set_repeater  },
		{DATEBOOK_NOTE, QOF_TYPE_STRING, (QofAccessFunc) datebook_getNote,
		 (QofSetterFunc) datebook_setNote},
		{DATEBOOK_DURATION, QOF_TYPE_DOUBLE, (QofAccessFunc) datebook_getDuration, NULL },
		{DATEBOOK_EXCEPTION, QOF_TYPE_KVP, (QofAccessFunc) qof_instance_get_slots, NULL },
		{QOF_PARAM_BOOK, QOF_ID_BOOK, (QofAccessFunc) qof_instance_get_book, NULL},
		{QOF_PARAM_GUID, QOF_TYPE_GUID, (QofAccessFunc) qof_instance_get_guid, NULL},
		{NULL},
	};

	qof_class_register (PILOT_LINK_QOF_DATEBOOK, NULL, params);

	pilot_qof_pack_register (&datebook_pack_def);

	return qof_object_register (&datebook_object_def);
}
