/***************************************************************************
 *            pilot-todo.c
 *
 *  Sun Jan 30 16:07:21 2005
 *  Copyright  2005  Neil Williams
 *  linux@codehelp.co.uk
 ****************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
 
#include "pilot-todo.h"
#include "pilot-qof.h"

#define TODO_VERSION todo_v1
#define QOF_TODO_DESC "Pilot-Link QOF ToDo"
#define TODO_LENGTH   "todo_length"
#define TODO_DATE     "date_due"
#define TODO_PRIORITY "todo_priority"
#define TODO_COMPLETE "todo_complete"
#define TODO_DESCRIPTION "todo_description"
#define TODO_NOTE     "todo_note"
#define TODO_CATEGORY "category"

static gchar* log_module = PQ_MOD_PILOT;

qof_todo* todo_create(QofBook *book)
{
	ToDo_t *qt;
	qof_todo *obj;
	time_t current;
	QofCollection *coll;
	GList *all;

	obj = g_new0(qof_todo, 1);
	qof_instance_init (&obj->inst, PILOT_LINK_QOF_TODO, book);
	coll = qof_book_get_collection (book, PILOT_LINK_QOF_TODO);
	all = qof_collection_get_data (coll);
	all = g_list_prepend (all, obj);
	qof_collection_set_data (coll, all);
	qt = &obj->wrap;
	current = time(NULL);
	localtime_r(&current, &qt->due);
	return obj;
}

static gint 
todo_getLength(qof_todo *t)
{
	ToDo_t *qt;
	
	g_return_val_if_fail(t != NULL, 0);
	qt = &t->wrap;
	return qt->indefinite;
}

static Timespec 
todo_getDateDue(qof_todo *t)
{
	ToDo_t *qt;
	Timespec ts;
	
	ts.tv_sec = 0;
	ts.tv_nsec = 0;
	g_return_val_if_fail(t != NULL, ts);
	qt = &t->wrap;
	timespecFromTime_t(&ts, mktime(&qt->due));
	return ts;
}

static gint 
todo_getPriority(qof_todo *t)
{
	ToDo_t *qt;

	g_return_val_if_fail(t != NULL, 0);
	qt = &t->wrap;
	return qt->priority;
}

static gint 
todo_getComplete(qof_todo *t)
{
	ToDo_t *qt;
	
	g_return_val_if_fail(t != NULL, 0);
	qt = &t->wrap;
	return qt->complete;
}

static char* 
todo_getDescription(qof_todo *t)
{
	ToDo_t *qt;
	
	g_return_val_if_fail(t != NULL, NULL);
	qt = &t->wrap;
	return qt->description;
}

static char* 
todo_getNote(qof_todo *t)
{
	ToDo_t *qt;
	
	g_return_val_if_fail(t != NULL, NULL);
	qt = &t->wrap;
	return qt->note;
}

static char* 
todo_getCategory(qof_todo *t)
{
	g_return_val_if_fail(t != NULL, NULL);
	return t->category;
}

static void 
todo_setLength(qof_todo *t, gint l)
{
	ToDo_t *qt;
	
	g_return_if_fail(t != NULL);
	qt = &t->wrap;
	qt->indefinite = l;
}

static void 
todo_setDateDue(qof_todo *t, Timespec ts)
{
	ToDo_t *qt;
	time_t s;

	g_return_if_fail(t != NULL);
	qt = &t->wrap;
	s = timespecToTime_t(ts);
	qt->due = *localtime(&s);
}

static void 
todo_setPriority(qof_todo *t, gint p)
{
	ToDo_t *qt;
	
	g_return_if_fail(t != NULL);
	if((p < 1)||(p >5)) { p = 1; }
	qt = &t->wrap;
	qt->priority = p;
}

static void 
todo_setComplete(qof_todo *t, gint c)
{
	ToDo_t *qt;
	
	g_return_if_fail(t != NULL);
	qt = &t->wrap;
	qt->complete = c;
}

static void 
todo_setDescription(qof_todo *t, const char *d)
{
	ToDo_t *qt;
	
	g_return_if_fail(t != NULL);
	qt = &t->wrap;
	qt->description = g_strdup(d);
}

static void 
todo_setNote(qof_todo *t, const char *n)
{
	ToDo_t *qt;
	
	g_return_if_fail(t != NULL);
	qt = &t->wrap;
	qt->note = g_strdup(n);
}

static void 
todo_setCategory(qof_todo *t, const char *n)
{
	g_return_if_fail(t != NULL);
	t->category = g_strdup(n);
}

int
todo_unpack(QofEntity *ent, gpointer user_data)
{
	pi_buffer_t *pi_buf;
	ToDo_t      *qt;
	qof_todo    *obj;
	pilot_qof_data *context;
	int size;

	context = (pilot_qof_data*)user_data;
	g_return_val_if_fail(context != NULL, -1);
	g_return_val_if_fail(ent != NULL, -1);
	ENTER (" ");
	obj = (qof_todo*)ent;
	qt = &obj->wrap;
	pi_buf = (pi_buffer_t*)context->pi_buf;
	size = 0;
	size = unpack_ToDo(qt, pi_buf, TODO_VERSION);
	todo_setCategory(obj, context->pi_cat.name[context->ent_category]);
	LEAVE (" ");
	return size;
}

int 
todo_pack(QofEntity *ent, gpointer user_data)
{
	ToDo_t       *qt;
	qof_todo     *obj;
	pilot_qof_data *context;
	int size;

	size = 0;
	context = (pilot_qof_data*)user_data;
	g_return_val_if_fail((context || ent), -1);
	ENTER (" ");
	obj = (qof_todo*)ent;
	qt = &obj->wrap;
	size = pack_ToDo(qt, context->pi_buf, TODO_VERSION);
	LEAVE (" ");
	return size;
}

int
qof_todo_free(QofEntity *ent, gpointer user_data)
{
	ToDo_t *qt;
	qof_todo *obj;

	g_return_val_if_fail(ent != NULL, -1);
	ENTER (" ");
	obj = (qof_todo*)ent;
	qt = &obj->wrap;
	free_ToDo(qt);
	LEAVE (" ");
	return 0;
}

int
todo_appinfo_unpack(QofEntity *ent, gpointer user_data)
{
	ToDoAppInfo_t app_t;
	pilot_qof_data *context;

	context = (pilot_qof_data*)user_data;
	g_return_val_if_fail(context != NULL, -1);
	ENTER (" ");
	unpack_ToDoAppInfo(&app_t, context->app_buf->data, context->app_buf->used);
	context->pi_cat = app_t.category;
	LEAVE (" ");
	return 0;
}

static const char*
todoPrintable (gpointer instance)
{
	return todo_getDescription((qof_todo*)instance);
}

static QofObject todo_object_def = {
      interface_version: QOF_OBJECT_VERSION,
      e_type:            PILOT_LINK_QOF_TODO,
      type_label:        QOF_TODO_DESC,
      create:            (gpointer)todo_create,
      book_begin:        NULL,
      book_end:          NULL,
      is_dirty:          qof_collection_is_dirty,
      mark_clean:        qof_collection_mark_clean,
      foreach:           qof_collection_foreach,
      printable:         todoPrintable,
      version_cmp:       (int (*)(gpointer,gpointer)) qof_instance_version_cmp,
};

static pilot_pack todo_pack_def = {
        e_type:          PILOT_LINK_QOF_TODO,
        pack_func:       todo_pack,
        unpack_func:     todo_unpack,
        free_pack_func:   qof_todo_free,
        palm_db_name:    "ToDoDB",
        app_info_unpack: todo_appinfo_unpack,
};

gboolean ToDoRegister (void)
{
	static QofParam params[] = {
	{ TODO_LENGTH,    QOF_TYPE_INT32,  (QofAccessFunc)todo_getLength,   (QofSetterFunc)todo_setLength },
	{ TODO_DATE,      QOF_TYPE_DATE,   (QofAccessFunc)todo_getDateDue,  (QofSetterFunc)todo_setDateDue },
	{ TODO_PRIORITY,  QOF_TYPE_INT32,  (QofAccessFunc)todo_getPriority, (QofSetterFunc)todo_setPriority },
	{ TODO_COMPLETE,  QOF_TYPE_INT32,  (QofAccessFunc)todo_getComplete, (QofSetterFunc)todo_setComplete },
	{ TODO_DESCRIPTION, QOF_TYPE_STRING, (QofAccessFunc)todo_getDescription, (QofSetterFunc)todo_setDescription },
	{ TODO_CATEGORY,  QOF_TYPE_STRING, (QofAccessFunc)todo_getCategory, (QofSetterFunc)todo_setCategory },
	{ TODO_NOTE,      QOF_TYPE_STRING, (QofAccessFunc)todo_getNote,     (QofSetterFunc)todo_setNote },
	{ QOF_PARAM_BOOK, QOF_ID_BOOK,     (QofAccessFunc)qof_instance_get_book, NULL },
	{ QOF_PARAM_GUID, QOF_TYPE_GUID,   (QofAccessFunc)qof_instance_get_guid, NULL },
	{ NULL },
	};
	qof_class_register (PILOT_LINK_QOF_TODO, NULL, params);

	pilot_qof_pack_register (&todo_pack_def);

	return qof_object_register (&todo_object_def);
}
