/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL plugin
 *
 * Copyright © 2006, 2007 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Mirco Müller <macslow@bangang.de>
 */

/*
 * PgmGlText uses the Pango text rendering pipeline to generate a pixmap with
 * the size of the drawable and then bind it as a texture. Each time a property
 * is changed, the pixmap is updated. When the drawable size is changed, the
 * pixmap is reallocated to this new one and then updated.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <math.h>
#include "pgmgltext.h"

GST_DEBUG_CATEGORY_EXTERN (pgm_gl_debug);
#define GST_CAT_DEFAULT pgm_gl_debug

static PgmGlDrawableClass *parent_class = NULL;

/* Private methods */

/* Gets the projected height of the drawable on the viewport */
static gfloat
get_projected_height (PgmGlDrawable *gldrawable,
                      gfloat canvas_height)
{
  PgmGlViewport *glviewport = gldrawable->glviewport;
  PgmViewport *viewport = PGM_VIEWPORT (glviewport);
  PgmCanvas *canvas = viewport->canvas;
  gfloat height;

  /* FIXME: That's just the projected height on z=0 */
  GST_OBJECT_LOCK (canvas);
  GST_OBJECT_LOCK (viewport);
  height = (canvas_height * viewport->projected_height) / canvas->height;
  GST_OBJECT_UNLOCK (viewport);
  GST_OBJECT_UNLOCK (canvas);

  return height;
}

/* Gets the projected width of the drawable on the viewport */
static gfloat
get_projected_width (PgmGlDrawable *gldrawable,
                           gfloat canvas_width)
{
  PgmGlViewport *glviewport = gldrawable->glviewport;
  PgmViewport *viewport = PGM_VIEWPORT (glviewport);
  PgmCanvas *canvas = viewport->canvas;
  gfloat width;

  /* FIXME: That's just the projected width on z=0 */
  GST_OBJECT_LOCK (canvas);
  GST_OBJECT_LOCK (viewport);
  width = (canvas_width * viewport->projected_width) / canvas->width;
  GST_OBJECT_UNLOCK (viewport);
  GST_OBJECT_UNLOCK (canvas);

  return width;
}

/* Defines texture coordinates */
static void
set_coordinates (PgmGlText *gltext)
{
  PgmTexture *texture = gltext->texture;

  GST_OBJECT_LOCK (gltext);
  PGM_TEXTURE_LOCK (texture);

  gltext->coord[2] = (PgmGlFloat) texture->width / texture->width_pot;
  gltext->coord[4] = gltext->coord[2];
  gltext->coord[5] = (PgmGlFloat) texture->height / texture->height_pot;
  gltext->coord[7] = gltext->coord[5];

  PGM_TEXTURE_UNLOCK (texture);
  GST_OBJECT_UNLOCK (gltext);
}

/* Stores outline width depending on PgmText outline width converting it
 * from canvas coordinates to viewport coordinates */
static void
set_outline_width (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);
  gfloat outline_width;

  outline_width = get_projected_height (gldrawable, text->outline_width);

  GST_OBJECT_LOCK (gltext);
  gltext->outline_width = outline_width;
  GST_OBJECT_UNLOCK (gltext);
}

/* Stores outline color depending on PgmText color normalizing components to
 * the range [0.0; 0.1] */
static void
set_outline_color (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  GST_OBJECT_LOCK (text);
  GST_OBJECT_LOCK (gltext);

  gltext->outline_color[0] = text->outline_r * INV_255;
  gltext->outline_color[1] = text->outline_g * INV_255;
  gltext->outline_color[2] = text->outline_b * INV_255;
  gltext->outline_color[3] = text->outline_a * INV_255;

  GST_OBJECT_UNLOCK (gltext);
  GST_OBJECT_UNLOCK (text);
}

/* Defines Pango spacing depending on PgmText line spacing converting it
 * from canvas coordinates to viewport coordinates */
static void
set_line_spacing (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);
  gfloat line_spacing;

  line_spacing = get_projected_height (gldrawable, text->line_spacing);

  GST_OBJECT_LOCK (gltext);
  pango_layout_set_spacing (gltext->layout, line_spacing * PANGO_SCALE);
  GST_OBJECT_UNLOCK (gltext);
}

/* Defines Pango font size depending on PgmText height converting it from
 * canvas coordinates to viewport coordinates */
static void
set_font_height (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);
  gfloat height;

  height = get_projected_height (gldrawable, text->height);

  GST_OBJECT_LOCK (gltext);

  pango_font_description_set_absolute_size (gltext->desc, height * PANGO_SCALE);
  pango_layout_set_font_description (gltext->layout, gltext->desc);

  GST_OBJECT_UNLOCK (gltext);
}

/* Defines Pango font family depending on PgmText font family */
static void
set_font_family (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  GST_OBJECT_LOCK (text);
  GST_OBJECT_LOCK (gltext);

  pango_font_description_set_family (gltext->desc, text->font_family);
  pango_layout_set_font_description (gltext->layout, gltext->desc);

  GST_OBJECT_UNLOCK (gltext);
  GST_OBJECT_UNLOCK (text);
}

/* Defines Pango text/markup depending on PgmText label */
static void
set_label (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  GST_OBJECT_LOCK (text);
  GST_OBJECT_LOCK (gltext);

  /* FIXME: defining a markup and right after a label keeps the
   *        previous markups on the new label. */
  if (text->use_markup)
    pango_layout_set_markup (gltext->layout, text->label, -1);
  else
    pango_layout_set_text (gltext->layout, text->label, -1);

  GST_OBJECT_UNLOCK (gltext);
  GST_OBJECT_UNLOCK (text);
}

/* Defines Pango width depending on PgmText width */
static void
set_width (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmViewport *viewport = PGM_VIEWPORT (gldrawable->glviewport);
  gfloat non_square_ratio;
  gfloat width;

  width = get_projected_width (gldrawable, gldrawable->width);

  GST_OBJECT_LOCK (gltext);

  GST_OBJECT_LOCK (viewport);
  non_square_ratio = 1.0f / viewport->pixel_aspect_ratio;
  GST_OBJECT_UNLOCK (viewport);

  pango_layout_set_width (gltext->layout,
                          width * PANGO_SCALE * non_square_ratio);

  GST_OBJECT_UNLOCK (gltext);
}

/* Defines Pango justification depending on PgmText justification */
static void
set_justify (PgmGlText *gltext)
{
#ifdef HAVE_PANGO_1_16
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  GST_OBJECT_LOCK (text);
  GST_OBJECT_LOCK (gltext);

  pango_layout_set_justify (gltext->layout, text->justify);

  GST_OBJECT_UNLOCK (gltext);
  GST_OBJECT_UNLOCK (text);
#endif /* HAVE_PANGO_1_16 */
}

/* Defines Pango weight depending on PgmText weight */
static void
set_weight (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  GST_OBJECT_LOCK (gltext);

  switch (text->weight)
    {
    case PGM_TEXT_WEIGHT_LIGHT:
      pango_font_description_set_weight (gltext->desc, PANGO_WEIGHT_LIGHT);
      break;

    case PGM_TEXT_WEIGHT_NORMAL:
      pango_font_description_set_weight (gltext->desc, PANGO_WEIGHT_NORMAL);
      break;

    case PGM_TEXT_WEIGHT_BOLD:
      pango_font_description_set_weight (gltext->desc, PANGO_WEIGHT_BOLD);
      break;

    default:
      break;
    }

  pango_layout_set_font_description (gltext->layout, gltext->desc);

  GST_OBJECT_UNLOCK (gltext);
}

/* Defines Pango stretching depending on PgmText stretching */
static void
set_stretch (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  GST_OBJECT_LOCK (gltext);

  switch (text->stretch)
    {
    case PGM_TEXT_STRETCH_CONDENSED:
      pango_font_description_set_stretch (gltext->desc,
                                          PANGO_STRETCH_CONDENSED);
      break;

    case PGM_TEXT_STRETCH_NORMAL:
      pango_font_description_set_stretch (gltext->desc, PANGO_STRETCH_NORMAL);
      break;

    case PGM_TEXT_STRETCH_EXPANDED:
      pango_font_description_set_stretch (gltext->desc,
                                          PANGO_STRETCH_EXPANDED);
      break;

    default:
      break;
    }

  pango_layout_set_font_description (gltext->layout, gltext->desc);

  GST_OBJECT_UNLOCK (gltext);
}

/* Defines Pango alignment depending on PgmText alignment */
static void
set_alignment (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  GST_OBJECT_LOCK (gltext);

  switch (text->alignment)
    {
    case PGM_TEXT_ALIGN_LEFT:
      pango_layout_set_alignment (gltext->layout, PANGO_ALIGN_LEFT);
      break;

    case PGM_TEXT_ALIGN_CENTER:
      pango_layout_set_alignment (gltext->layout, PANGO_ALIGN_CENTER);
      break;

    case PGM_TEXT_ALIGN_RIGHT:
      pango_layout_set_alignment (gltext->layout, PANGO_ALIGN_RIGHT);
      break;

    default:
      break;
    }

  GST_OBJECT_UNLOCK (gltext);
}

/* Defines Pango ellipsizing depending on PgmText ellipsizing */
static void
set_ellipsize (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  GST_OBJECT_LOCK (gltext);

  switch (text->ellipsize)
    {
    case PGM_TEXT_ELLIPSIZE_NONE:
      pango_layout_set_ellipsize (gltext->layout, PANGO_ELLIPSIZE_NONE);
      break;

    case PGM_TEXT_ELLIPSIZE_START:
      pango_layout_set_ellipsize (gltext->layout, PANGO_ELLIPSIZE_START);
      break;

    case PGM_TEXT_ELLIPSIZE_MIDDLE:
      pango_layout_set_ellipsize (gltext->layout, PANGO_ELLIPSIZE_MIDDLE);
      break;

    case PGM_TEXT_ELLIPSIZE_END:
      pango_layout_set_ellipsize (gltext->layout, PANGO_ELLIPSIZE_END);
      break;

    default:
      break;
    }

  GST_OBJECT_UNLOCK (gltext);
}

/* Defines Pango wrapping depending on PgmText wrapping */
static void
set_wrap (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  GST_OBJECT_LOCK (gltext);

  switch (text->wrap)
    {
    case PGM_TEXT_WRAP_WORD:
      pango_layout_set_wrap (gltext->layout, PANGO_WRAP_WORD);
      break;

    case PGM_TEXT_WRAP_CHAR:
      pango_layout_set_wrap (gltext->layout, PANGO_WRAP_CHAR);
      break;

    case PGM_TEXT_WRAP_WORD_CHAR:
      pango_layout_set_wrap (gltext->layout, PANGO_WRAP_WORD_CHAR);
      break;

    default:
      break;
    }

  GST_OBJECT_UNLOCK (gltext);
}

/* Defines Pango gravity depending on PgmText gravity */
static void
set_gravity (PgmGlText *gltext)
{
#ifdef HAVE_PANGO_1_16
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  GST_OBJECT_LOCK (gltext);

  switch (text->gravity)
    {
    case PGM_TEXT_GRAVITY_SOUTH:
      pango_context_set_base_gravity (gltext->pango_ctx, PANGO_GRAVITY_SOUTH);
      break;

    case PGM_TEXT_GRAVITY_EAST:
      pango_context_set_base_gravity (gltext->pango_ctx, PANGO_GRAVITY_EAST);
      break;

    case PGM_TEXT_GRAVITY_NORTH:
      pango_context_set_base_gravity (gltext->pango_ctx, PANGO_GRAVITY_NORTH);
      break;

    case PGM_TEXT_GRAVITY_WEST:
      pango_context_set_base_gravity (gltext->pango_ctx, PANGO_GRAVITY_WEST);
      break;

    case PGM_TEXT_GRAVITY_AUTO:
      pango_context_set_base_gravity (gltext->pango_ctx, PANGO_GRAVITY_AUTO);
      break;

    default:
      break;
    }

  GST_OBJECT_UNLOCK (gltext);
#endif /* HAVE_PANGO_1_16 */
}

/* Defines Pango style depending on PgmText style */
static void
set_style (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  GST_OBJECT_LOCK (gltext);

  switch (text->style)
    {
    case PGM_TEXT_STYLE_NORMAL:
      pango_font_description_set_style (gltext->desc, PANGO_STYLE_NORMAL);
      break;

    case PGM_TEXT_STYLE_OBLIQUE:
      pango_font_description_set_style (gltext->desc, PANGO_STYLE_OBLIQUE);
      break;

    case PGM_TEXT_STYLE_ITALIC:
      pango_font_description_set_style (gltext->desc, PANGO_STYLE_ITALIC);
      break;

    default:
      break;
    }

  pango_layout_set_font_description (gltext->layout, gltext->desc);

  GST_OBJECT_UNLOCK (gltext);
}

/* Defines Pango variant depending on PgmText variant */
static void
set_variant (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  GST_OBJECT_LOCK (gltext);

  switch (text->variant)
    {
    case PGM_TEXT_VARIANT_NORMAL:
      pango_font_description_set_variant (gltext->desc, PANGO_VARIANT_NORMAL);
      break;

    case PGM_TEXT_VARIANT_SMALL_CAPS:
      pango_font_description_set_variant (gltext->desc,
                                          PANGO_VARIANT_SMALL_CAPS);
      break;

    default:
      break;
    }

  pango_layout_set_font_description (gltext->layout, gltext->desc);

  GST_OBJECT_UNLOCK (gltext);
}

/* Update all the properties of the text */
static void
update_properties (PgmGlText *gltext)
{
  set_font_height (gltext);
  set_font_family (gltext);
  set_stretch (gltext);
  set_style (gltext);
  set_variant (gltext);
  set_alignment (gltext);
  set_ellipsize (gltext);
  set_justify (gltext);
  set_line_spacing (gltext);
  set_width (gltext);
  set_wrap (gltext);
  set_gravity (gltext);
  set_label (gltext);
  set_outline_width (gltext);
  set_outline_color (gltext);
}

/* Cairo uses premultiplied-alpha to represent pixels (mostly for compositing).
 * This function unpremultiplies the alpha of the native endian ARGB buffer. */
static void
unpremultiply_alpha (guint32 *buffer,
                     guint width,
                     guint height)
{
  guint size = width * height;
  guint8 a, r, g, b;
  guint i;

  for (i = 0; i < size; i++)
    {
      a = (buffer[i] & 0xff000000) >> 24;
      if (a > 0 && a < 255)
        {
          r = (((buffer[i] & 0x00ff0000) >> 16) * 255 + a / 2) / a;
          g = (((buffer[i] & 0x0000ff00) >>  8) * 255 + a / 2) / a;
          b = (((buffer[i] & 0x000000ff) >>  0) * 255 + a / 2) / a;
          buffer[i] = a << 24 | r << 16 | g << 8 | b;
        }
    }
}

/* Allocates the pixmap, the Pango pipeline using a Cairo backend and
 * generates the texture */
static void
create_pixmap (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmViewport *viewport = PGM_VIEWPORT (gldrawable->glviewport);
  PgmContextTask *task;
  gfloat pixel_aspect_ratio;

  gltext->width = get_projected_width (gldrawable, gldrawable->width);
  gltext->height = get_projected_height (gldrawable, gldrawable->height);

  GST_OBJECT_LOCK (gltext);

  GST_OBJECT_LOCK (viewport);
  pixel_aspect_ratio = viewport->pixel_aspect_ratio;
  GST_OBJECT_UNLOCK (viewport);

  gltext->size = gltext->width * gltext->height * 4;
  gltext->buffer = (guchar *) g_slice_alloc0 (gltext->size);
  gltext->surface = cairo_image_surface_create_for_data (gltext->buffer,
                                                         CAIRO_FORMAT_ARGB32,
                                                         gltext->width,
                                                         gltext->height,
                                                         gltext->width * 4);

  gltext->cairo_ctx = cairo_create (gltext->surface);
  cairo_scale (gltext->cairo_ctx, pixel_aspect_ratio, 1.0f);
  gltext->layout = pango_cairo_create_layout (gltext->cairo_ctx);
  gltext->pango_ctx = pango_layout_get_context (gltext->layout);

  pgm_texture_set_buffer (gltext->texture, gltext->buffer, PGM_IMAGE_BGRA,
                          gltext->width, gltext->height, gltext->size, 0);

  GST_OBJECT_UNLOCK (gltext);

  set_coordinates (gltext);

  /* And request a texture generation */
  task = pgm_context_task_new (PGM_CONTEXT_GEN_TEXTURE, gltext->texture);
  pgm_context_push_immediate_task (gldrawable->glviewport->context, task);
}

/* Frees the pixmap and the Cairo backend */
static void
free_pixmap (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmContextTask *task;

  GST_OBJECT_LOCK (gltext);

  g_object_unref (gltext->layout);
  gltext->layout = NULL;
  cairo_destroy (gltext->cairo_ctx);
  gltext->cairo_ctx = NULL;
  cairo_surface_destroy (gltext->surface);
  gltext->surface = NULL;
  if (gltext->buffer)
    {
      g_slice_free1 (gltext->size, (gpointer) gltext->buffer);
      gltext->buffer = NULL;
    }

  GST_OBJECT_UNLOCK (gltext);

  /* Request texture clean up */
  task = pgm_context_task_new (PGM_CONTEXT_CLEAN_TEXTURE, gltext->texture);
  pgm_context_push_immediate_task (gldrawable->glviewport->context, task);
}

/* Updates text properties and draw it on the pixmap */
static void
update_pixmap (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmContextTask *task;

  GST_OBJECT_LOCK (gltext);

  /* Make context fully transparent */
  cairo_set_operator (gltext->cairo_ctx, CAIRO_OPERATOR_SOURCE);
  cairo_set_source_rgba (gltext->cairo_ctx, 0.0f, 0.0f, 0.0f, 0.0f);
  cairo_paint (gltext->cairo_ctx);

  /* Then draw the text */
  cairo_set_source_rgba (gltext->cairo_ctx, 1.0f, 1.0f, 1.0f, 1.0f);
  pango_cairo_show_layout (gltext->cairo_ctx, gltext->layout);

  /* And the stroke */
  if (gltext->outline_width > 0.0f)
    {
      pango_cairo_layout_path (gltext->cairo_ctx, gltext->layout);
      cairo_set_source_rgba (gltext->cairo_ctx,
                             gltext->outline_color[0],
                             gltext->outline_color[1],
                             gltext->outline_color[2],
                             gltext->outline_color[3]);
      cairo_set_line_width (gltext->cairo_ctx, gltext->outline_width);
      cairo_stroke (gltext->cairo_ctx);
    }

  GST_OBJECT_UNLOCK (gltext);

#if 0
  /* Debugging */
  cairo_surface_write_to_png (gltext->surface, "/tmp/pgmgltext.png");
#endif

  /* Get rid of the premultiplied alpha */
  unpremultiply_alpha ((guint32*) gltext->buffer, gltext->width, gltext->height);

  /* Request texture upload */
  task = pgm_context_task_new (PGM_CONTEXT_UPLOAD_TEXTURE, gltext->texture);
  pgm_context_push_deferred_task (gldrawable->glviewport->context, task);
}

/* Do a complete regeneration of a text */
static void
regenerate_text (PgmGlText *gltext)
{
  free_pixmap (gltext);
  create_pixmap (gltext);
  update_properties (gltext);
  update_pixmap (gltext);
}

/* Update a text by simply generating the new one in the current allocated
 * memory if the size has not changed. If it has changed, the whole text is
 * regenerated. */
static void
update_text (PgmGlText *gltext)
{
  gboolean need_regeneration;

  GST_OBJECT_LOCK (gltext);
  need_regeneration = gltext->size_updated;
  GST_OBJECT_UNLOCK (gltext);

  if (!need_regeneration)
    update_pixmap (gltext);
  else
    {
      GST_OBJECT_LOCK (gltext);
      gltext->size_updated = FALSE;
      GST_OBJECT_UNLOCK (gltext);

      regenerate_text (gltext);
    }
}

/* PgmGlDrawable methods */

static void
pgm_gl_text_draw (PgmGlDrawable *gldrawable)
{
  PgmGlText *gltext = PGM_GL_TEXT (gldrawable);
  PgmContextProcAddress *gl;

  GST_DEBUG_OBJECT (gldrawable, "draw");

  gl = gldrawable->glviewport->context->gl;

  pgm_texture_bind (gltext->texture);

  gl->enable_client_state (PGM_GL_VERTEX_ARRAY);
  gl->enable_client_state (PGM_GL_TEXTURE_COORD_ARRAY);

  GST_OBJECT_LOCK (gltext);
  gl->vertex_pointer (3, PGM_GL_FLOAT, 0, gltext->vertex);
  gl->tex_coord_pointer (2, PGM_GL_FLOAT, 0, gltext->coord);
  gl->color_4fv (gltext->fg_color);
  GST_OBJECT_UNLOCK (gltext);

  gl->draw_arrays (PGM_GL_QUADS, 0, 4);
  gl->disable_client_state (PGM_GL_VERTEX_ARRAY);
  gl->disable_client_state (PGM_GL_TEXTURE_COORD_ARRAY);

  pgm_texture_unbind (gltext->texture);
}

static void
pgm_gl_text_regenerate (PgmGlDrawable *gldrawable)
{
  PgmGlText *gltext = PGM_GL_TEXT (gldrawable);
  gboolean updated;

  GST_DEBUG_OBJECT (gldrawable, "regenerate");

  GST_OBJECT_LOCK (gltext);
  updated = gltext->size_updated;
  GST_OBJECT_UNLOCK (gltext);

  if (updated)
    {
      regenerate_text (gltext);

      GST_OBJECT_LOCK (gltext);
      gltext->size_updated = FALSE;
      GST_OBJECT_UNLOCK (gltext);
    }
}

static void
pgm_gl_text_update_projection (PgmGlDrawable *gldrawable)
{
  PgmGlText *gltext = PGM_GL_TEXT (gldrawable);

  GST_DEBUG_OBJECT (gldrawable, "update_projection");

  GST_OBJECT_LOCK (gltext);
  gltext->size_updated = TRUE;
  GST_OBJECT_UNLOCK (gltext);
}

static void
pgm_gl_text_set_size (PgmGlDrawable *gldrawable)
{
  PgmGlText *gltext = PGM_GL_TEXT (gldrawable);
  guint i;

  GST_DEBUG_OBJECT (gldrawable, "set_size");

  GST_OBJECT_LOCK (gltext);

  for (i = 0; i < 12; i++)
    gltext->vertex[i] = gldrawable->bg_vertex[i];

  gltext->size_updated = TRUE;

  GST_OBJECT_UNLOCK (gltext);
}

static void
pgm_gl_text_set_position (PgmGlDrawable *gldrawable)
{
  PgmGlText *gltext = PGM_GL_TEXT (gldrawable);
  guint i;

  GST_DEBUG_OBJECT (gldrawable, "set_position");

  GST_OBJECT_LOCK (gltext);

  for (i = 0; i < 12; i++)
    gltext->vertex[i] = gldrawable->bg_vertex[i];

  GST_OBJECT_UNLOCK (gltext);
}

static void
pgm_gl_text_set_fg_color (PgmGlDrawable *gldrawable)
{
  PgmGlText *gltext = PGM_GL_TEXT (gldrawable);
  PgmDrawable *drawable;

  GST_DEBUG_OBJECT (gldrawable, "set_fg_color");

  drawable = gldrawable->drawable;

  GST_OBJECT_LOCK (drawable);
  GST_OBJECT_LOCK (gltext);

  gltext->fg_color[0] = drawable->fg_r * INV_255;
  gltext->fg_color[1] = drawable->fg_g * INV_255;
  gltext->fg_color[2] = drawable->fg_b * INV_255;
  gltext->fg_color[3] = drawable->fg_a * drawable->opacity * SQR_INV_255;

  GST_OBJECT_UNLOCK (gltext);
  GST_OBJECT_UNLOCK (drawable);
}

static void
pgm_gl_text_set_opacity (PgmGlDrawable *gldrawable)
{
  PgmGlText *gltext = PGM_GL_TEXT (gldrawable);
  PgmDrawable *drawable;

  GST_DEBUG_OBJECT (gldrawable, "set_opacity");

  drawable = gldrawable->drawable;

  GST_OBJECT_LOCK (drawable);
  GST_OBJECT_LOCK (gltext);

  gltext->fg_color[3] = drawable->fg_a * drawable->opacity * SQR_INV_255;

  GST_OBJECT_UNLOCK (gltext);
  GST_OBJECT_UNLOCK (drawable);
}

static void
pgm_gl_text_sync (PgmGlDrawable *gldrawable)
{
  PgmGlText *gltext = PGM_GL_TEXT (gldrawable);

  GST_DEBUG_OBJECT (gldrawable, "sync");

  create_pixmap (gltext);
  update_properties (gltext);

  /* Synchronize various other properties */
  pgm_gl_text_set_position (gldrawable);
  pgm_gl_text_set_fg_color (gldrawable);

  update_pixmap (gltext);
}

/* GObject stuff */

G_DEFINE_TYPE (PgmGlText, pgm_gl_text, PGM_TYPE_GL_DRAWABLE);

static void
pgm_gl_text_dispose (GObject *object)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (object);
  PgmGlText *gltext = PGM_GL_TEXT (object);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (gltext, "dispose");

  free_pixmap (gltext);
  pango_font_description_free (gltext->desc);

  gst_object_unref (gldrawable->drawable);
  pgm_context_remove_tasks_with_data (gldrawable->glviewport->context,
                                      gltext->texture);
  task = pgm_context_task_new (PGM_CONTEXT_FREE_TEXTURE, gltext->texture);
  pgm_context_push_immediate_task (gldrawable->glviewport->context, task);
  gltext->texture = NULL;

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_gl_text_class_init (PgmGlTextClass *klass)
{
  GObjectClass *gobject_class;
  PgmGlDrawableClass *drawable_class;

  parent_class = g_type_class_peek_parent (klass);

  gobject_class = G_OBJECT_CLASS (klass);
  drawable_class = PGM_GL_DRAWABLE_CLASS (klass);

  /* GObject virtual table */
  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_gl_text_dispose);

  /* PgmGlDrawable virtual table */
  drawable_class->sync = GST_DEBUG_FUNCPTR (pgm_gl_text_sync);
  drawable_class->draw = GST_DEBUG_FUNCPTR (pgm_gl_text_draw);
  drawable_class->regenerate = GST_DEBUG_FUNCPTR (pgm_gl_text_regenerate);
  drawable_class->update_projection =
    GST_DEBUG_FUNCPTR (pgm_gl_text_update_projection);
  drawable_class->set_size = GST_DEBUG_FUNCPTR (pgm_gl_text_set_size);
  drawable_class->set_position = GST_DEBUG_FUNCPTR (pgm_gl_text_set_position);
  drawable_class->set_fg_color = GST_DEBUG_FUNCPTR (pgm_gl_text_set_fg_color);
  drawable_class->set_opacity = GST_DEBUG_FUNCPTR (pgm_gl_text_set_opacity);
}

static void
pgm_gl_text_init (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "init");

  gltext->desc = pango_font_description_new ();
  gltext->size_updated = FALSE;
}

/* Public methods */

PgmGlDrawable *
pgm_gl_text_new (PgmDrawable *drawable,
                 PgmGlViewport *glviewport)
{
  PgmGlDrawable *gldrawable;
  PgmGlText *gltext;

  gltext = g_object_new (PGM_TYPE_GL_TEXT, NULL);

  GST_DEBUG_OBJECT (gltext, "created new gltext");

  gltext->texture = pgm_texture_new (glviewport->context);

  gldrawable = PGM_GL_DRAWABLE (gltext);
  gldrawable->drawable = gst_object_ref (drawable);
  gldrawable->glviewport = glviewport;
  pgm_gl_drawable_sync (gldrawable);

  return gldrawable;
}

void
pgm_gl_text_set_label (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_label");

  set_label (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_markup (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_markup");

  set_label (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_font_family (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_font_family");

  set_font_family (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_font_height (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_font_height");

  set_font_height (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_ellipsize (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_ellipsize");

  set_ellipsize (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_justify (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_justify");

  set_justify (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_alignment (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_alignment");

  set_alignment (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_wrap (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_wrap");

  set_wrap (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_gravity (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_gravity");

  set_gravity (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_stretch (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_stretch");

  set_stretch (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_style (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_style");

  set_style (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_variant (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_variant");

  set_variant (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_weight (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_weight");

  set_weight (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_line_spacing (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_line_spacing");

  set_line_spacing (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_outline_color (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_outline_color");

  set_outline_color (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_outline_width (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_outline_width");

  set_outline_width (gltext);
  update_text (gltext);
}
