# -*- mode: python; coding: utf-8 -*-
#
# Pigment Python tools
#
# Copyright © 2006, 2007 Fluendo Embedded S.L.
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

import os
import pgm
from pypgmtools.graph.group import Group
from pypgmtools.graph.image import Image
from pypgmtools.timing import implicit

VERTICAL, HORIZONTAL = 0, 1

class List(Group):

    def __init__(self,
                 canvas,
                 layer,
                 width             = 1.0,
                 height            = 2.0,
                 visible_items     = 7,
                 orientation       = VERTICAL,
                 selected_item     = 0,
                 selector_file     = None,
                 selector_offset   = (0.0, 0.0, 0.0),
                 selector_size     = (1.0, 1.0)):
        Group.__init__(self, canvas, layer)

        # FIXME: Group.size is not overrided
        self._width            = width
        self._height           = height
        # visible_items must be odd
        if visible_items % 2 != 1:
            raise Exception("Please provide an odd number of visible items")
        self._visible_items    = visible_items
        self._selected_item    = selected_item
        self._canvas           = canvas
        self._widget_height    = height / (visible_items + 2)
        self._widget_width     = width / (visible_items + 2)
        self._orientation      = orientation
        self._attributes       = []
        self.widgets   = []
        self.animated_widgets = []
        self._selector         = None
        self._selector_offset  = selector_offset

        self._layout()

        if selector_file:
            selector = Image()
            selector.set_from_fd(os.open(selector_file, os.O_RDONLY))
            selector.opacity   = 0
            selector.size      = selector_size
            selector.fg_color  = (255, 255, 255, 255)
            selector.bg_color  = (0, 0, 0, 0)
            selector.alignment = pgm.IMAGE_TOP_LEFT
            selector.layout    = pgm.IMAGE_FILLED
            selector.visible   = True
            self.add(selector)
            self._selector     = selector
            self._animated_selector = implicit.AnimatedObject(selector)
            self._animated_selector.setup_next_animations(duration = 300)


    def _compute_position(self, index):
        if self._orientation == VERTICAL:
            x = 0.0
            y = self._widget_height * index
        elif self._orientation == HORIZONTAL:
            x = self._widget_width * index
            y = 0.0

        position = (x, y, 0.0)
        return position

    def _compute_opacity(self, index):
        # floor division
        middle_index = self._visible_items // 2 + 1
        if index == middle_index:
            return 255
        else:
            max_opacity = 100
            step = max_opacity / middle_index

            if index > middle_index:
                index = -index + 2*middle_index

            opacity = index * step
            return opacity

    def _compute_zoom(self, index):
        # floor division
        middle_index = self._visible_items // 2 + 1
        if index == middle_index:
            return 1.0
        else:
            max_zoom = 0.8
            min_zoom = 0.3
            step = (max_zoom - min_zoom) / middle_index

            if index > middle_index:
                index = -index + 2*middle_index

            zoom = min_zoom + index * step
            return zoom


    def _compute_attributes(self, index):
        return (self._compute_position(index),
                self._compute_opacity(index),
                self._compute_zoom(index))

    def _layout(self):
        if self._visible_items == 0:
            return

        self._attributes = []
        for i in xrange(self._visible_items + 2):
            self._attributes.append(self._compute_attributes(i))

    def _update(self):
        # floor division
        start = self._selected_item - self._visible_items // 2 - 1
        stop = min(start + self._visible_items + 2, len(self.widgets))

        i = max(0, -start)

        for widget in self.animated_widgets[max(start, 0):stop]:
            widget.opacity  = self._attributes[i][1]
            widget.position = self._attributes[i][0]
            zoom = self._attributes[i][2]
            if self._orientation == VERTICAL:
                h = self._widget_height*zoom
                w = self._width*zoom
                widget.size = (w, h)

            elif self._orientation == HORIZONTAL:
                w = self._widget_width*zoom
                h = self._height*zoom
                widget.size = (w, h)
            i += 1

    def width__get(self):
        return self._width

    def width__set(self, width):
        self._width = width

        self._widget_width = self._width / self._visible_items

        self._layout()
        self._update()

    def height__get(self):
        return self._height

    def height__set(self, height):
        self._height = height

        self._widget_height = self._height / self._visible_items

        self._layout()
        self._update()

    def visible_items__get(self):
        return self._visible_items

    def visible_items__set(self, visible_items):
        # visible_items must be odd
        if visible_items % 2 != 1:
            raise Exception("Please provide an odd number of visible items")
        self._visible_items = visible_items

        self._widget_height = self._height / self._visible_items
        self._widget_width = self._width / self._visible_items

        self._layout()
        self._update()

    def selected_item__get(self):
        return self._selected_item

    def selected_item__set(self, index):
        if index >= 0 and index < len(self.widgets):
            s = self._visible_items // 2
            delta = self._selected_item - index

            if delta <= 0:
                increment = 1
            else:
                increment = -1

            if abs(delta) > s:
                for widget in self.animated_widgets:
                    widget.stop_animations()
                for widget in self.widgets:
                    widget.opacity = 0

                for i in xrange(index-(increment*s)+increment, index+increment, increment):
                    self._selected_item = i
                    self._update()
            else:
                for i in xrange(self._selected_item+increment, index+increment, increment):
                    self._selected_item = i
                    self._update()
                

    def orientation__set(self, orientation):
        self._orientation = orientation
        self._layout()
        self._update()

    def orientation__get(self):
        return self._orientation

    def show_selector(self):
        self._animated_selector.opacity = 255

    def hide_selector(self):
        self._animated_selector.opacity = 0

    def __len__(self):
        return len(self.widgets)

    def insert(self, index, widget):
        if widget in self.widgets:
            return

        # text insertion
        if len(self.widgets) == 0:
            widget.position = self._compute_position(self._visible_items // 2 + 1)
        elif abs(index) < len(self.widgets):
            # insert it at the position of self.animated_widgets[index]
            widget.position = self.animated_widgets[index].position
        else:
            widget.position = self._compute_position(self._visible_items + 1)

        # FIXME: should take the zoom into account
        if self._orientation == VERTICAL:
            widget.size = (self._width, self._widget_height)
        elif self._orientation == HORIZONTAL:
            widget.size = (self._widget_width, self._height)

        widget.opacity = 0
        widget.fg_color = (255, 255, 255, 255)

        widget.visible = True

        Group.add(self, widget)
        self.widgets.insert(index, widget)

        animated_widget = implicit.AnimatedObject(widget)
        animated_widget.setup_next_animations(duration = 300,
                                              transformation = implicit.DECELERATE)
        animated_widget.mode = implicit.REPLACE
        self.animated_widgets.insert(index, animated_widget)

        if index <= self._selected_item:
            self._selected_item += 1

        # update only if the inserted item is visible
        if self._is_visible(index):
            self._update()

    def _is_visible(self, index):
        # FIXME: handle index < 0
        s = self._visible_items // 2
        return index >= self._selected_item - s and \
               index <= self._selected_item + s

    def append(self, widget):
        self.insert(len(self.widgets), widget)

    def remove(self, widget):
        try:
            # if the widget is part of the List
            index = self.widgets.index(widget)
        except ValueError:
            # if the widget is just part of the Group
            Group.remove(self, widget)
        else:
            self.pop(index)

    def pop(self, index):
        animated_widget = self.animated_widgets.pop(index)
        animated_widget.stop_animations()
        static = self.widgets.pop(index)
        Group.remove(self, static)

        # prepare update if the item to be removed item is visible
        to_update = self._is_visible(index)

        # FIXME: handle index < 0
        if index < self._selected_item and index >= 0:
            self._selected_item -= 1
        elif index == self._selected_item:
            if index == len(self.widgets) and index > 0:
                self._selected_item -= 1

        if to_update:
            self._update()

        return static

    def __getitem__(self, index):
        return self.widgets[index]


if __name__ == "__main__":
    import os
    import pgm
    import gobject
    import gst
    from pypgmtools.graph.text import Text
    from pypgmtools.graph.image import Image
    from pypgmtools.timing import implicit

    def create_text(label):
        txt = Text()
        txt.label = label
        txt.font_family = "Bitstream DejaVu"
        txt.font_height = 0.225
        txt.fg_color = (255, 255, 255, 255)
        txt.bg_color = (255, 0, 0, 255)
        txt.ellipsize = pgm.TEXT_ELLIPSIZE_END
        txt.visible = True
        return txt

    def create_img(img_file):
        img = Image()
        img.set_from_fd(os.open(img_file, os.O_RDONLY))
        img.fg_color = (255, 255, 255, 255)
        img.bg_color = (100, 200, 100, 155)
        img.visible = True
        return img

    def create_video(video_uri):
        img = Image()
        img.fg_color = (255, 255, 255, 255)
        img.bg_color = (0, 0, 0, 0)
        img.alignment = pgm.IMAGE_LEFT
        img.visible = True

        # GStreamer pipeline setup
        pipeline = gst.element_factory_make('playbin')
        sink = gst.element_factory_make('pgmimagesink')
        pipeline.set_property('uri', video_uri)
        pipeline.set_property('video-sink', sink)
        sink.set_property('image', img)
        pipeline.set_state(gst.STATE_PLAYING)

        return img


    def on_key_press(viewport, event, gl, widget, step_size):
        if event.type == pgm.KEY_PRESS:
            # quit on q or ESC
            if event.keyval == pgm.keysyms.q or event.keyval == pgm.keysyms.Escape:
                pgm.main_quit()

            elif event.keyval == pgm.keysyms.b:
                list_widget.selected_item = 0

            elif event.keyval == pgm.keysyms.g:
                list_widget.selected_item = len(list_widget) - 1

            elif event.keyval == pgm.keysyms.h:
                list_widget.visible_items += 2
                print list_widget.visible_items

            elif event.keyval == pgm.keysyms.n:
                list_widget.visible_items -= 2

            elif event.keyval == pgm.keysyms.s:
                if list_widget.orientation == VERTICAL:
                    list_widget.orientation = HORIZONTAL
                elif list_widget.orientation == HORIZONTAL:
                    list_widget.orientation = VERTICAL

            elif event.keyval == pgm.keysyms.Down or \
                 event.keyval == pgm.keysyms.Right:
                list_widget.selected_item += 1

            elif event.keyval == pgm.keysyms.Up or \
                 event.keyval == pgm.keysyms.Left:
                list_widget.selected_item -= 1

            # add an example entry at the 3rd position
            elif event.keyval == pgm.keysyms.space:
                list_widget.insert(0, create_text("T"))
                #list_widget.append(create_text("Test"))

            # remove the currently selected item
            elif event.keyval == pgm.keysyms.Return:
                list_widget.pop(list_widget.selected_item)

    def on_delete(viewport, event):
        pgm.main_quit()



    # OpenGL viewport creation
    factory = pgm.ViewportFactory('opengl')
    gl = factory.create()
    gl.title = 'TextList widget-test'

    # Canvas and image drawable creation
    canvas = pgm.Canvas()

    # Bind the canvas to the OpenGL viewport
    gl.set_canvas(canvas)
    gl.show()

    list_widget = List(canvas, pgm.DRAWABLE_MIDDLE, orientation=HORIZONTAL)
    list_widget.position = (0.5, 1.0, 0.0)
    list_widget.width = 3.0
    list_widget.height = 1.0
    list_widget.visible_items = 13

    for i in xrange(150):
        #list_widget.append(create_text("%s" % i))
        list_widget.append(create_img("examples/pictures/fluendo.png"))
        """
        t = create_text("%s" % i)
        i = create_img("examples/pictures/fluendo.png")
        g = Group(canvas, pgm.DRAWABLE_MIDDLE)
        g.add(t)
        g.add(i)
        g.visible = True
        list_widget.append(g)
        """

    #list_widget.append(create_video("file:///home/kaleo/Desktop/media/videos/t_mariogalaxy_e36_h264.wmv"))

    list_widget.selected_item = 3
    list_widget.visible = True

    animated_list_widget = implicit.AnimatedObject(list_widget)


    # Let's start a mainloop
    gl.connect('key-press-event',
               on_key_press,
               gl,
               animated_list_widget,
               0.25)
    gl.connect('delete-event', on_delete)
    gobject.timeout_add(10, gl.update)
    pgm.main()

